% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BWC.R
\name{bwc}
\alias{bwc}
\alias{d_bwc}
\alias{r_bwc}
\alias{fit_bwc_mm}
\alias{fit_bwc_mle}
\title{Density evaluation, sampling, and parameter estimation of the
bivariate wrapped Cauchy distribution}
\usage{
d_bwc(x, mu, xi)

r_bwc(n, mu, xi)

fit_bwc_mm(x, hom = FALSE, indep = FALSE)

fit_bwc_mle(
  x,
  start = NULL,
  lower = c(-pi, -pi, 0, 0, -1 + 0.001),
  upper = c(pi, pi, 1 - 0.001, 1 - 0.001, 1 - 0.001),
  hom = FALSE,
  indep = FALSE,
  ...
)
}
\arguments{
\item{x}{matrix of size \code{c(nx, 2)} with the angles on which the density
is evaluated.}

\item{mu}{circular means of the density, a vector of length \code{2}.}

\item{xi}{a vector of length \code{3} with the marginal concentrations
\eqn{(\xi_1, \xi_2)}, and the dependence parameter \eqn{\rho}.}

\item{n}{sample size.}

\item{hom}{assume a homogeneous distribution with equal marginal
concentrations? Defaults to \code{FALSE}.}

\item{indep}{set the dependence parameter to zero? Defaults to \code{FALSE}.}

\item{start}{a vector of length \code{5} with the initial values for the
maximum likelihood optimizer. If \code{NULL} (default), the method of
moments estimates are employed.}

\item{lower, upper}{vectors of length \code{5} with the bounds for the
likelihood optimizer. Default to \code{c(-pi, -pi, 0, 0, -1 + 1e-3)} and
\code{c(pi, pi, 1 - 1e-3, 1 - 1e-3, 1 - 1e-3)}.}

\item{...}{further parameters passed to
\code{\link[=sdetorus]{mleOptimWrapper}}.}
}
\value{
\itemize{
  \item \code{d_bwc}: a vector of length \code{nx} with the density evaluated
  at \code{x}.
  \item \code{r_bwc}: a matrix of size \code{c(n, 2)} with the random sample.
  \item \code{fit_mme_bwc, fit_mle_bwc}: a list with the parameters
  \eqn{(\mu_1, \mu_2, \xi_1, \xi_2, \rho)} and the object \code{opt}
  containing the optimization summary.
}
}
\description{
Computation of the density of a bivariate wrapped Cauchy:
\deqn{f(\theta_1, \theta_2)=c(\xi_1,\xi_2,\rho)\{c_0(\xi_1,\xi_2,\rho)-
c_1(\xi_1,\xi_2,\rho) \cos (\theta_1-\mu_1)-
c_2(\xi_1,\xi_2,\rho)\cos (\theta_2-\mu_2)-\\
c_3(\xi_1,\xi_2,\rho) \cos (\theta_1-\mu_1) \cos (\theta_2-\mu_2)-
c_4(\xi_1,\xi_2,\rho) \sin (\theta_1-\mu_1) \sin (\theta_2-\mu_2)\}^{-1}.}

Simulation of samples from a bivariate wrapped Cauchy.

Maximum likelihood and method of moments estimation of the
parameters \eqn{(\mu_1, \mu_2, \xi_1, \xi_2, \rho)}.
}
\examples{
## Density evaluation

mu <- c(0, 0)
xi <- c(0.3, 0.5, 0.4)
nth <- 50
th <- seq(-pi, pi, l = nth)
x <- as.matrix(expand.grid(th, th))
d <- d_bwc(x = x, mu = mu, xi = xi)
filled.contour(th, th, matrix(d, nth, nth), col = viridisLite::viridis(20),
               levels = seq(0, max(d), l = 20))

## Sampling and estimation

n <- 100
samp <- r_bwc(n = n, mu = mu, xi = xi)
(param_mm <- fit_bwc_mm(samp)$par)
(param_mle <- fit_bwc_mle(samp)$par)
}
\references{
Kato, S. and Pewsey, A. (2015). A Möbius transformation-induced distribution
on the torus. \emph{Biometrika}, 102(2):359--370. \doi{10.1093/biomet/asv003}
}
\author{
The original code for \code{r_bwc} was supplied by
Arthur Pewsey.
}
