% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rptree.R
\name{rpf_knn_query}
\alias{rpf_knn_query}
\title{Query a random projection forest index for nearest neighbors}
\usage{
rpf_knn_query(
  query,
  reference,
  forest,
  k,
  cache = TRUE,
  n_threads = 0,
  verbose = FALSE,
  obs = "R"
)
}
\arguments{
\item{query}{Matrix of \code{n} query items, with observations in the rows and
features in the columns. Optionally, the data may be passed with the
observations in the columns, by setting \code{obs = "C"}, which should be more
efficient. The \code{reference} data must be passed in the same orientation as
\code{query}. Possible formats are \code{\link[base:data.frame]{base::data.frame()}}, \code{\link[base:matrix]{base::matrix()}}
or \code{\link[Matrix:sparseMatrix]{Matrix::sparseMatrix()}}. Sparse matrices should be in \code{dgCMatrix}
format. Dataframes will be converted to \code{numerical} matrix format
internally, so if your data columns are \code{logical} and intended to be used
with the specialized binary \code{metric}s, you should convert it to a logical
matrix first (otherwise you will get the slower dense numerical version).}

\item{reference}{Matrix of \code{m} reference items, with observations in the rows
and features in the columns. The nearest neighbors to the queries are
calculated from this data and should be the same data used to build the
\code{forest}. Optionally, the data may be passed with the observations in the
columns, by setting \code{obs = "C"}, which should be more efficient. The
\code{query} data must be passed in the same format and orientation as
\code{reference}. Possible formats are \code{\link[base:data.frame]{base::data.frame()}}, \code{\link[base:matrix]{base::matrix()}} or
\code{\link[Matrix:sparseMatrix]{Matrix::sparseMatrix()}}. Sparse matrices should be in \code{dgCMatrix} format.}

\item{forest}{A random partition forest, created by \code{\link[=rpf_build]{rpf_build()}},
representing partitions of the data in \code{reference}.}

\item{k}{Number of nearest neighbors to return. You are unlikely to get good
results if you choose a value substantially larger than the value of
\code{leaf_size} used to build the \code{forest}.}

\item{cache}{if \code{TRUE} (the default) then candidate indices found in the
leaves of the forest are cached to avoid recalculating the same distance
repeatedly. This incurs an extra memory cost which scales with \code{n_threads}.
Set this to \code{FALSE} to disable distance caching.}

\item{n_threads}{Number of threads to use. Note that the parallelism in the
search is done over the observations in \code{query} not the trees in the
\code{forest}. Thus a single observation will not see any speed-up from
increasing \code{n_threads}.}

\item{verbose}{If \code{TRUE}, log information to the console.}

\item{obs}{set to \code{"C"} to indicate that the input \code{data} orientation stores
each observation as a column. The default \code{"R"} means that observations are
stored in each row. Storing the data by row is usually more convenient, but
internally your data will be converted to column storage. Passing it
already column-oriented will save some memory and (a small amount of) CPU
usage.}
}
\value{
the approximate nearest neighbor graph as a list containing:
\itemize{
\item \code{idx} an n by k matrix containing the nearest neighbor indices.
\item \code{dist} an n by k matrix containing the nearest neighbor distances.
}

\code{k} neighbors per observation are not guaranteed to be found. Missing data
is represented with an index of \code{0} and a distance of \code{NA}.
}
\description{
Run queries against a "forest" of Random Projection Trees (Dasgupta and
Freund, 2008), to return nearest neighbors taken from the reference data used
to build the forest.
}
\examples{
# Build a forest of 10 trees from the odd rows
iris_odd <- iris[seq_len(nrow(iris)) \%\% 2 == 1, ]
iris_odd_forest <- rpf_build(iris_odd, n_trees = 10)

iris_even <- iris[seq_len(nrow(iris)) \%\% 2 == 0, ]
iris_even_nn <- rpf_knn_query(
  query = iris_even, reference = iris_odd,
  forest = iris_odd_forest, k = 15
)
}
\references{
Dasgupta, S., & Freund, Y. (2008, May).
Random projection trees and low dimensional manifolds.
In \emph{Proceedings of the fortieth annual ACM symposium on Theory of computing}
(pp. 537-546).
\doi{10.1145/1374376.1374452}.
}
\seealso{
\code{\link[=rpf_build]{rpf_build()}}
}
