#' Get GEFS ensemble forecast data for a specific lat/lon.
#'
#' Fetches GEFS forecast data for every 6 hours out to 384 hours past
#' selected date. GEFS is an ensemble of 21 models that can be
#' summarized to estimate likelihoods of forecasts.
#'
#' @export
#'
#' @param var the variable to get. Must be one of the variables listed in
#' `gefs_variables()`
#' @param lat the latitude. Values must be sequential and are rounded to the
#' nearest GEFS available latitude.
#' @param lon the longitude. Values must be sequential and are rounded to the
#' nearest GEFS available longitude.
#' @param date A date/string formatted as YYYYMMDD.
#' @param forecast_time a string indicating which time of day UTC the
#' forecast is from. Options are "0000", "0600", "1200", "1800".
#' @param ens_idx sequential list of ensembles to fetch. Default is all 21.
#' Note that the ensembles are labelled 0-20, so ens_idx=1:3 will return
#' ensembles 0, 1, and 2.
#' @param time_idx sequential list of time increments to return. List is the
#' index of times, which are in 6 hour increments. (e.g. c(1,2) fetches the
#' 6 and 12 hour forecast.)
#' @param dims (not implemented) indices for additional dimensions to be
#' included between lat, lon, ens, and time.
#' @param raw logical to indicate whether to return raw data matrix or
#' reshaped data frame.
#' @param ... additional parameters passed to `ncvar_get`
#' @return a list containing metadata and accompanying data frame of
#' forecast values. If lat/lon are not specified, the $data is an
#' unprocessed matrix.
#'
#' @references
#'
#' - Data description - \url{http://bit.ly/noaagefs}.
#' - Adapted from Python code written by Von P. Walden, Washington State
#' University
#'
#' @author Nicholas Potter \email{potterzot@@gmail.com}
#' @examples \dontrun{
#'
#' #avialable latitudes and longitudes
#' gefs_latitudes()
#' gefs_longitudes()
#'
#' #get a list of all gefs variables
#' gefs_variables()
#'
#' #All GEFS dimensions
#' gefs_dimensions()
#'
#' #values for a specific dimension
#' gefs_dimension_values("height_above_ground")
#'
#' #example location.
#' lat <- 46.28125
#' lon <- -118.2188
#'
#' #Get forecast for a certain variable.
#' forecast <- gefs("Total_precipitation_surface_6_Hour_Accumulation_ens",
#'   lat, lon)
#'
#' #Fetch a different date (available up to 10 days prior to today)
#' forecast_yesterday_prec <- gefs(
#'    "Total_precipitation_surface_6_Hour_Accumulation_ens",
#'    lat, lon, date=format(as.Date(Sys.time()) - 1, "%Y%m%d"))
#'
#' #specific ensemble and times, for the 1800 forecast.
#' # here ensembles 1-3 (ensembles are numbered starting with 0)
#' # and time for 2 days from today at 1800
#' date <- format(as.Date(Sys.time()) - 1, "%Y%m%d")
#' var <- "Temperature_height_above_ground_ens"
#' gefs(var, lat, lon, date = date, forecast_time = "1800", ens_idx=2:4,
#'   time_idx=1:8)
#'
#' #One ensemble, all latitudes and longitudes (this is a big file) for the
#' # next 3 days.
#' # gefs(var, ens=1, time=1:12)
#' }
#'
gefs <- function(var, lat, lon, ...) {
  check4pkg("ncdf4")
  gefs_GET(var, lat, lon, ...)
}

#' @rdname gefs
gefs_CONNECT <- function(date = format(Sys.time(), "%Y%m%d"),
                         forecast_time = c("0000", "0600", "1200", "1800")) {


  # Until bug #127 is resolved
  if (is_windows()) warning("gefs not implemented on windows yet", .call = FALSE)

  #forecast time
  forecast_time <- match.arg(forecast_time)

  #url parts
  gefs_url_pre <- 'http://thredds.ucar.edu/thredds/dodsC/grib/NCEP/GEFS/Global_1p0deg_Ensemble/members/GEFS_Global_1p0deg_Ensemble_'
  gefs_url_suf <- ".grib2"

  #final url
  gefs_url <- paste0(gefs_url_pre, date, "_", forecast_time, gefs_url_suf)

  #open the connection
  #nc_open(gefs_url) #ncdf4 version
  ncdf4::nc_open(gefs_url)
}

#' @rdname gefs
gefs_GET <- function(var, lat, lon,
                     date = format(Sys.time(), "%Y%m%d"),
                     forecast_time = c("0000", "0600", "1200", "1800"),
                     ens_idx = 1:21,
                     time_idx = 1:65,
                     dims = NULL,
                     raw = FALSE,
                     ...) {

  ###Sanity Checks
  if (missing(var)) stop("Need to specify the variable to get. A list of variables is available from gefs_variables().")

  # lats and lons must be sequential and within ranges
  lats <- sort(round(lat, 0))
  if (!all(lats == seq(lats[1], length.out = length(lats)))) stop("Latitudes must be sequential.")
  if (any(lats < -90 | lats > 90)) stop("Latitudes must be in c(-90,90).")

  lons <- sort(round(lon, 0))
  if (!all(lons == seq(lons[1], length.out = length(lons)))) stop("Longitudes must be sequential.")
  if (any(lons < -180 | lons > 360)) stop("Longitudes must be in c(-180,180) or c(0,360).")


  #get a connection
  con <- gefs_CONNECT(date, forecast_time)

  # Rename extra dimensions, to be changed later
  if (!is.null(dims)) { warning("Can't select additional dimensions yet.",
                                .call = FALSE)
  } else {
    additional_dims <- dims
  }


  #Get a subset of data to speed up access
  v <- con$var[[var]] # lon, lat, height_above_ground, ens (ensemble), time1
  varsize <- v$varsize
  ndims <- v$ndims
  n_time <- varsize[ndims] #time is always the last dimension

  # Set the indices for each dimension
  dim_idxs <- list()
  for (i in 1:length(v$dim)) {
    dn <- v$dim[[i]]$name
    if(dn == "lon") {
      dim_idxs[[i]] <- if(!missing(lon)) which(v$dim[[i]]$vals %in% (round(lon,0) %% 360)) else 1:v$dim[[i]]$len
    } else if (dn == "lat") {
      dim_idxs[[i]] <- if(!missing(lat)) which(v$dim[[2]]$vals %in% round(lat, 0)) else 1:v$dim[[2]]$len
    } else if (dn == "ens") {
      dim_idxs[[i]] <- ens_idx
    } else if (dn %in% c("time1", "time2")) {
      dim_idxs[[i]] <- time_idx
    } else if (dn %in% names(additional_dims)) {
      dim_idxs[[i]] <- which(v$dim[[i]]$vals %in% additional_dims[[dn]])
    } else {
      dim_idxs[[i]] <- 1:v$dim[[i]]$len
    }
  }
  names(dim_idxs) <- lapply(1:length(v$dim), function(i) { v$dim[[i]]$name })

  #start indices of dimensions to read from data
  start <- sapply(dim_idxs, function(d) { min(d) })
  count_n <- sapply(dim_idxs, function(d) { length(d) })

  ##ncdf4 version
  d_raw <- ncdf4::ncvar_get(con, v, start = start, count = count_n, ...)

  #create the data frame
  #For now, if lat/lon are not specified, just return a matrix.
  if (!raw) {
    dim_vals <- lapply(1:length(dim_idxs), function(i) { v$dim[[i]]$vals[dim_idxs[[i]]] })
    names(dim_vals) <- names(dim_idxs)
    d = cbind(as.data.frame(as.vector(d_raw)), expand.grid(dim_vals))
    names(d)[[1]] <- var
  } else {
    d <- d_raw
  }

  fname <- strsplit(con$filename, "_")[[1]]
  date <- fname[7]
  forecast_time <- strsplit(fname, ".grib2")[[8]]
  list(forecast_date = date,
       forecast_time = forecast_time,
       dimensions    = dims,
       data          = d)
}

########################
# helper functions

#' @export
#'
#' @param con an ncdf4 connection.
#' @rdname gefs
gefs_latitudes <- function(con = NULL, ...) {
  gefs_dimension_values("lat", con)
}

#' @export
#' @rdname gefs
gefs_longitudes <- function(con = NULL, ...) {
  gefs_dimension_values("lon", con)
}

#' @export
#' @rdname gefs
gefs_variables <- function(con = NULL, ...) {
  if (is.null(con)) con = gefs_CONNECT(...)
  names(con$var)
}

#' @export
#' @rdname gefs
gefs_dimensions <- function(con = NULL, ...) {
  if (is.null(con)) con = gefs_CONNECT(...)
  names(con$dim)
}

#' @export
#'
#' @param dim (character) the dimension.
#' @rdname gefs
gefs_dimension_values <- function(dim, con = NULL, ...) {
  if (is.null(dim) || missing(dim)) stop("dim cannot be NULL or missing.")
  if (is.null(con)) con = gefs_CONNECT(...)
  con$dim[[dim]]$vals
}

