

#' making paths uniform
#' @param path path to be sanitized
#' @return sanitized path
#' @keywords internal
sanitize_path <- function(path){
  path <- stringr::str_replace(    path, "^ *", "")
  path <- ifelse( !grepl("^/", path),  paste0("/", path), path)
  return(path)
}




#' transforming permissions into regular expressions (values)
#'
#' @param permission_value the value column of permissions (the paths)
#' @keywords internal
sanitize_permission_values <- function(permission_value){
  tmp <- sanitize_path(permission_value)
  tmp <- stringr::str_replace_all(tmp, "\\?", "\\\\?") # escape questionmarks
  tmp <- stringr::str_replace_all(tmp, "\\*",".*")     # translate '*' to '.*'
  tmp <- stringr::str_replace_all(tmp, "^/","^/")
  tmp <- stringr::str_replace_all(tmp, "/$","")
  tmp <- stringr::str_replace_all(tmp, "^\\^$","^/")

  return(tmp)
}

#' transforming permissions into regular expressions (whole permission)
#'
#' @param permissions the permissions to be transformed
#' @keywords internal
sanitize_permissions <- function(permissions){
  tmp <- permissions
  # epressing Disallow "" as Allow "*"
  iffer <- grepl("Disallow", tmp$field, ignore.case = TRUE) & grepl("^ *$", tmp$value)
  if( sum(iffer) > 0 ){
    tmp[iffer, ]$field <- "Allow"
    tmp[iffer, ]$value      <- "/"
  }
  # permission path sanitization
  tmp$value <- sanitize_permission_values(tmp$value)
  return(tmp)
}


#' check if a bot has permissions to access page
#' @param permissions data.frame generated by \code{rt_get_permissions()} with
#'  three columns: 'useragent' (name of the bot), 'permission' (permission
#'  granted: Disallow/Allow) and 'value' (path for which permission applies)
#' @param bot name of the bot, defaults to "*"
#' @seealso \link{paths_allowed}
#' @param path path for which to check bot's permission, defaults to "/"
#' @return TRUE permission granted; FALSE permission denied; NA path_allowed()
#' cannot determine if path is allowed or not allowed, will result in a warning
#' as well
#' @keywords internal

path_allowed <- function(permissions, path="/", bot="*"){
  # checking and initializetion
  stopifnot(length(bot)==1)
  if( is.null(bot) | bot=="" | is.na(bot) ) bot <- "*"
  perm_sanitized <- sanitize_permissions(permissions)
  path <- sanitize_path(path)

  # subsetting to permissions relevant to bot
  perm_applicable <-
    perm_sanitized[
      grepl("\\*", perm_sanitized$useragent) | tolower(bot)==tolower(perm_sanitized$useragent),
      ]

  # checking which permissions are applicable to path
  if( dim(perm_applicable)[1] > 0 ){
    perm_applicable <-
      perm_applicable[
        sapply(perm_applicable$value, grepl, x=path),
      ]
  }

  # deciding upon rules
  # no permissions --> TRUE
  if( dim(perm_applicable)[1]==0 ){
    return(TRUE)
  }
  # only disallows --> FALSE
  if ( all(grepl("disallow", perm_applicable$field, ignore.case = TRUE)) ){
    return(FALSE)
  }
  # only allows --> TRUE
  if ( all(grepl("^allow", perm_applicable$field, ignore.case = TRUE)) ){
    return(TRUE)
  }
  # diverse permissions but bot specific all disallow
  if ( all(grepl("disallow", with(perm_applicable, field[tolower(useragent)==tolower(bot)]), ignore.case = TRUE)) ){
    return(FALSE)
  }
  # diverse permissions but bot specific all allow
  if ( all(grepl("^allow", with(perm_applicable, field[tolower(useragent)==tolower(bot)]), ignore.case = TRUE)) ){
    return(TRUE)
  }
  # diverse permissions --> longest permision wins
  if (
    any(grepl("disallow", perm_applicable$field, ignore.case = TRUE)) &
    any(grepl("^allow", perm_applicable$field, ignore.case = TRUE))
  ){
    perm_path_lengths <- stringr::str_count(perm_applicable$value)
    iffer <- perm_path_lengths == max(perm_path_lengths)
    # take longest permission applicable and return TRUE if it Allows, false if it Disallows
    if( sum(iffer) == 1 ){
      return(
        !grepl("disallow", perm_applicable[iffer,]$field, ignore.case = TRUE)
      )
    }
    if( sum(iffer) > 1 ){
      if( any( grepl("disallow", perm_applicable[iffer,]$field, ignore.case = TRUE) ) ){
        return(FALSE)
      }
    }
  }
  # next missing case ...

  # message and return in case that case is not covered
  warning(
    paste0(
      " robotstxt::path_allowed() says:",
      " Encountered problems while trying to determine bot permissions, returning NA instead of TRUE/FALSE",
      sep="\n"
    )
  )
  #dev# return( named_list( path, bot, perm_applicable ) )
  return(NA)
}


#' check if a bot has permissions to access page(s)
#'
#' wrapper to \code{\link{path_allowed}}
#' @param domain Domain for which paths should be checked. Defaults to "auto".
#'   If set to "auto" function will try to guess the domain by parsing the paths
#'   argument. Note however, that these are educated guesses which might utterly
#'   fail. To be on the save side, provide appropriate domains manually.
#' @param bot name of the bot, defaults to "*"
#' @param paths paths for which to check bot's permission, defaults to "/"
#' @seealso \link{path_allowed}
#' @export
paths_allowed <- function(paths="/", domain="auto", bot="*"){
  # dev # paths <- c("https://pmeissner.com/_layouts/index.html","https://wikipedia.org", "http://pmeissner.com/index.html")
  if( all(domain == "auto") ){
    domain <- guess_domain(paths)
    paths  <- remove_domain(paths)
  }
  if( length(unique(domain))==1 ){
    domain <- domain[1]
  }
  permissions <-
    lapply(domain, function(x){robotstxt(x)$permissions} )
  worker <- function(path, permissions, bot, domain){
    if( is.na(domain) ){
      return(NA)
    }else{
      path_allowed(permissions = permissions, path = path, bot = bot)
    }
  }
  tmp <- mapply(worker, path=paths, permissions=permissions, bot=bot, domain)
  names(tmp) <- NULL
  return(tmp)
}


#' function guessing domain from path
#' @param x path aka URL from which to infer domain
guess_domain <- function(x){
  if(length(x)>1){
    return(unlist(lapply(x, guess_domain)))
  }else{
    hostname <- httr::parse_url(x)$hostname
    if( is.null(hostname) ){
      hostname <- NA
    }
    return(hostname)
  }
}

#' function to remove domain from path
#' @param x path aka URL from which to first infer domain and then remove it
remove_domain <- function(x){
  unlist(lapply(
      x,
      function(x){
        if( is.na(x) ){
          return(x)
        }else{
          stringr::str_replace(x, paste0("^.*", guess_domain(x)), "")
        }
      }
  ))
}








