#' Summarize interpolation results
#'
#' @description
#' Summarize the estimated mean, standard error, and confidence interval.
#'
#' @param object A \code{sandwich.ci} object generated by \code{\link{sandwich.ci}}, or a \code{sandwich.model} object generated by \code{\link{sandwich.model}}.
#' @param ... Ignored.
#'
#' @examples
#' data(hs.data)
#' hs.sw <- sandwich.model(object=hs.data, sampling.attr="Population", type="shp")
#' summary(object=hs.sw)
#'
#' hs.sw.ci <- sandwich.ci(object=hs.sw, level=.95)
#' summary(object=hs.sw.ci)
#'
#' @return Summarized statistics for the estimated mean, standard error, and confidence interval.
#'
#' @seealso \code{\link{sandwich.model}}, \code{\link{sandwich.ci}}
#'
#' @name summary
#'
NULL
# ---- End of roxygen documentation ----

#' @method summary sandwich.model
#' @export
#' @rdname summary
summary.sandwich.model <- function(object, ...){

  #--------------------------- Check inputs ----------------------------------
  if (!is.element("mean", names(object$object)) |
      !is.element("se", names(object$object)) |
      !is.element("df", names(object$object))){
    stop("Should run the sandwich.model function first.")
  }

  #---------------- Summarize Sandwich estimates ----------------------
  cols = c("mean", "se")
  print(paste("Sample size: ", object$sample_size))
  print(paste("Number of SSH strata: ", object$ssh_num))
  print(paste("Number of reporting units: ", object$reporting_num))
  summary(object$object[cols])
}


#' @method summary sandwich.ci
#' @export
#' @rdname summary
summary.sandwich.ci <- function(object, ...){

  #--------------------------- Check inputs ----------------------------------
  if (!is.element("ci.low", names(object$object$object)) |
      !is.element("ci.up", names(object$object$object))){
    stop("Should run the sandwich.ci function first.")
  }

  #---------------- Summarize Sandwich confidence intervals ----------------------
  cols = c("ci.low", "ci.up")
  summary(object$object$object[cols])
}


