% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pcNet.R
\name{pcNet}
\alias{pcNet}
\title{Computes a gene regulatory network based on principal component regression}
\usage{
pcNet(X, nComp = 3, scaleScores = TRUE, symmetric = FALSE, q = 0,
  verbose = TRUE)
}
\arguments{
\item{X}{A filtered and normalized gene expression matrix with cells as columns and genes as rows.}

\item{nComp}{An integer value. The number of principal components in PCA to generate the networks. Should be greater than 2 and lower than the total number of genes.}

\item{scaleScores}{A boolean value (\code{TRUE/FALSE}), if \code{TRUE}, the weights will be normalized such that the maximum absolute value is 1.}

\item{symmetric}{A boolean value (\code{TRUE/FALSE}), if \code{TRUE}, the weights matrix returned will be symmetric.}

\item{q}{A decimal value between 0 and 1. Defines the cut-off threshold of top q\% relationships to be returned.}

\item{verbose}{A boolean value (\code{TRUE/FALSE}), if \code{TRUE}, a progress bar is shown.}
}
\value{
A gene regulatory network in dgCMatrix format.
}
\description{
This function computes a gene regulatory network based on principal component regression (PCR), a technique based on principal component analysis. In PCR, the outcome variable is regressed over a \code{nComp} number of for principal components computed from a set of covariates to estimate the unknown regression coefficients in the model. \code{pcNet} function computes the PCR coefficients for each gene one at a time using all the others as covariates, to construct an all by all gene regulatory network.
}
\details{
Principal component regression may be broadly divided into three major steps: \enumerate{
\item Perform PCA on the observed covariates data matrix to obtain \code{nComp} number of the principal components.
\item Regress the observed vector of outcomes on the selected principal components as covariates, using ordinary least squares regression to get a vector of estimated regression coefficients
\item Transform this vector back to the scale of the actual covariates, using the eigenvectors corresponding to the selected principal components to get the final PCR estimator for estimating the regression coefficients characterizing the original model.
}
}
\examples{
library(scTenifoldNet)

# Simulating of a dataset following a negative binomial distribution with high sparcity (~67\%)
nCells = 2000
nGenes = 100
set.seed(1)
X <- rnbinom(n = nGenes * nCells, size = 20, prob = 0.98)
X <- round(X)
X <- matrix(X, ncol = nCells)
rownames(X) <- c(paste0('ng', 1:90), paste0('mt-', 1:10))

# Performing Single cell quality control
qcOutput <- scQC(
  X = X,
  minLibSize = 30,
  removeOutlierCells = TRUE,
  minPCT = 0.05,
  maxMTratio = 0.1
)

# Computing a single-cell gene regulatory network using principal component regression
# Non-symmetric
pcnetOutput <- pcNet(X = qcOutput, nComp = 3, scaleScores = TRUE, symmetric = FALSE, q = 0)
pcnetOutput[1:10,1:10]

# Symmetric
pcnetOutput <- pcNet(X = qcOutput, nComp = 3, scaleScores = TRUE, symmetric = TRUE, q = 0)
pcnetOutput[1:5,1:5]
}
\references{
\itemize{
\item Gill, Ryan, Somnath Datta, and Susmita Datta. "dna: An R package for differential network analysis." Bioinformation 10.4 (2014): 233.
\item Jolliffe, Ian T. "A note on the use of principal components in regression." Journal of the Royal Statistical Society: Series C (Applied Statistics) 31.3 (1982): 300-303.
\item Massy, William F. "Principal components regression in exploratory statistical research." Journal of the American Statistical Association 60.309 (1965): 234-256.
}
}
