% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutual_difference.R
\name{mutual_difference}
\alias{mutual_difference}
\title{Decomposes the difference between two M indices}
\usage{
mutual_difference(
  data1,
  data2,
  group,
  unit,
  weight = NULL,
  method = "shapley",
  se = FALSE,
  CI = 0.95,
  n_bootstrap = 100,
  base = exp(1),
  ...
)
}
\arguments{
\item{data1}{A data frame with same structure as \code{data2}.}

\item{data2}{A data frame with same structure as \code{data1}.}

\item{group}{A categorical variable or a vector of variables
contained in \code{data}. Defines the first dimension
over which segregation is computed.}

\item{unit}{A categorical variable or a vector of variables
contained in \code{data}. Defines the second dimension
over which segregation is computed.}

\item{weight}{Numeric. (Default \code{NULL})}

\item{method}{Either "shapley" (the default), "km" (Karmel and Maclachlan method), or
"mrc" (Mora and Ruiz-Castillo method).}

\item{se}{If \code{TRUE}, the segregation estimates are bootstrapped to provide
standard errors and to apply bias correction. The bias that is reported
has already been applied to the estimates (i.e. the reported estimates are "debiased")
(Default \code{FALSE})}

\item{CI}{If \code{se = TRUE}, compute the confidence (CI*100)% confidence interval
in addition to the bootstrap standard error.
This is based on percentiles of the bootstrap distribution, and a valid interpretation
relies on a larger number of bootstrap iterations. (Default \code{0.95})}

\item{n_bootstrap}{Number of bootstrap iterations. (Default \code{100})}

\item{base}{Base of the logarithm that is used in the calculation.
Defaults to the natural logarithm.}

\item{...}{Only used for additional arguments when
when \code{method} is set to \code{shapley} or \code{km}. See \link{ipf} for details.}
}
\value{
Returns a data.table with columns \code{stat} and \code{est}. The data frame contains
  the following rows defined by \code{stat}:
  \code{M1} contains the M for \code{data1}.
  \code{M2} contains the M for \code{data2}.
  \code{diff} is the difference between \code{M2} and \code{M1}.
  The sum of the five rows following \code{diff} equal \code{diff}.

  \code{additions} contains the change in M induces by \code{unit} and \code{group} categories
  present in \code{data2} but not \code{data1}, and \code{removals} the reverse.

  All methods return the following three terms:
  \code{unit_marginal} is the contribution of unit composition differences.
  \code{group_marginal} is the contribution of group composition differences.
  \code{structural} is the contribution unexplained by the marginal changes, i.e. the structural
    difference. Note that the interpretation of these terms depend on the exact method used.

  When using "km", one additional row is returned:
   \code{interaction} is the contribution of differences in the joint marginal distribution
     of \code{unit} and \code{group}.

  When "shapley_detailed" is used, an additional column "unit" is returned, along with
    six additional rows for each unit that is present in both \code{data1} and \code{data2}.
    The five rows have the following meaning:
    \code{p1} (\code{p2}) is the proportion of the unit in \code{data1} (\code{data2})
    once non-intersecting units/groups have been removed. The changes in local linkage are
    given by \code{ls_diff1} and \code{ls_diff2}, and their average is given by
    \code{ls_diff_mean}. The row named \code{total}
    summarizes the contribution of
    the unit towards structural change
    using the formula \code{.5 * p1 * ls_diff1 + .5 * p2 * ls_diff2}.
    The sum of all "total" components equals structural change.

  If \code{se} is set to \code{TRUE}, an additional column \code{se} contains
  the associated bootstrapped standard errors, an additional column \code{CI} contains
  the estimate confidence interval as a list column, an additional column \code{bias} contains
  the estimated bias, and the column \code{est} contains the bias-corrected estimates.
}
\description{
Uses one of three methods to decompose the difference between two M indices:
(1) "shapley" / "shapley_detailed": a method based on the Shapley decomposition
with a few advantages over the Karmel-Maclachlan method
(recommended and the default, Deutsch et al. 2006),
(2) "km": the method based on Karmel-Maclachlan (1988),
(3) "mrc": the method developed by Mora and Ruiz-Castillo (2009).
All methods have been extended to account for missing units/groups in either data input.
}
\details{
The Shapley method is an improvement over the Karmel-Maclachlan method (Deutsch et al. 2006).
It is based on several margins-adjusted data inputs
and yields symmetrical results (i.e. \code{data1} and \code{data2} can be switched).
When "shapley_detailed" is used, the structural component is further decomposed into
the contributions of individuals units.

The Karmel-Maclachlan method (Karmel and Maclachlan 1988) adjusts
the margins of \code{data1} to be similar to the margins of \code{data2}. This process
is not symmetrical.

The Shapley and Karmel-Maclachlan methods are based on iterative proportional fitting (IPF),
first introduced by Deming and Stephan (1940).
Depending on the size of the dataset, this may take a few seconds (see \link{ipf} for details).

The method developed by Mora and Ruiz-Castillo (2009) uses an algebraic approach to estimate the
size of the components. This will often yield substantively different results from the Shapley
and Karmel-Maclachlan methods. Note that this method is not symmetric in terms of what is
defined as \code{group} and \code{unit} categories, which may yield contradictory results.

A problem arises when there are \code{group} and/or \code{unit} categories in \code{data1}
that are not present in \code{data2} (or vice versa).
All methods estimate the difference only
for categories that are present in both datasets, and report additionally
the change in M that is induced by these cases as
\code{additions} (present in \code{data2}, but not in \code{data1}) and
\code{removals} (present in \code{data1}, but not in \code{data2}).
}
\examples{
# decompose the difference in school segregation between 2000 and 2005,
# using the Shapley method
mutual_difference(schools00, schools05, group = "race", unit = "school",
    weight = "n", method = "shapley", precision = .1)
# => the structural component is close to zero, thus most change is in the marginals.
# This method gives identical results when we switch the unit and group definitions,
# and when we switch the data inputs.

# the Karmel-Maclachlan method is similar, but only adjust the data in the forward direction...
mutual_difference(schools00, schools05, group = "school", unit = "race",
    weight = "n", method = "km", precision = .1)

# ...this means that the results won't be identical when we switch the data inputs
mutual_difference(schools05, schools00, group = "school", unit = "race",
    weight = "n", method = "km", precision = .1)

# the MRC method indicates a much higher structural change...
mutual_difference(schools00, schools05, group = "race", unit = "school",
    weight = "n", method = "mrc")

# ...and is not symmetric
mutual_difference(schools00, schools05, group = "school", unit = "race",
    weight = "n", method = "mrc")
}
\references{
W. E. Deming, F. F. Stephan. 1940. "On a Least Squares Adjustment of a Sampled Frequency Table
   When the Expected Marginal Totals are Known."
   The Annals of Mathematical Statistics 11(4): 427-444.

T. Karmel and M. Maclachlan. 1988.
  "Occupational Sex Segregation — Increasing or Decreasing?" Economic Record 64: 187-195.

R. Mora and J. Ruiz-Castillo. 2009. "The Invariance Properties of the
  Mutual Information Index of Multigroup Segregation." Research on Economic Inequality 17: 33-53.

J. Deutsch, Y. Flückiger, and J. Silber. 2009.
      "Analyzing Changes in Occupational Segregation: The Case of Switzerland (1970–2000)."
       Research on Economic Inequality 17: 171–202.
}
