% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auto_cov.R
\name{add_exo_cov}
\alias{add_exo_cov}
\alias{auto_exo_cov}
\alias{auto_cov}
\title{Add Covariances Between
Exogenous Variables}
\usage{
add_exo_cov(model, FUN = "sem", print = TRUE)

auto_exo_cov(model, FUN = "sem", print = TRUE)
}
\arguments{
\item{model}{The model syntax to
which the covariances are to be
added.}

\item{FUN}{Name (as string) of the
\code{lavaan} wrapper to be called.
Normally should be \code{"sem"}, the
default.}

\item{print}{Logical. Whether the
generated syntax should also be
printed by \code{\link[=cat]{cat()}}. Default is
\code{TRUE}.}
}
\value{
\code{\link[=add_exo_cov]{add_exo_cov()}} returns a one-element
character vector of the syntax, with
lines separated by "\\n". The
generated syntax is appended to the
input model syntax.

\code{\link[=auto_exo_cov]{auto_exo_cov()}} returns a
one-element character vector of the
generated syntax, with lines
separated by "\\n".
}
\description{
It generates the
'lavaan' model syntax for exogenous
variables in a \code{lavaan} model.
}
\details{
The function \code{\link[lavaan:sem]{lavaan::sem()}}
usually will set covariances between
"exogenous" variables free when
\code{fixed.x = FALSE} ("exogenous" is
defined here as variables that appear
on the right hand side but not on the
left hand side of the \code{~} operator`).
However, if a covariance between the
residual term of an endogenous
variable and an exogenous variable is
manually set to free, \code{\link[lavaan:sem]{lavaan::sem()}}
may not set the aforementioned
covariances free. Users will need to
free them manually, and there may be
a lot of them in some models.

This function gets a model syntax
and generates the syntax for these
covariances. Users can then inspect
it, modify it if necessary, and then
copy and paste it to the model
syntax.
}
\section{Functions}{
\itemize{
\item \code{add_exo_cov()}: Add covariances
between exogenous variables to the
model syntax and than return the
modified model syntax.

\item \code{auto_exo_cov()}: Generate the
model syntax for the covariances
between exogenous variables.

}}
\examples{

library(lavaan)
set.seed(8976223)
n <- 100
x <- rnorm(n)
m <- .5 * x + rnorm(n, 0, sqrt(.4))
z <- rnorm(n)
y <- .4 * m + .3 * z * m + rnorm(n, 0, .5)
dat <- data.frame(x, m, z, y)
dat$zm <- dat$z * dat$m
mod0 <-
"
m ~ x
y ~ m + z + zm
m ~~ z + zm
"
fit <- sem(mod0, dat, fixed.x = FALSE)

# Add covariances. Also printed by default.
mod0_cov <- add_exo_cov(mod0)

# Fit the model
fit_cov <- sem(mod0_cov, dat, fixed.x = FALSE)

# Manually adding the covariances
mod1 <-
"
m ~ x
y ~ m + z + zm
m ~~ z + zm
z  ~~ zm + x
zm ~~ x
"
fit1 <- sem(mod1, dat, meanstructure = TRUE, fixed.x = FALSE)

# Compare the results

# No manual covariances
fit

# Automatically generated covariances
fit_cov

# Manually added covariances
fit1


}
\author{
Shu Fai Cheung
\url{https://orcid.org/0000-0002-9871-9448}
}
