% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semmcci-mc.R
\name{MC}
\alias{MC}
\title{Monte Carlo Confidence Intervals}
\usage{
MC(
  object,
  R = 20000L,
  alpha = c(0.001, 0.01, 0.05),
  decomposition = "chol",
  pd = TRUE,
  tol = 1e-06
)
}
\arguments{
\item{object}{object of class \code{lavaan}.}

\item{R}{Positive integer.
Number of Monte Carlo replications.}

\item{alpha}{Numeric vector.
Significance level.
Default value is \code{alpha = c(0.001, 0.01, 0.05)}.}

\item{decomposition}{Character string.
Matrix decomposition of the sampling variance-covariance matrix for the data generation.
If \code{decomposition = "chol"}, use Cholesky decomposition.
If \code{decomposition = "eigen"}, use eigenvalue decomposition.
If \code{decomposition = "svd"}, use singular value decomposition.
If \code{decomposition = NULL}, try Cholesky decomposition.
If Cholesky decomposition fails, try eigenvalue decomposition.
Finally, if eigenvalue decomposition fails, try singular value decomposition.}

\item{pd}{Logical.
If \code{pd = TRUE}, check if the sampling variance-covariance matrix is positive definite using \code{tol} if \code{decomposition \%in\% c("eigen", "svd")}.}

\item{tol}{Numeric.
Tolerance used for \code{pd}..}
}
\value{
Returns an object of class \code{semmcci} which is a list with the following elements:
\describe{
\item{\code{R}}{Number of Monte Carlo replications.}
\item{\code{alpha}}{Significance level specified.}
\item{\code{lavaan}}{\code{lavaan} object.}
\item{\code{decomposition}}{Matrix decomposition used to generate multivariate normal random variates.}
\item{\code{thetahat}}{Parameter estimates.}
\item{\code{thetahatstar}}{Sampling distribution of parameter estimates.}
}
}
\description{
Calculates Monte Carlo confidence intervals
for free and defined parameters
}
\details{
A sampling distribution of parameter estimates is generated
from the multivariate normal distribution
using the parameter estimates and the sampling variance-covariance matrix.
Confidence intervals for free and defined parameters
are generated using the simulated sampling distribution.
Parameters can be defined using the \verb{:=} operator
in the \code{lavaan} model syntax.
}
\examples{
library(semmcci)
library(lavaan)

# Generate Data ------------------------------------------------------------
n <- 1000
a <- 0.50
b <- 0.50
cp <- 0.25
s2_em <- 1 - a^2
s2_ey <- 1 - cp^2 - a^2 * b^2 - b^2 * s2_em - 2 * cp * a * b
em <- rnorm(n = n, mean = 0, sd = sqrt(s2_em))
ey <- rnorm(n = n, mean = 0, sd = sqrt(s2_ey))
X <- rnorm(n = n)
M <- a * X + em
Y <- cp * X + b * M + ey
df <- data.frame(X, M, Y)

# Fit Model in lavaan ------------------------------------------------------
model <- "
  Y ~ cp * X + b * M
  M ~ a * X
  indirect := a * b
  direct := cp
  total := cp + (a * b)
"
fit <- sem(data = df, model = model)

# Monte Carlo --------------------------------------------------------------
MC(
  fit,
  R = 100L, # use a large value e.g., 20000L for actual research
  alpha = c(0.001, 0.01, 0.05)
)
}
\author{
Ivan Jacob Agaloos Pesigan
}
\keyword{mc}
