% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semmcci-mc-mi.R
\name{MCMI}
\alias{MCMI}
\title{Monte Carlo Confidence Intervals (Multiple Imputation)}
\usage{
MCMI(
  lav,
  mi,
  R = 20000L,
  alpha = c(0.001, 0.01, 0.05),
  decomposition = "eigen",
  pd = TRUE,
  tol = 1e-06,
  seed = NULL
)
}
\arguments{
\item{lav}{Object of class \code{lavaan}.}

\item{mi}{Object of class \code{mids} (output of \code{\link[mice:mice]{mice::mice()}}),
object of class \code{amelia} (output of \code{\link[Amelia:amelia]{Amelia::amelia()}}),
or a list of multiply imputed data sets.}

\item{R}{Positive integer.
Number of Monte Carlo replications.}

\item{alpha}{Numeric vector.
Significance level \eqn{\alpha}.}

\item{decomposition}{Character string.
Matrix decomposition of the sampling variance-covariance matrix
for the data generation.
If \code{decomposition = "chol"}, use Cholesky decomposition.
If \code{decomposition = "eigen"}, use eigenvalue decomposition.
If \code{decomposition = "svd"}, use singular value decomposition.}

\item{pd}{Logical.
If \code{pd = TRUE},
check if the sampling variance-covariance matrix
is positive definite using \code{tol}.}

\item{tol}{Numeric.
Tolerance used for \code{pd}.}

\item{seed}{Integer.
Random seed for reproducibility.}
}
\value{
Returns an object of class \code{semmcci} which is
a list with the following elements:
\describe{
\item{call}{Function call.}
\item{args}{List of function arguments.}
\item{thetahat}{Parameter estimates \eqn{\hat{\theta}}.}
\item{thetahatstar}{Sampling distribution of parameter estimates
\eqn{\hat{\theta}^{\ast}}.}
\item{fun}{Function used ("MCMI").}
}
}
\description{
Calculates Monte Carlo confidence intervals
for free and defined parameters.
Missing values are handled using multilple imputation.
}
\details{
A sampling distribution of parameter estimates is generated
from the multivariate normal distribution
using the parameter estimates and the sampling variance-covariance matrix
obtained using multiple imputation.
Confidence intervals for free and defined parameters
are generated using the simulated sampling distribution.
Parameters can be defined using the \verb{:=} operator
in the \code{lavaan} model syntax.
}
\examples{
library(semmcci)
library(lavaan)

# Data ---------------------------------------------------------------------
data("Tal.Or", package = "psych")
df <- mice::ampute(Tal.Or)$amp

# Monte Carlo (Multiple Imputation) ----------------------------------------
## Multiple Imputation -----------------------------------------------------
mi <- mice::mice(
  data = df,
  print = FALSE,
  m = 5L, # use a large value e.g., 100L for actual research,
  seed = 42
)

## Fit Model in lavaan -----------------------------------------------------
model <- "
  reaction ~ cp * cond + b * pmi
  pmi ~ a * cond
  cond ~~ cond
  indirect := a * b
  direct := cp
  total := cp + (a * b)
"
fit <- sem(data = df, model = model) # use default listwise deletion

## MCMI() ------------------------------------------------------------------
MCMI(
  fit,
  mi = mi,
  R = 20L, # use a large value e.g., 20000L for actual research
  alpha = 0.05
)

}
\references{
Pesigan, I. J. A., & Cheung, S. F. (2023).
Monte Carlo confidence intervals for the indirect effect with missing data.
\emph{Behavior Research Methods}.
\doi{10.3758/s13428-023-02114-4}

Rubin, D. B. (1987).
\emph{Multiple imputation for nonresponse in surveys}.
John Wiley & Sons, Inc.
}
\seealso{
Other Monte Carlo in Structural Equation Modeling Functions: 
\code{\link{MCStd}()},
\code{\link{MC}()}
}
\concept{Monte Carlo in Structural Equation Modeling Functions}
\keyword{mc}
\keyword{semmcci}
