#'@title Add Standard Error Estimates to Parameter Estimates (Edge
#' Labels)
#'
#'@description Add standard error estimates, in parentheses, to
#'   parameter estimates (edge labels) in a [qgraph::qgraph] object.
#'
#'@details Modify a [qgraph::qgraph] object generated by
#'   \code{\link[semPlot]{semPaths}} (currently in parentheses) to the
#'   labels. Require the original object used in the
#'   \code{\link[semPlot]{semPaths}} call.
#'
#'Currently supports only plots based on \code{\link[lavaan]{lavaan}}
#'output.
#'
#'This function is a variant of, and can be combined with,  the
#'  \code{\link{mark_sig}} function.
#'
#'@return If the input is a [qgraph::qgraph] object, the function
#'  returns a qgraph based on the original one, with standard error
#'  estimates appended. If the input is a list of qgraph objects, the
#'  function returns a list of the same length.
#'
#'@param semPaths_plot A qgraph object generated by
#'   \code{\link[semPlot]{semPaths}}, or a similar qgraph object
#'   modified by other [semptools] functions.
#'
#'@param object The object used by semPaths to generate the plot. Use
#'   the same argument name used in \code{\link[semPlot]{semPaths}} to
#'   make the meaning of this argument obvious. Currently only object
#'   of class \linkS4class{lavaan} is supported.
#'
#'@param sep A character string to separate the coefficient and the
#'   standard error (in parentheses). Default to " " (one space). Use
#'   \code{"\n"} to enforce a line break.
#'
#'@param digits Integer indicating number of decimal places for the
#'   appended standard errors. Default is 2L.
#'
#'@param ests A data.frame from the
#'   \code{\link[lavaan]{parameterEstimates}} function. Only used when
#'   \code{object} is not specified.
#'
#'@examples
#'mod_pa <-
#'   'x1 ~~ x2
#'    x3 ~  x1 + x2
#'    x4 ~  x1 + x3
#'   '
#'fit_pa <- lavaan::sem(mod_pa, pa_example)
#'lavaan::parameterEstimates(fit_pa)[ , c("lhs", "op", "rhs",
#'                                        "est", "pvalue", "se")]
#'m <- matrix(c("x1",   NA,   NA,
#'                NA, "x3", "x4",
#'              "x2",   NA,   NA), byrow = TRUE, 3, 3)
#'p_pa <- semPlot::semPaths(fit_pa, whatLabels = "est",
#'                          style = "ram",
#'                          nCharNodes = 0, nCharEdges = 0,
#'                          layout = m)
#'p_pa2 <- mark_se(p_pa, fit_pa)
#'plot(p_pa2)
#'
#'mod_cfa <-
#'  'f1 =~ x01 + x02 + x03
#'   f2 =~ x04 + x05 + x06 + x07
#'   f3 =~ x08 + x09 + x10
#'   f4 =~ x11 + x12 + x13 + x14
#'  '
#'fit_cfa <- lavaan::sem(mod_cfa, cfa_example)
#'lavaan::parameterEstimates(fit_cfa)[ , c("lhs", "op", "rhs",
#'                                         "est", "pvalue", "se")]
#'p_cfa <- semPlot::semPaths(fit_cfa, whatLabels = "est",
#'                           style = "ram",
#'                           nCharNodes = 0, nCharEdges = 0)
#'# Place standard errors on a new line
#'p_cfa2 <- mark_se(p_cfa, fit_cfa, sep = "\n")
#'plot(p_cfa2)
#'
#'mod_sem <-
#' 'f1 =~ x01 + x02 + x03
#'  f2 =~ x04 + x05 + x06 + x07
#'  f3 =~ x08 + x09 + x10
#'  f4 =~ x11 + x12 + x13 + x14
#'  f3 ~  f1 + f2
#'  f4 ~  f1 + f3
#' '
#'fit_sem <- lavaan::sem(mod_sem, sem_example)
#'lavaan::parameterEstimates(fit_sem)[ , c("lhs", "op", "rhs",
#'                                         "est", "pvalue", "se")]
#'p_sem <- semPlot::semPaths(fit_sem, whatLabels = "est",
#'                           style = "ram",
#'                           nCharNodes = 0, nCharEdges = 0)
#'# Mark significance, and then add standard errors
#'p_sem2 <- mark_sig(p_sem, fit_sem)
#'p_sem3 <- mark_se(p_sem2, fit_sem, sep = "\n")
#'plot(p_sem3)
#'
#'@importFrom rlang .data
#'@export

mark_se <- function(semPaths_plot, object, sep = " ", digits = 2L,
                    ests = NULL) {
  if ("triangle" %in% semPaths_plot$graphAttributes$Nodes$shape) {
    rlang::abort(paste("The semPaths plot seems to have one or",
                       "more intercepts. Models with intercepts",
                       "are not supported yet. Consider setting",
                       "'intercepts = FALSE' in semPaths."))
  }
  if (is.null(ests)) {
    ests <- lavaan::parameterEstimates(object, se = TRUE, ci = FALSE,
                                       zstat = FALSE, pvalue = FALSE)
  }
  if (inherits(semPaths_plot, "list")) {
    if (length(semPaths_plot) != dplyr::n_distinct(ests$group)) {
      rlang::abort(paste("length of qgraph list does not match",
                         "number of groups in model fit object."))
    }
    ests_list <- split(ests, ests$group)
    mapply(mark_se, semPaths_plot, ests = ests_list, SIMPLIFY = FALSE)
  } else {
    if (!missing(object) && lavaan::lavInspect(object, "ngroups") > 1) {
      rlang::abort(paste("length of qgraph list does not match", 
                         "number of groups in model fit object."))
    }
    Nodes_names <- semPaths_plot$graphAttributes$Nodes$names
    if (!is.null(names(Nodes_names))) {
      Nodes_names <- names(Nodes_names)
    }
    if (!all(Nodes_names %in% union(ests$lhs, ests$rhs))) {
      abort_nomatch(Nodes_names, union(ests$lhs, ests$rhs))
    }
    Edgelist <- data.frame(
      from_names = Nodes_names[semPaths_plot$Edgelist$from],
      to_names   = Nodes_names[semPaths_plot$Edgelist$to],
      semPaths_plot$Edgelist, stringsAsFactors = FALSE)
    graphAttributes_Edges <- data.frame(
      from_names = Nodes_names[semPaths_plot$Edgelist$from],
      to_names   = Nodes_names[semPaths_plot$Edgelist$to],
      semPaths_plot$graphAttributes$Edges, stringsAsFactors = FALSE)
    edge_labels <- dplyr::select(graphAttributes_Edges,
                                 .data$from_names, .data$to_names, labels)
    ests_ses <- dplyr::select(ests, .data$lhs, .data$rhs, .data$se)
    ests_ses_rev <- dplyr::rename(ests_ses, se_rev = .data$se)
    edge_ses <- dplyr::left_join(edge_labels, ests_ses,
                                 by = c("from_names" = "rhs",
                                        "to_names" = "lhs"))
    edge_ses <- dplyr::left_join(edge_ses, ests_ses_rev,
                                 by = c("from_names" = "lhs",
                                        "to_names" = "rhs"))
    edge_ses <- dplyr::mutate(edge_ses,
                              se = pmax(.data$se, .data$se_rev, na.rm = TRUE))
    labels_old <- semPaths_plot$graphAttributes$Edges$labels
    labels_new <- paste0(labels_old, sep,
                         "(", formatC(edge_ses$se, digits, format = "f"), ")")
    semPaths_plot$graphAttributes$Edges$labels <- labels_new
    semPaths_plot
  }
}
