% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CalcPairLL.R
\name{CalcPairLL}
\alias{CalcPairLL}
\title{Calculate Likelihoods for Alternative Relationships}
\usage{
CalcPairLL(
  Pairs = NULL,
  GenoM = NULL,
  Pedigree = NULL,
  LifeHistData = NULL,
  AgePrior = TRUE,
  SeqList = NULL,
  Complex = "full",
  Herm = "no",
  Err = 1e-04,
  ErrFlavour = "version2.0",
  Tassign = 0.5,
  Tfilter = -2,
  quiet = FALSE,
  Plot = TRUE
)
}
\arguments{
\item{Pairs}{dataframe with columns \code{ID1} and \code{ID2}, and
 optionally
 \describe{
\item{Sex1}{Sex of ID1, 1=female, 2=male, 3=unknown, or \code{NA} to take
  from \code{LifeHistData}. The sex of individuals occurring as parent in
  \code{Pedigree} cannot be altered.}
\item{Sex2}{Sex of ID2}
\item{AgeDif}{Age difference in whole time units, BirthYear1 - BirthYear2
  (i.e. positive if ID2 is born before ID1). If \code{NA}, calculated from
  \code{LifeHistData}. Use '999' to explicitly specify 'unknown'.}
\item{focal}{relationship character abbreviation; PO, FS, HS, GP or U. See
Details for its effect and explanation of abbreviations. Default: U}
\item{patmat}{1=maternal relatives, 2=paternal relatives. Only relevant for
  HS & GP, for which it defaults to Sex1, or 1 if Sex1=3, but is currently
  only predictably implemented for pairs of two genotyped individuals.
  Always equal to Sex2 for PO pairs when Sex2 is known.}
\item{dropPar1}{Drop the parents of \code{ID1} before calculating the pair
likelihood, rather than conditioning on them; choose from 'none', 'dam',
'sire', or 'both'. See example. If e.g. the pair shares a common mother,
'none' and 'sire' will condition on this shared mother and not calculate the
likelihood that they are maternal siblings, while dropPar1='dam' or 'both'
will calculate that likelihood, and the other likelihoods as if the mother
of ID1 were unknown.}
\item{dropPar2}{as \code{dropPar1}, for \code{ID2}}
}}

\item{GenoM}{numeric matrix with genotype data: One row per individual, and
one column per SNP, coded as 0, 1, 2 or -9 (missing). See also
\code{\link{GenoConvert}}.}

\item{Pedigree}{dataframe with columns id-dam-sire; likelihoods will be
calculated conditional on the pedigree. May include non-genotyped
individuals, which will be treated as dummy individuals.}

\item{LifeHistData}{dataframe with 3 columns (optionally 5):
 \describe{
 \item{ID}{max. 30 characters long}
 \item{Sex}{1 = female, 2 = male, 3 = unknown, 4 = hermaphrodite,
           other numbers or NA = unknown}
\item{BirthYear }{birth or hatching year, integer, with missing values as NA
  or any negative value.}
\item{BY.min}{minimum birth year, only used if BirthYear is missing}
\item{BY.max}{maximum birth year, only used if BirthYear is missing} }
If the species has multiple generations per year, use an integer coding such
that the candidate parents' `Birth year' is at least one smaller than their
putative offspring's. Column names are ignored, so ensure column order is ID
- sex - birth year (- BY.min - BY.max). Individuals do not need to be in the
same order as in `GenoM', nor do all genotyped individuals need to be
included.}

\item{AgePrior}{logical (\code{TRUE/FALSE}) whether to estimate the ageprior
from \code{Pedigree} and \code{LifeHistData}, or a matrix as generated by
\code{\link{MakeAgePrior}} and included in the \code{\link{sequoia}}
output. The \code{AgePrior} affects which relationships are considered
possible: only those where \eqn{P(A|R) / P(A) > 0}.  When \code{TRUE},
\code{\link{MakeAgePrior}} is called using its default values. When
\code{FALSE}, all relationships are considered possible for all age
differences, except that parent-offspring pairs cannot have age difference
zero, and grand-parental pairs have an age difference of at least two.}

\item{SeqList}{list with output from \code{\link{sequoia}}. If input
parameter \code{Pedigree=NULL}, \code{SeqList$Pedigree} will be used if
present, and \code{SeqList$PedigreePar} otherwise. If \code{SeqList$Specs}
is present, input parameters with the same name as its items are ignored.
The list elements 'LifeHist', 'AgePriors', and 'ErrM' are also used if
present, and override the corresponding input parameters.}

\item{Complex}{Breeding system complexity. Either "full" (default), "simp"
(simplified, no explicit consideration of inbred relationships), "mono"
(monogamous).}

\item{Herm}{Hermaphrodites, either "no", "A" (distinguish between dam and
sire role, default if at least 1 individual with sex=4), or "B" (no
distinction between dam and sire role). Both of the latter deal with
selfing.}

\item{Err}{estimated genotyping error rate, as a single number or 3x3 matrix.
Details below. The error rate is presumed constant across SNPs, and
missingness is presumed random with respect to actual genotype.}

\item{ErrFlavour}{function that takes \code{Err} (single number) as input,
and returns a 3x3 matrix of observed (columns) conditional on actual (rows)
genotypes, or choose from inbuilt options 'version2.0', 'version1.3', or
'version1.1', referring to the sequoia version in which they were the
default. Ignored if \code{Err} is a matrix. See \code{\link{ErrToM}}.}

\item{Tassign}{minimum LLR required for acceptance of proposed relationship,
relative to next most likely relationship. Higher values result in more
conservative assignments. Must be zero or positive.}

\item{Tfilter}{threshold log10-likelihood ratio (LLR) between a proposed
relationship versus unrelated, to select candidate relatives. Typically a
negative value, related to the fact that unconditional likelihoods are
calculated during the filtering steps. More negative values may decrease
non-assignment, but will increase computational time.}

\item{quiet}{logical, suppress messages}

\item{Plot}{logical, display scatter plots by \code{\link{PlotPairLL}}.}
}
\value{
The \code{Pairs} dataframe including all optional columns listed
  above, plus the additional columns:
 \item{LL_xx}{Log10-Likelihood of this pair having relationship xx, with xx
   being one of PO, FS, etc. as detailed below.}
 \item{TopRel}{Abbreviation of most likely relationship}
 \item{LLR}{Likelihood ratio between most-likely and second most likely
   relationships}

\strong{Relationship abbreviations:}
  \item{PO}{Parent - offspring}
  \item{FS}{Full siblings}
  \item{HS}{Half siblings}
  \item{GP}{Grandparent}
  \item{FA}{Full avuncular}
  \item{HA}{Half avuncular and other 3rd degree relationships}
  \item{U}{Unrelated}
  \item{2nd}{Unclear which type of 2nd degree relatives
    (HS, GP, or FA)}
  \item{??}{Unclear which type of 1st, 2nd or 3rd degree
    relatives}

\strong{Likelihood special codes:}
  \item{222}{Maybe (via) other parent (e.g. focal="GP", but as likely to be
    maternal as paternal grandparent, and therefore not assignable)}
  \item{333}{Excluded from comparison (shouldn't occur)}
  \item{444}{Not implemented (e.g. would create an odd double/triple
    relationship in combination with the provided pedigree)}
  \item{777}{Impossible (e.g. cannot be both full sibling and grandparent)}
  \item{888}{Already assigned in the provided pedigree (see \code{dropPar}
    arguments)}
  \item{999}{NA}
}
\description{
For each specified pair of individuals, calculate the
  log10-likelihoods of being PO, FS, HS, GP, FA, HA, U (see Details).
  Individuals must be genotyped or have at least one genotyped offspring.

  \strong{NOTE} values \eqn{>0} are various \code{NA} types, see 'Likelihood
  special codes' in 'Value' section below.
}
\details{
The same pair may be included multiple times, e.g. with different
  sex, age difference, or focal relationship, to explore their effect on the
  likelihoods. Likelihoods are only calculated for relationships that are
  possible given the age difference, e.g. PO (parent-offspring) is not
  calculated for pairs with an age difference of 0.

  Non-genotyped individuals can be included if they have at least one
  genotyped offspring and can be turned into a dummy (see
  \code{\link{getAssignCat}}); to establish this a pedigree must be provided.

  \strong{Warning 1}: There is no check whether the input pedigree is genetically
  sensible, it is simply conditioned upon. Checking whether a pedigree is
  compatible with the SNP data can be done with \code{\link{CalcOHLLR}}.

  \strong{Warning 2}: Conditioning on a \code{Pedigree} can make computation
  orders of magnitude slower.
}
\section{Double relationships & focal relationship}{

  Especially when Complex='full', not only the seven relationship
  alternatives listed above are considered, but a whole range of possible
  double and even triple relationships. For example, mother A and offspring B
  (PO) may also be paternal half-siblings (HS, A and A's mother mated with
  same male), grandmother and grand-offspring (GP, B's father is A's son), or
  paternal aunt (B's father is a full or half sib of A).

  The likelihood reported as 'LL_PO' is the most-likely one of the possible
  alternatives, among those that are not impossible due to age differences or
  due to the pedigree (as reconstructed up to that point). Whether e.g. the
  likelihood to be both PO & HS is counted as PO or as HS, depends on the
  situation and is determined by the variable 'focal': During parentage
  assignment, it is counted as PO but not HS, while during sibship
  clustering, it is counted as HS but not PO -- not omitting from the
  alternative relationship would result in a deadlock.
}

\examples{
# likelihoods underlying parent LLR in pedigree:
data(LH_HSg5, SimGeno_example)
Seq.HSg5 <- sequoia(SimGeno_example, LH_HSg5, Module="par")
tail(Seq.HSg5$PedigreePar)
# take bottom 3 individuals:
Pairs <- data.frame(ID1 = c("a01190", "b01191", "a01192"),
                    ID2 = rep(c("a00012", "b00007"), each=3),
                    AgeDif = 1, focal = "PO")
# LLRdam & LLRsire:
CalcPairLL(Pairs, SimGeno_example)
# LLRpair is min. of dam & sire LLR, conditional on co-parent:
CalcPairLL(cbind(Pairs, dropPar1=rep(c("dam", "sire"), each=3)),
           SimGeno_example, SeqList = Seq.HSg5)

\donttest{
# likelihoods underlying LLR in getMaybeRel output:
data(Ped_griffin, SeqOUT_griffin, package="sequoia")
Geno.griffin <- SimGeno(Ped_griffin, nSnp=200, SnpError = 0.01, ParMis=0.4)
MR <- GetMaybeRel(GenoM = Geno.griffin,
                  LifeHistData = SeqOUT_griffin$LifeHist,
                  Module = "par", Err = 0.001)
FivePairs <- MR$MaybePar[1:5, c("ID1", "ID2", "Sex1", "Sex2")]
FivePairs$AgeDif <- NA   # pretend unknown age differences
PairLL <- CalcPairLL(Pairs = rbind( cbind(FivePairs, focal = "PO"),
                                    cbind(FivePairs, focal = "HS"),
                                    cbind(FivePairs, focal = "GP")),
                     GenoM = Geno.griffin,
                     Err = 0.005, Plot=FALSE)
PairLL[c(1, 6, 11), ]
# LL(FS)==222 : HSHA, HSGP, FAHA more likely than FS
# LL(GP) higher when focal=HS: GP via 'other' parent also considered
# LL(FA) higher when focal=PO: FAHA, or FS of 'other' parent
}

}
\seealso{
\code{\link{PlotPairLL}} to plot alternative relationship pairs from
  the output; \code{\link{CalcOHLLR}} to calculate LLR for parents &
  parent-pairs in a pedigree; \code{\link{GetRelM}} to find all pairwise
  relatives according to the pedigree; \code{\link{GetMaybeRel}} to get
  likely relative pairs not in the pedigree.
}
