% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bi_selection.R
\name{BiSelection}
\alias{BiSelection}
\title{Stability selection of predictors and/or outcomes}
\usage{
BiSelection(
  xdata,
  ydata = NULL,
  group_x = NULL,
  group_y = NULL,
  LambdaX = NULL,
  LambdaY = NULL,
  AlphaX = NULL,
  AlphaY = NULL,
  ncomp = 1,
  scale = TRUE,
  pi_list = seq(0.6, 0.9, by = 0.01),
  K = 100,
  tau = 0.5,
  seed = 1,
  n_cat = 3,
  family = "gaussian",
  implementation = SparsePLS,
  resampling = "subsampling",
  cpss = FALSE,
  PFER_method = "MB",
  PFER_thr = Inf,
  FDP_thr = Inf,
  n_cores = 1,
  output_data = FALSE,
  verbose = TRUE,
  beep = NULL,
  ...
)
}
\arguments{
\item{xdata}{matrix of predictors with observations as rows and variables as
columns.}

\item{ydata}{optional vector or matrix of outcome(s). If \code{family} is set
to \code{"binomial"} or \code{"multinomial"}, \code{ydata} can be a vector
with character/numeric values or a factor.}

\item{group_x}{vector encoding the grouping structure among predictors. This
argument indicates the number of variables in each group. Only used for
models with group penalisation (e.g. \code{implementation=GroupPLS} or
\code{implementation=SparseGroupPLS}).}

\item{group_y}{optional vector encoding the grouping structure among
outcomes. This argument indicates the number of variables in each group.
Only used if \code{implementation=GroupPLS} or
\code{implementation=SparseGroupPLS}.}

\item{LambdaX}{matrix of parameters controlling the number of selected
variables (for sparse PCA/PLS) or groups (for group and sparse group PLS)
in X.}

\item{LambdaY}{matrix of parameters controlling the number of selected
variables (for sparse PLS) or groups (for group or sparse group PLS) in Y.
Only used if \code{family="gaussian"}.}

\item{AlphaX}{matrix of parameters controlling the level of sparsity within
groups in X. Only used if \code{implementation=SparseGroupPLS}.}

\item{AlphaY}{matrix of parameters controlling the level of sparsity within
groups in X. Only used if \code{implementation=SparseGroupPLS} and
\code{family="gaussian"}.}

\item{ncomp}{number of components.}

\item{scale}{logical indicating if the data should be scaled (i.e.
transformed so that all variables have a standard deviation of one).}

\item{pi_list}{vector of thresholds in selection proportions. If
\code{n_cat=3}, these values must be \code{>0.5} and \code{<1}. If
\code{n_cat=2}, these values must be \code{>0} and \code{<1}.}

\item{K}{number of resampling iterations.}

\item{tau}{subsample size. Only used if \code{resampling="subsampling"} and
\code{cpss=FALSE}.}

\item{seed}{value of the seed to initialise the random number generator and
ensure reproducibility of the results (see \code{\link[base]{set.seed}}).}

\item{n_cat}{number of categories used to compute the stability score.
Possible values are 2 or 3.}

\item{family}{type of PLS model. This parameter must be set to
\code{family="gaussian"} for continuous outcomes, or to
\code{family="binomial"} for categorical outcomes. Only used if
\code{ydata} is provided.}

\item{implementation}{function to use for feature selection. Possible
functions are: \code{SparsePCA}, \code{SparsePLS}, \code{GroupPLS},
\code{SparseGroupPLS}.}

\item{resampling}{resampling approach. Possible values are:
\code{"subsampling"} for sampling without replacement of a proportion
\code{tau} of the observations, or \code{"bootstrap"} for sampling with
replacement generating a resampled dataset with as many observations as in
the full sample. Alternatively, this argument can be a function to use for
resampling. This function must use arguments named \code{data} and
\code{tau} and return the IDs of observations to be included in the
resampled dataset.}

\item{cpss}{logical indicating if complementary pair stability selection
should be done. For this, the algorithm is applied on two non-overlapping
subsets of half of the observations. A feature is considered as selected if
it is selected for both subsamples. With this method, the data is split
\code{K/2} times (\code{K} models are fitted). Only used if
\code{PFER_method="MB"}.}

\item{PFER_method}{method used to compute the upper-bound of the expected
number of False Positives (or Per Family Error Rate, PFER). If
\code{PFER_method="MB"}, the method proposed by Meinshausen and Bühlmann
(2010) is used. If \code{PFER_method="SS"}, the method proposed by Shah and
Samworth (2013) under the assumption of unimodality is used.}

\item{PFER_thr}{threshold in PFER for constrained calibration by error
control. If \code{PFER_thr=Inf} and \code{FDP_thr=Inf}, unconstrained
calibration is used (the default).}

\item{FDP_thr}{threshold in the expected proportion of falsely selected
features (or False Discovery Proportion) for constrained calibration by
error control. If \code{PFER_thr=Inf} and \code{FDP_thr=Inf}, unconstrained
calibration is used (the default).}

\item{n_cores}{number of cores to use for parallel computing (see
\code{\link[parallel]{mclapply}}). Only available on Unix systems.}

\item{output_data}{logical indicating if the input datasets \code{xdata} and
\code{ydata} should be included in the output.}

\item{verbose}{logical indicating if a loading bar and messages should be
printed.}

\item{beep}{sound indicating the end of the run. Possible values are:
\code{NULL} (no sound) or an integer between 1 and 11 (see argument
\code{sound} in \code{\link[beepr]{beep}}).}

\item{...}{additional parameters passed to the functions provided in
\code{implementation} or \code{resampling}.}
}
\value{
An object of class \code{bi_selection}. A list with: \item{summary}{a
  matrix of the best stability scores and corresponding parameters
  controlling the level of sparsity in the underlying algorithm for different
  numbers of components. Possible columns include: \code{comp} (component
  index), \code{nx} (number of predictors to include, parameter of the
  underlying algorithm), \code{alphax} (sparsity within the predictor groups,
  parameter of the underlying algorithm), \code{pix} (threshold in selection
  proportion for predictors), \code{ny} (number of outcomes to include,
  parameter of the underlying algorithm), \code{alphay} (sparsity within the
  outcome groups, parameter of the underlying algorithm), \code{piy}
  (threshold in selection proportion for outcomes), \code{S} (stability
  score). Columns that are not relevant to the model are not reported (e.g.
  \code{alpha_x} and \code{alpha_y} are not returned for sparse PLS models).}
  \item{summary_full}{a matrix of the best stability scores for different
  combinations of parameters controlling the sparsity and components.}
  \item{selectedX}{a binary matrix encoding stably selected predictors.}
  \item{selpropX}{a matrix of calibrated selection proportions for
  predictors.} \item{selectedY}{a binary matrix encoding stably selected
  outcomes. Only returned for PLS models.} \item{selpropY}{a matrix of
  calibrated selection proportions for outcomes. Only returned for PLS
  models.} \item{selected}{a binary matrix encoding stable relationships
  between predictor and outcome variables. Only returned for PLS models.}
  \item{selectedX_full}{a binary matrix encoding stably selected predictors.}
  \item{selpropX_full}{a matrix of selection proportions for predictors.}
  \item{selectedY_full}{a binary matrix encoding stably selected outcomes.
  Only returned for PLS models.} \item{selpropY_full}{a matrix of selection
  proportions for outcomes. Only returned for PLS models.} \item{coefX}{an
  array of estimated loadings coefficients for the different components
  (rows), for the predictors (columns), as obtained across the \code{K}
  visited models (along the third dimension).} \item{coefY}{an array of
  estimated loadings coefficients for the different components (rows), for
  the outcomes (columns), as obtained across the \code{K} visited models
  (along the third dimension). Only returned for PLS models.} \item{method}{a
  list with \code{type="bi_selection"} and values used for arguments
  \code{implementation}, \code{family}, \code{scale}, \code{resampling},
  \code{cpss} and \code{PFER_method}.} \item{params}{a list with values used
  for arguments \code{K}, \code{group_x}, \code{group_y}, \code{LambdaX},
  \code{LambdaY}, \code{AlphaX}, \code{AlphaY}, \code{pi_list}, \code{tau},
  \code{n_cat}, \code{pk}, \code{n} (number of observations),
  \code{PFER_thr}, \code{FDP_thr} and \code{seed}. The datasets \code{xdata}
  and \code{ydata} are also included if \code{output_data=TRUE}.} The rows of
  \code{summary} and columns of \code{selectedX}, \code{selectedY},
  \code{selpropX}, \code{selpropY}, \code{selected}, \code{coefX} and
  \code{coefY} are ordered in the same way and correspond to components and
  parameter values stored in \code{summary}. The rows of \code{summary_full}
  and columns of \code{selectedX_full}, \code{selectedY_full},
  \code{selpropX_full} and \code{selpropY_full} are ordered in the same way
  and correspond to components and parameter values stored in
  \code{summary_full}.
}
\description{
Performs stability selection for dimensionality reduction. The underlying
variable selection algorithm (e.g. sparse PLS) is run with different
combinations of parameters controlling the sparsity (e.g. number of selected
variables per component) and thresholds in selection proportions. These
hyper-parameters are jointly calibrated by maximisation of the stability
score.
}
\details{
In stability selection, a feature selection algorithm is fitted on
  \code{K} subsamples (or bootstrap samples) of the data with different
  parameters controlling the sparsity (\code{LambdaX}, \code{LambdaY},
  \code{AlphaX}, and/or \code{AlphaY}). For a given (set of) sparsity
  parameter(s), the proportion out of the \code{K} models in which each
  feature is selected is calculated. Features with selection proportions
  above a threshold pi are considered stably selected. The stability
  selection model is controlled by the sparsity parameter(s) (denoted by
  \eqn{\lambda}) for the underlying algorithm, and the threshold in selection
  proportion:

  \eqn{V_{\lambda, \pi} = \{ j: p_{\lambda}(j) \ge \pi \} }

  For sparse and sparse group dimensionality reduction, "feature" refers to
  variable (variable selection model). For group PLS, "feature" refers to
  group (group selection model). For (sparse) group PLS, groups need to be
  defined \emph{a priori} and specified in arguments \code{group_x} and/or
  \code{group_y}.

  These parameters can be calibrated by maximisation of a stability score
  (see \code{\link{StabilityScore}}) derived from the likelihood under the
  assumption of uniform (uninformative) selection:

  \eqn{S_{\lambda, \pi} = -log(L_{\lambda, \pi})}

  It is strongly recommended to examine the calibration plot carefully to
  check that the grids of parameters \code{Lambda} and \code{pi_list} do not
  restrict the calibration to a region that would not include the global
  maximum (see \code{\link{CalibrationPlot}}). In particular, the grid
  \code{Lambda} may need to be extended when the maximum stability is
  observed on the left or right edges of the calibration plot.

  To control the expected number of False Positives (Per Family Error Rate)
  in the results, a threshold \code{PFER_thr} can be specified. The
  optimisation problem is then constrained to sets of parameters that
  generate models with an upper-bound in PFER below \code{PFER_thr} (see
  Meinshausen and Bühlmann (2010) and Shah and Samworth (2013)).

  Possible resampling procedures include defining (i) \code{K} subsamples of
  a proportion \code{tau} of the observations, (ii) \code{K} bootstrap samples
  with the full sample size (obtained with replacement), and (iii) \code{K/2}
  splits of the data in half for complementary pair stability selection (see
  arguments \code{resampling} and \code{cpss}). In complementary pair
  stability selection, a feature is considered selected at a given resampling
  iteration if it is selected in the two complementary subsamples.

  For categorical outcomes (argument \code{family} is \code{"binomial"} or
  \code{"multinomial"}), the proportions of observations from each category
  in all subsamples or bootstrap samples are the same as in the full sample.

  To ensure reproducibility of the results, the starting number of the random
  number generator is set to \code{seed}.

  For parallelisation, stability selection with different sets of parameters
  can be run on \code{n_cores} cores. This relies on forking with
  \code{\link[parallel]{mclapply}} (specific to Unix systems).
}
\examples{
\donttest{
oldpar <- par(no.readonly = TRUE)
par(mar = c(12, 5, 1, 1))

## Sparse Principal Component Analysis

# Data simulation
set.seed(1)
simul <- SimulateComponents(pk = c(5, 3, 4))

# sPCA: sparsity on X (unsupervised)
stab <- BiSelection(
  xdata = simul$data,
  ncomp = 3,
  LambdaX = 1:(ncol(simul$data) - 1),
  implementation = SparsePCA
)
print(stab)

# Calibration plot
CalibrationPlot(stab)

# Visualisation of the results
summary(stab)
plot(stab)
SelectedVariables(stab)


## Sparse/Group Partial Least Squares

# Data simulation (continuous outcomes)
set.seed(1)
simul <- SimulateRegression(n = 100, pk = 15, q = 3, family = "gaussian")
x <- simul$xdata
y <- simul$ydata

# sPLS: sparsity on X
stab <- BiSelection(
  xdata = x, ydata = y,
  family = "gaussian", ncomp = 3,
  LambdaX = 1:(ncol(x) - 1),
  implementation = SparsePLS
)
CalibrationPlot(stab)
summary(stab)
plot(stab)

# sPLS: sparsity on both X and Y
stab <- BiSelection(
  xdata = x, ydata = y,
  family = "gaussian", ncomp = 3,
  LambdaX = 1:(ncol(x) - 1),
  LambdaY = 1:(ncol(y) - 1),
  implementation = SparsePLS,
  n_cat = 2
)
CalibrationPlot(stab)
summary(stab)
plot(stab)

# sgPLS: sparsity on X
stab <- BiSelection(
  xdata = x, ydata = y, K = 10,
  group_x = c(2, 8, 5),
  family = "gaussian", ncomp = 3,
  LambdaX = 1:2, AlphaX = seq(0.1, 0.9, by = 0.1),
  implementation = SparseGroupPLS
)
CalibrationPlot(stab)
summary(stab)

# sgPLS: sparsity on both X and Y
stab <- BiSelection(
  xdata = x, ydata = y, K = 10,
  group_x = c(2, 8, 5), group_y = c(1, 2),
  family = "gaussian", ncomp = 3,
  LambdaX = 1:2, AlphaX = seq(0.1, 0.9, by = 0.2),
  LambdaY = 1:2, AlphaY = seq(0.1, 0.9, by = 0.2),
  implementation = SparseGroupPLS,
  n_cat = 2
)
CalibrationPlot(stab)
CalibrationPlot(stab,
  params = c("nx", "alphax", "ny", "alphay")
)
summary(stab)

# gPLS: sparsity on X
stab <- BiSelection(
  xdata = x, ydata = y,
  group_x = c(2, 8, 5),
  family = "gaussian", ncomp = 3,
  LambdaX = 1:2,
  implementation = GroupPLS
)
CalibrationPlot(stab)
summary(stab)

# gPLS: sparsity on both X and Y
stab <- BiSelection(
  xdata = x, ydata = y,
  group_x = c(2, 8, 5), group_y = c(1, 2),
  family = "gaussian", ncomp = 3,
  LambdaX = 1:2, LambdaY = 1:2,
  implementation = GroupPLS
)
CalibrationPlot(stab)
summary(stab)


## Sparse/Group PLS-DA (Discriminant Analysis)

# Data simulation (binary outcomes)
set.seed(1)
simul <- SimulateRegression(n = 200, pk = 15, q = 3, family = "binomial")
x <- simul$xdata
y <- simul$ydata

# sPLS-DA: sparsity on X
stab <- BiSelection(
  xdata = x, ydata = cbind(y),
  family = "binomial", ncomp = 3,
  LambdaX = 1:(ncol(x) - 1),
  implementation = SparsePLS
)
CalibrationPlot(stab)
summary(stab)

# sgPLS-DA: sparsity on X
stab <- BiSelection(
  xdata = x, ydata = cbind(y), K = 10,
  group_x = c(2, 8, 5),
  family = "binomial", ncomp = 3,
  LambdaX = 1:2, AlphaX = seq(0.1, 0.9, by = 0.1),
  implementation = SparseGroupPLS
)
CalibrationPlot(stab)
summary(stab)

# gPLS-DA: sparsity on X
stab <- BiSelection(
  xdata = x, ydata = cbind(y),
  group_x = c(2, 8, 5),
  family = "binomial", ncomp = 3,
  LambdaX = 1:2,
  implementation = GroupPLS
)
CalibrationPlot(stab)
summary(stab)

par(oldpar)
}
}
\references{
\insertRef{ourstabilityselection}{sharp}

  \insertRef{stabilityselectionSS}{sharp}

  \insertRef{stabilityselectionMB}{sharp}

  \insertRef{sparsegroupPLS}{sharp}

  \insertRef{sparsePLS}{sharp}

  \insertRef{sparsePCASVD}{sharp}

  \insertRef{sparsePCA}{sharp}
}
\seealso{
\code{\link{SparsePCA}}, \code{\link{SparsePLS}},
  \code{\link{GroupPLS}}, \code{\link{SparseGroupPLS}},
  \code{\link{VariableSelection}}, \code{\link{Resample}},
  \code{\link{StabilityScore}}

Other stability selection functions: 
\code{\link{GraphicalModel}()},
\code{\link{VariableSelection}()}
}
\concept{stability selection functions}
