% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MutationProfiling.R
\name{slideWindowTune}
\alias{slideWindowTune}
\title{Parameter tuning for sliding window approach}
\usage{
slideWindowTune(
  db,
  sequenceColumn = "sequence_alignment",
  germlineColumn = "germline_alignment_d_mask",
  dbMutList = NULL,
  mutThreshRange,
  windowSizeRange,
  verbose = TRUE
)
}
\arguments{
\item{db}{\code{data.frame} containing sequence data.}

\item{sequenceColumn}{name of the column containing IMGT-gapped sample sequences.}

\item{germlineColumn}{name of the column containing IMGT-gapped germline sequences.}

\item{dbMutList}{if supplied, this should be a list consisting of \code{data.frame}s 
returned as \code{$pos} in the nested list produced by 
\link{calcObservedMutations} with \code{returnRaw=TRUE}; otherwise, 
\link{calcObservedMutations} is called on columns \code{sequenceColumn}
and \code{germlineColumn} of \code{db}. Default is \code{NULL}.}

\item{mutThreshRange}{range of threshold on the number of mutations in \code{windowSize} 
consecutive nucleotides to try. Must be between 1 and 
maximum \code{windowSizeRange} inclusive.}

\item{windowSizeRange}{range of length of consecutive nucleotides to try. The lower end
must be at least 2.}

\item{verbose}{whether to print out messages indicating current progress. Default
is \code{TRUE}.}
}
\value{
a list of logical matrices. Each matrix corresponds to a \code{windowSize} in 
          \code{windowSizeRange}. Each column in a matrix corresponds to a \code{mutThresh} in
          \code{mutThreshRange}.
}
\description{
Apply \link{slideWindowDb} over a search grid made of combinations of \code{mutThresh} and 
\code{windowSize} to help with picking a pair of values for these parameters. Parameter 
tuning can be performed by choosing a combination that gives a reasonable number of 
filtered/remaining sequences.
}
\details{
If, in a given combination of \code{mutThresh} and \code{windowSize}, \code{mutThresh} 
          is greater than \code{windowSize}, \code{NA}s will be returned for that particular
          combination. A message indicating that the combination has been "skipped" will be 
          printed if \code{verbose=TRUE}.
          
          If \link{calcObservedMutations} was previously run on \code{db} and saved, supplying
          \code{$pos} from the saved result as \code{dbMutList} could save time by skipping a
          second call of \link{calcObservedMutations}. This could be helpful especially when 
          \code{db} is large.
}
\examples{
# Load and subset example data
data(ExampleDb, package="alakazam")
db <- ExampleDb[1:5, ]

# Try out thresholds of 2-4 mutations in window sizes of 7-9 nucleotides. 
# In this case, all combinations are legal.
slideWindowTune(db, mutThreshRange=2:4, windowSizeRange=7:9)

# Illegal combinations are skipped, returning NAs.
slideWindowTune(db, mutThreshRange=2:4, windowSizeRange=2:4, 
                verbose=FALSE)
                                                            
# Run calcObservedMutations separately
exDbMutList <- sapply(1:5, function(i) {
    calcObservedMutations(inputSeq=db[["sequence_alignment"]][i],
                          germlineSeq=db[["germline_alignment_d_mask"]][i],
                          returnRaw=TRUE)$pos })
slideWindowTune(db, dbMutList=exDbMutList, 
                mutThreshRange=2:4, windowSizeRange=2:4)
                                                           
}
\seealso{
\link{slideWindowDb} is called on \code{db} for tuning. See \link{slideWindowTunePlot} 
          for visualization. See \link{calcObservedMutations} for generating \code{dbMutList}.
}
