\name{node_mixture}
\alias{node_mixture}

\title{
Generate Data from a Mixture of Node Definitions
}
\description{
This node type allows users to apply different nodes to different subsets of the already generated data, making it possible to generate data for arbitrary mixture distributions. It is similar to \code{\link{node_conditional_distr}} and \code{\link{node_conditional_prob}}, with the main difference being that the former only allow univariate distributions conditional on categorical variables, while this function allows any kind of node definition and condition. This makes it, for example, possible to generate data for a variable from different regression models for different subsets of simulated individuals.
}
\usage{
node_mixture(data, parents, name, distr, default=NA)
}
\arguments{
  \item{data}{
A \code{data.table} (or something that can be coerced to a \code{data.table}) containing all columns specified by \code{parents}.
  }
  \item{parents}{
A character vector specifying the names of the parents that this particular child node has. This vector should include all nodes that are used in the conditions and the \code{\link{node}} calls specified in \code{distr}.
  }
  \item{name}{
A single character string specifying the name of the node.
  }
  \item{distr}{
A unnamed list that specifies both the conditions and the \code{\link{node}} definitions. It should be specified in a similar way as the \code{\link[data.table]{fcase}} function in pairs of conditions (coded as strings) and \code{\link{node}} definitions. This means that a condition comes first, for example \code{"A==0"}, followed by some call \code{\link{node}} and so on. Arbitrary numbers of those pairs are allowed with no restrictions to what can be specified in the \code{\link{node}} calls. The \code{name} argument has to be specified in all \code{\link{node}} calls, but it does not matter which value is used as it will be ignored in further processing. Currently only supports time-fixed nodes defined using the \code{\link{node}} function, not time-dependent nodes defined using the \code{\link{node_td}} function. See examples.
  }
  \item{default}{
A single value of some kind, used as a default value for those individuals not covered by all the conditions defined in \code{distr}. Defaults to \code{NA}.
  }
}
\author{
Robin Denz
}
\details{
Internally, the data is generated by extracting only the relevant part of the already generated \code{data} as defined by the condition and using \code{\link{node}} function to generate the new response-part. This generation is done in the order in which the \code{distr} was specified, meaning that data for the first condition is checked first and so on. There are no safeguards to guarantee that the conditions do not overlap. For example, users are free to set the first condition to something like \code{A > 10} and the next one to \code{A > 11}, in which case the value for every individual with \code{A > 11} is generated twice (first with the first specification, secondly with the next specification). In this case, only the last generated value is retained.

Note that it is also possible to use the mixture node itself inside the conditions or \code{\link{node}} calls in \code{distr}, because it is directly added to the \code{data} before the first condition is applied (by setting everyone to the \code{default} value). See examples.

Additionally, because the output of each of the parts of the mixture distributions is forced into one vector, they might be coerced from one class to another, depending on the input to \code{distr} and the order used. This also needs to be taken care of by the user.
}
\value{
Returns a vector of length \code{nrow(data)}. The class of the vector is determined by what is specified in \code{distr}.
}
\seealso{
\code{\link{empty_dag}}, \code{\link{node}}, \code{\link{node_td}}, \code{\link{sim_from_dag}}, \code{\link{sim_discrete_time}}
}
\examples{
library(simDAG)

set.seed(1234)

## different linear regression models per level of a different covariate
# here, A is the group that is used for the conditioning, B is a predictor
# and Y is the mixture distributed outcome
dag <- empty_dag() +
  node("A", type="rbernoulli") +
  node("B", type="rnorm") +
  node("Y", type="mixture", parents="A",
       distr=list(
         "A==0", node(".", type="gaussian", formula= ~ -2 + B*2, error=1),
         "A==1", node(".", type="gaussian", formula= ~ 3 + B*5, error=1)
       ))
data <- sim_from_dag(dag, n_sim=100)
head(data)

# also works with multiple conditions
dag <- empty_dag() +
  node(c("A", "C"), type="rbernoulli") +
  node("B", type="rnorm") +
  node("Y", type="mixture", parents=c("A", "C"),
    distr=list(
      "A==0 & C==1", node(".", type="gaussian", formula= ~ -2 + B*2, error=1),
      "A==1", node(".", type="gaussian", formula= ~ 3 + B*5, error=1)
    ))
data <- sim_from_dag(dag, n_sim=100)
head(data)

# using the mixture node itself in the condition
# see cookbook vignette, section on outliers for more info
dag <- empty_dag() +
  node(c("A", "B", "C"), type="rnorm") +
  node("Y", type="mixture", parents=c("A", "B", "C"),
       distr=list(
         "TRUE", node(".", type="gaussian", formula= ~ -2 + A*0.1 + B*1 + C*-2,
                      error=1),
         "Y > 2", node(".", type="rnorm", mean=10000, sd=500)
       ))
data <- sim_from_dag(dag, n_sim=100)
}
