#' Create and modify variable sets for batch simulation
#'
#' A variable set contains the descriptions of the relevant variables in a
#' batch simulation. Use \code{new_var_set} to create an S3 \code{var_set}
#' object, and use \code{add_var} to add descriptions of variables.
#'
#' @describeIn new_var_set Create a \code{var_set}.
#'
#' @param var_set A \code{var_set} object.
#' @param par_name,var_name The name of the parameter and variable
#' in the simulation function
#' @param start,end,by The data points where you want to test the variables.
#' Passed to \code{seq}.
#'
#' @examples
#' test <- new_var_set()
#' test <- test %>%
#'   add_var("par1", "var1", 1, 2, 0.1) %>%
#'   add_var("par2", "var2", 1, 2, 0.1)
#' @seealso \code{\link{make_var_grid}} for making grids from variable sets;
#' \code{\link{batch_simulation}} for running batch simulation and a
#' concrete example.
#'
#' @export
new_var_set <- function() {
  result <- list()
  class(result) <- c("var_set", "list")
  attr(result, "nvar") <- 0
  attr(result, "npar") <- 0
  return(result)
}

#' @describeIn new_var_set Add a variable to the \code{var_set}.
#' @export
add_var <- function(var_set, par_name, var_name, start, end, by) {
  var_set[[par_name]][[var_name]] <- tibble::lst(start, end, by) # <U+8FD9><U+91CC>var_name<U+5E94><U+8BE5><U+76F4><U+63A5><U+53D8><U+6210>name<U+5427><U+FF1F>
  attr(var_set, "nvar") <- attr(var_set, "nvar") + 1
  attr(var_set, "npar") <- length(var_set)
  return(var_set)
}

#' @describeIn new_var_set The number of variables.
#' @export
nvar <- function(var_set) attr(var_set, "nvar")

#' @describeIn new_var_set The number of parameters.
#' @export
npar <- function(var_set) attr(var_set, "npar")

#' Print a var_set
#'
#' @param x The object.
#' @param detail Do you want to print the object details as a full list?
#' @param ... Not in use.
#' @method print var_set
#' @export
print.var_set <- function(x, detail = FALSE, ...) {
  if (detail) {
    print.default(x)
  } else {
    cat(
      sprintf("A `var_set` with %d parameter(s) and %d variable(s)", npar(x), nvar(x))
    )
  }
}

#' Fill a vector of values into a structure list.
#'
#' @param vec A vector of values.
#' @param struct A list with a certain structure.
#'
#' @export
fill_in_struct <- function(vec, struct) {
  if (!("var_set" %in% class(struct))) stop("Wrong input class. `struct` should be a `var_set`.")
  vec_index <- 1
  for (i in 1:length(struct)) {
    for (j in 1:length(struct[[i]])) {
      struct[[i]][[j]] <- vec[vec_index]
      vec_index <- vec_index + 1
    }
  }
  class(struct) <- c("var_list", "list")
  return(struct)
}

#' Make variable grids for batch simulation
#'
#' This is the main function for making the variable grids.
#'
#' @param var_set A \code{var_set} object. See \code{\link{new_var_set}}
#' and \code{\link{add_var}}.
#'
#' @return A \code{var_grid} object.
#'
#' @seealso \code{\link{batch_simulation}} for a concrete example.
#'
#' @export
make_var_grid <- function(var_set) {
  var_set_seq <- list()
  var_set_par <- list()
  for (i in names(var_set)) {
    for (j in names(var_set[[i]])) {
      var_set_seq[[j]] <- seq(var_set[[i]][[j]]$start, var_set[[i]][[j]]$end, var_set[[i]][[j]]$by)
      var_set_par[[j]] <- names(var_set)[i]
    }
  }
  var_grid_num <- expand.grid(var_set_seq)

  var_grid_list <- data.frame(var_list = rep(NA, nrow(var_grid_num)))

  var_grid_list$var_list <- apply(var_grid_num, 1, fill_in_struct, var_set)

  var_grid <- cbind(var_grid_list, as.data.frame(var_grid_num))

  result <- var_grid
  class(result) <- c("var_grid", "data.frame")
  attr(result, "var_set_seq") <- var_set_seq
  attr(result, "var_set_par") <- var_set_par
  attr(result, "nvar") <- nvar(var_set)
  attr(result, "npar") <- npar(var_set)
  return(result)
}

#' Print a var_grid
#'
#' @inheritParams print.var_set
#' @method print var_grid
#' @export
print.var_grid <- function(x, detail = FALSE, ...) {
  if (detail) print.default(x)
  cat(
    sprintf(
      "A `var_grid` with %d parameter(s), %d variable(s), and %d condition(s)",
      npar(x), nvar(x), length(x$var_list)
    )
  )
}

#' @describeIn batch_simulation Modify a single simulation.
#' @param var_list An \code{var_list} object generated by \code{\link{fill_in_struct}}.
#' @export
modified_simulation <- function(sim_fun, var_list, default_list, bigmemory = TRUE) {
  sim_fun_list <- default_list
  for (i in names(var_list)) {
    for (j in names(var_list[[i]])) {
      sim_fun_list[[i]][[j]] <- var_list[[i]][[j]]
    }
  }
  result <- do.call(sim_fun, sim_fun_list)

  if (bigmemory & is.matrix(result)) result <- as.hash_big.matrix(result)
  return(result)
}


#' A simple simulation function for testing
#'
#' @param par1,par2 Two parameters. \code{par1} contains \code{var1};
#' \code{par2} contains \code{var2} and \code{var3}.
#' @param length The length of simulation.
#'
#' @return A matrix of simulation results.
#'
#' @seealso \code{\link{batch_simulation}} for a concrete example.
#'
#' @export
#'
sim_fun_test <- function(par1, par2, length = 1000) {
  output <- matrix(nrow = length, ncol = 3)
  colnames(output) <- c("out1", "out2", "out3")
  output[1, ] <- c(par1$var1, par2$var2, rnorm(1, sd = 0.01))
  for (i in 2:length) {
    output[i, 1] <- 0.5 * output[i - 1, 1] + output[i - 1, 2] + par2$var3 + par1$var1 * par2$var2
    output[i, 2] <- -0.5 * output[i - 1, 1] + output[i - 1, 2] + par2$var3
    output[i, 3] <- output[i - 1, 3] + rnorm(1, sd = 0.01)
  }
  return(output)
}

#' Do the batch simulation
#'
#' This is the main function for the batch simulation.
#'
#' @param var_grid A \code{var_grid} object. See \code{\link{make_var_grid}}.
#' @param sim_fun The simulation function. See \code{\link{sim_fun_test}}
#' for an example.
#' @param default_list A list of default values for \code{sim_fun}.
#' @param bigmemory Use \code{\link{hash_big.matrix-class}} to store large matrices?
#'
#' @return A \code{batch_simulation} object, also a data frame.
#' The first column, \code{var}, is a list of
#' \code{var_list} that contains all the variables; the second to the second
#' last columns are the values of the variables; the last column is the
#' output of the simulation function.
#'
#' @examples
#' test <- new_var_set()
#' test <- test %>%
#'   add_var("par1", "var1", 1, 2, 0.1) %>%
#'   add_var("par2", "var2", 1, 2, 0.1)
#' test_grid <- make_var_grid(test)
#' test_result <- batch_simulation(test_grid, sim_fun_test,
#'   default_list = list(
#'     par1 = list(var1 = 0),
#'     par2 = list(var2 = 0, var3 = 0)
#'   ), bigmemory = FALSE
#' )
#' test_result
#' @export
batch_simulation <- function(var_grid, sim_fun, default_list, bigmemory = TRUE) {
  result <- var_grid %>%
    dplyr::mutate(output = purrr::map(var_list, function(x) modified_simulation(sim_fun, x, default_list, bigmemory)))
  class(result) <- c("batch_simulation", "data.frame")
  attr(result, "sim_fun") <- sim_fun
  return(result)
}

#' Print a batch_simulation
#' @inheritParams print.var_set
#' @method print batch_simulation
#' @export
print.batch_simulation <- function(x, detail = FALSE, ...) {
  if (detail) {
    print.default(x)
  } else {
    cat(
      sprintf("Output(s) from %d simulations.", nrow(x))
    )
  }
}

#' Attach all matrices in a batch simulation
#'
#' @param bs A \code{\link{batch_simulation}} object.
#' @param backingpath Passed to \code{\link[bigmemory]{as.big.matrix}}.
#' @export
attach_all_matrices <- function(bs, backingpath = "bp") {
  if (!"batch_simulation" %in% class(bs)) stop("Wrong input class. bs should be a `batch_simulation`.")
  bs <- bs %>%
    dplyr::mutate(output = purrr::map(output, attach.hash_big.matrix, backingpath = backingpath))
  return(bs)
}
