% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/sjPlotPCA.R
\name{sjp.pca}
\alias{sjp.pca}
\title{Plot PCA results}
\usage{
sjp.pca(data, numberOfFactors = NULL, factorLoadingTolerance = 0.1,
  plotEigenvalues = FALSE, digits = 2, title = NULL,
  axisLabels.y = NULL, type = "b", geom.size = 0.6,
  geom.colors = "RdBu", breakTitleAt = 50, breakLabelsAt = 20,
  showValueLabels = TRUE, showCronbachsAlpha = TRUE, printPlot = TRUE)
}
\arguments{
\item{data}{A data frame with factors (each columns one variable) that should be used
to compute a PCA, or a \code{\link{prcomp}} object.}

\item{numberOfFactors}{A predefined number of factors to use for the calculating the varimax
rotation. By default, this value is \code{NULL} and the amount of factors is
calculated according to the Kaiser-criteria. See paramater \code{plotEigenvalues}.}

\item{factorLoadingTolerance}{Specifies the minimum difference a variable needs to have between
factor loadings (components) in order to indicate a clear loading on just one factor and not
diffusing over all factors. For instance, a variable with 0.8, 0.82 and 0.84 factor loading
on 3 possible factors can not be clearly assigned to just one factor and thus would be removed
from the principal component analysis. By default, the minimum difference of loading values
between the highest and 2nd highest factor should be 0.1}

\item{plotEigenvalues}{If \code{TRUE}, a plot showing the Eigenvalues according to the
Kaiser criteria is plotted to determine the number of factors.}

\item{digits}{The amount of decimals used. Default is 2.}

\item{title}{Title of the diagram, plotted above the whole diagram panel.}

\item{axisLabels.y}{The item labels that are printed on the y-axis. If no item labels are
provided (default), the data frame's column names are used. Item labels must
be a string vector, e.g.: \code{axisLabels.y = c("Var 1", "Var 2", "Var 3")}.}

\item{type}{Plot type resp. geom type. May be one of following: \code{"circle"} or \code{"tile"}
circular or tiled geoms, or \code{"bar"} for a bar plot. You may use initial letter only
for this parameter.}

\item{geom.size}{Specifies the circle size factor. The circle size depends on the correlation
value multiplicated with this factor. Default is 10.}

\item{geom.colors}{A color palette for fillng the geoms. If not specified, the diverging \code{"RdBl"} color palette
from the color brewer palettes is used, resulting in red colors for negative and blue colors
for positive factor loadings, that become lighter the weaker the loadings are. Use any
color palette that is suitbale for the \code{scale_fill_gradientn} parameter of ggplot2.}

\item{breakTitleAt}{Wordwrap for diagram title. Determines how many chars of the title are displayed in
one line and when a line break is inserted into the title. Default is 50.}

\item{breakLabelsAt}{Wordwrap for diagram labels. Determines how many chars of the category labels are displayed in
one line and when a line break is inserted. Default is 12.}

\item{showValueLabels}{Whether factor loading values should be plotted to each geom.
Default is \code{TRUE}.}

\item{showCronbachsAlpha}{If \code{TRUE} (default), the cronbach's alpha value for each factor scale will be calculated,
i.e. all variables with the highest loading for a factor are taken for the
reliability test. The result is an alpha value for each factor dimension.
Only applies when \code{data} is a data frame and no \code{\link{prcomp}} object.}

\item{printPlot}{If \code{TRUE} (default), plots the results as graph. Use \code{FALSE} if you don't
want to plot any graphs. In either case, the ggplot-object will be returned as value.}
}
\value{
(Invisibly) returns a \code{\link{structure}} with
         \itemize{
           \item the varimax-rotated factor loading matrix (\code{varim})
           \item the column indices of removed variables (for more details see next list item) (\code{removed.colindex})
           \item an updated data frame containing all factors that have a clear loading on a specific scale in case \code{data} was a data frame (See parameter \code{factorLoadingTolerance} for more details) (\code{removed.df})
           \item the ggplot-object (\code{plot}),
           \item the data frame that was used for setting up the ggplot-object (\code{df}).
           }
}
\description{
Performes a principle component analysis on a data frame or matrix (with
               varimax rotation) and plots the factor solution as ellipses or tiles. \cr \cr
               In case a data frame is used as parameter, the cronbach's alpha value for
               each factor scale will be calculated, i.e. all variables with the highest
               loading for a factor are taken for the reliability test. The result is
               an alpha value for each factor dimension.
}
\note{
This PCA uses the \code{\link{prcomp}} function and the \code{\link{varimax}} rotation.
}
\examples{
# randomly create data frame with 7 items, each consisting of 4 categories
likert_4 <- data.frame(sample(1:4,
                              500,
                              replace = TRUE,
                              prob = c(0.2, 0.3, 0.1, 0.4)),
                       sample(1:4,
                              500,
                              replace = TRUE,
                              prob = c(0.5, 0.25, 0.15, 0.1)),
                       sample(1:4,
                              500,
                              replace = TRUE,
                              prob = c(0.4, 0.15, 0.25, 0.2)),
                       sample(1:4,
                              500,
                              replace = TRUE,
                              prob = c(0.25, 0.1, 0.4, 0.25)),
                       sample(1:4,
                              500,
                              replace = TRUE,
                              prob = c(0.1, 0.4, 0.4, 0.1)),
                       sample(1:4,
                              500,
                              replace = TRUE),
                       sample(1:4,
                              500,
                              replace = TRUE,
                              prob = c(0.35, 0.25, 0.15, 0.25)))

# Create variable labels
colnames(likert_4) <- c("V1", "V2", "V3", "V4", "V5", "V6", "V7")

# plot results from PCA as square-tiled "heatmap"
sjp.pca(likert_4, type = "tile")

# plot results from PCA as bars
sjp.pca(likert_4, type = "bar")

# manually compute PCA
pca <- prcomp(na.omit(likert_4),
              retx = TRUE,
              center = TRUE,
              scale. = TRUE)
# plot results from PCA as circles, including Eigenvalue-diagnostic.
# note that this plot does not compute the Cronbach's Alpha
sjp.pca(pca,
        plotEigenvalues = TRUE,
        type = "circle",
        geom.size = 10)


# -------------------------------
# Data from the EUROFAMCARE sample dataset
# -------------------------------
library(sjmisc)
data(efc)

# retrieve variable and value labels
varlabs <- get_var_labels(efc)

# recveive first item of COPE-index scale
start <- which(colnames(efc) == "c82cop1")
# recveive last item of COPE-index scale
end <- which(colnames(efc) == "c90cop9")

# create data frame with COPE-index scale
mydf <- as.data.frame(efc[, c(start:end)])
colnames(mydf) <- varlabs[c(start:end)]

sjp.pca(mydf)
sjp.pca(mydf, type = "tile")

# -------------------------------
# auto-detection of labels
# -------------------------------
efc <- set_var_labels(efc, varlabs)
sjp.pca(efc[, c(start:end)], type = "circle", geom.size = 10)
}
\seealso{
\itemize{
           \item \href{http://www.strengejacke.de/sjPlot/sjp.pca/}{sjPlot manual: sjp.pca}
           \item \code{\link{sjt.pca}}
           }
}

