% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vtl.R
\name{estimateVTL}
\alias{estimateVTL}
\title{Estimate vocal tract length}
\usage{
estimateVTL(
  formants,
  method = c("regression", "meanDispersion", "meanFormant")[1],
  interceptZero = TRUE,
  speedSound = 35400,
  checkFormat = TRUE,
  output = c("simple", "detailed")[1],
  plot = FALSE
)
}
\arguments{
\item{formants}{formant frequencies in any format recognized by
\code{\link{soundgen}}: a character string like \code{aaui} referring to
default presets for speaker "M1"; a vector of formant frequencies like
\code{c(550, 1600, 3200)}; or a list with multiple values per formant like
\code{list(f1 = c(500, 550), f2 = 1200))}}

\item{method}{the method of estimating vocal tract length (see details)}

\item{interceptZero}{if TRUE, forces the regression curve to pass through the
origin, which corresponds to a closed-open tube (method = "regression" only)}

\item{speedSound}{speed of sound in warm air, by default 35400 cm/s. Stevens
(2000) "Acoustic phonetics", p. 138}

\item{checkFormat}{if FALSE, only a list of properly formatted formant
frequencies is accepted}

\item{output}{"simple" (default) = just the VTL; "detailed" = a list of
additional stats (see Value below)}

\item{plot}{if TRUE, plots the regression line whose slope gives formant
dispersion (method = "regression" only) and Label size shows the influence
of each observation; the second plot shows how VTL varies depending on the
number of formants used}
}
\value{
If \code{output = 'simple'} (default), returns the estimated vocal
  tract length in cm. If \code{output = 'detailed'} and \code{method =
  'regression'}, returns a list with extra stats used for plotting. Namely,
  \code{$regressionInfo$infl} gives the influence of each observation
  calculated as the absolute change in VTL with vs without the observation *
  10 + 1 (the size of labels on the first plot). \code{$vtlPerFormant$vtl}
  gives the VTL as it would be estimated if only the first \code{nFormants}
  were used.
}
\description{
Estimates the length of vocal tract based on formant frequencies, assuming
that the vocal tract can be modeled as a tube open at one end.
}
\details{
If \code{method = 'meanFormant'}, vocal tract length (VTL) is calculated
separately for each formant as \eqn{(2 * formant_number - 1) * speedSound /
(4 * formant_frequency)}, and then the resulting VTLs are averaged. If
\code{method = 'meanDispersion'}, formant dispersion is calculated as the
mean distance between formants, and then VTL is calculated as \eqn{speed of
sound / 2 / formant dispersion}. If \code{method = 'regression'}, formant
dispersion is estimated using the regression method described in Reby et al.
(2005) "Red deer stags use formants as assessment cues during intrasexual
agonistic interactions". For a review of these and other VTL-related summary
measures of formant frequencies, refer to Pisanski et al. (2014) "Vocal
indicators of body size in men and women: a meta-analysis". See also
\code{\link{schwa}} for VTL estimation with additional information on formant
frequencies.
}
\examples{
estimateVTL(NA)
estimateVTL(500)
estimateVTL(c(600, 1850, 2800, 3600, 5000), plot = TRUE)
estimateVTL(c(600, 1850, 2800, 3600, 5000), plot = TRUE, output = 'detailed')

# Multiple measurements are OK
estimateVTL(
  formants = list(f1 = c(540, 600, 550),
  f2 = 1650, f3 = c(2400, 2550)),
  plot = TRUE, output = 'detailed')
# NB: this is better than averaging formant values. Cf.:
estimateVTL(
  formants = list(f1 = mean(c(540, 600, 550)),
  f2 = 1650, f3 = mean(c(2400, 2550))),
  plot = TRUE)

# Missing values are OK
estimateVTL(c(600, 1850, 3100, NA, 5000), plot = TRUE)
estimateVTL(list(f1 = 500, f2 = c(1650, NA, 1400), f3 = 2700), plot = TRUE)

# Normally it makes sense to assume that the vocal tract is a closed-open
# tube, forcing the intercept to be zero. To relax this assumption:
estimateVTL(c(600, 1850, 3100, NA, 5000), interceptZero = FALSE, plot = TRUE)

# Note that VTL estimates based on the commonly reported 'meanDispersion'
# depend only on the first and last formant
estimateVTL(c(500, 1400, 2800, 4100), method = 'meanDispersion')
estimateVTL(c(500, 1100, 2300, 4100), method = 'meanDispersion') # identical
# ...but this is not the case for 'meanFormant' and 'regression' methods
estimateVTL(c(500, 1400, 2800, 4100), method = 'meanFormant')
estimateVTL(c(500, 1100, 2300, 4100), method = 'meanFormant') # much longer

\dontrun{
# Compare the results produced by the three methods
nIter = 1000
out = data.frame(meanFormant = rep(NA, nIter), meanDispersion = NA, regression = NA)
for (i in 1:nIter) {
  # generate a random formant configuration
  f = runif(1, 300, 900) + (1:6) * rnorm(6, 1000, 200)
  out$meanFormant[i]    = estimateVTL(f, method = 'meanFormant')
  out$meanDispersion[i] = estimateVTL(f, method = 'meanDispersion')
  out$regression[i]     = estimateVTL(f, method = 'regression')
}
pairs(out)
cor(out)
# 'meanDispersion' is pretty different, while 'meanFormant' and 'regression'
# give broadly comparable results
}
}
\seealso{
\code{\link{schwa}}
}
