## The MASS::negative.binomial code evaluates .THeta <- theta, making it unsuitable for a call without the shape param
negbin <- function (shape = stop("negbin's 'shape' must be specified"), link = "log") {
  mc <- match.call()
  linktemp <- substitute(link)
  if (!is.character(linktemp)) 
    linktemp <- deparse(linktemp)
  if (linktemp %in% c("log", "identity", "sqrt")) 
    stats <- make.link(linktemp)
  else if (is.character(link)) {
    stats <- make.link(link)
    linktemp <- link
  }
  else {
    if (inherits(link, "link-glm")) {
      stats <- link
      if (!is.null(stats$name)) 
        linktemp <- stats$name
    }
    else stop(gettextf("\"%s\" link not available for negative binomial family; available links are \"identity\", \"log\" and \"sqrt\"", 
                       linktemp))
  }
  variance <- function(mu) mu + mu^2/shape
  validmu <- function(mu) all(mu > 0)
  dev.resids <- function(y, mu, wt) 2 * wt * (y * log(pmax(1, y)/mu) - (y + shape) * log((y + shape)/(mu + shape)))
  aic <- function(y, n, mu, wt, dev) {
    term <- (y + shape) * log(mu + shape) - y * log(mu) + 
      lgamma(y + 1) - shape * log(shape) + lgamma(shape) - 
      lgamma(shape + y)
    2 * sum(term * wt)
  }
  initialize <- expression({
    if (any(y < 0)) stop("negative values not allowed for the negative binomial family")
    n <- rep(1, nobs)
    mustart <- y + (y == 0)/6
  })
  simfun <- function(object, nsim) {
    wts <- object$prior.weights
    if (any(wts != 1)) 
      warning("ignoring prior weights")
    ftd <- fitted(object)
    resu <- MASS::rnegbin(nsim * length(ftd), ftd, shape)
    if (nsim>1L) resu <- matrix(resu,ncol=nsim)
    resu
  }
  ## all closures defined here have parent.env the environment(spaMM_Gamma) ie <environment: namespace:spaMM>
  ## changes the parent.env of all these functions (aic, dev.resids, simfun, validmu, variance): 
  parent.env(environment(aic)) <- environment(stats::binomial) ## parent = <environment: namespace:stats>
  structure(list(family = structure("negbin",
                                    withArgs=quote(paste("Neg.binomial(shape=",signif(shape,4),")",sep=""))), 
                 link = linktemp, linkfun = stats$linkfun, 
                 linkinv = stats$linkinv, variance = variance, dev.resids = dev.resids, 
                 aic = aic, mu.eta = stats$mu.eta, initialize = initialize, 
                 validmu = validmu, valideta = stats$valideta, simulate = simfun), 
            class = "family")
}