% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/constrLaplacianRank.R
\name{cluster_k_component_graph}
\alias{cluster_k_component_graph}
\title{Cluster a k-component graph from data using the Constrained Laplacian Rank algorithm

Cluster a k-component graph on the basis of an observed data matrix.
Check out https://mirca.github.io/spectralGraphTopology for code examples.}
\usage{
cluster_k_component_graph(Y, k = 1, m = 5, lmd = 1, eigtol = 1e-09,
  edgetol = 1e-06, maxiter = 1000)
}
\arguments{
\item{Y}{a pxn data matrix, where p is the number of nodes and n is the number of
features (or data points per node)}

\item{k}{the number of components of the graph}

\item{m}{the maximum number of possible connections for a given node used
to build an affinity matrix}

\item{lmd}{L2-norm regularization hyperparameter}

\item{eigtol}{value below which eigenvalues are considered to be zero}

\item{edgetol}{value below which edge weights are considered to be zero}

\item{maxiter}{the maximum number of iterations}
}
\value{
A list containing the following elements:
\item{\code{Laplacian}}{the estimated Laplacian Matrix}
\item{\code{Adjacency}}{the estimated Adjacency Matrix}
\item{\code{eigvals}}{the eigenvalues of the Laplacian Matrix}
\item{\code{lmd_seq}}{sequence of lmd values at every iteration}
\item{\code{elapsed_time}}{elapsed time at every iteration}
}
\description{
Cluster a k-component graph from data using the Constrained Laplacian Rank algorithm

Cluster a k-component graph on the basis of an observed data matrix.
Check out https://mirca.github.io/spectralGraphTopology for code examples.
}
\examples{
library(clusterSim)
library(spectralGraphTopology)
library(igraph)
set.seed(1)
# number of nodes per cluster
N <- 30
# generate datapoints
twomoon <- shapes.two.moon(N)
# estimate underlying graph
graph <- cluster_k_component_graph(twomoon$data, k = 2)
# build network
net <- graph_from_adjacency_matrix(graph$Adjacency, mode = "undirected", weighted = TRUE)
# colorify nodes and edges
colors <- c("#706FD3", "#FF5252", "#33D9B2")
V(net)$cluster <- twomoon$clusters
E(net)$color <- apply(as.data.frame(get.edgelist(net)), 1,
                      function(x) ifelse(V(net)$cluster[x[1]] == V(net)$cluster[x[2]],
                                        colors[V(net)$cluster[x[1]]], '#000000'))
V(net)$color <- c(colors[1], colors[2])[twomoon$clusters]
# plot network
plot(net, layout = twomoon$data, vertex.label = NA, vertex.size = 3)
}
\references{
Nie, Feiping and Wang, Xiaoqian and Jordan, Michael I. and Huang, Heng.
            The Constrained Laplacian Rank Algorithm for Graph-based Clustering, 2016,
            AAAI'16. http://dl.acm.org/citation.cfm?id=3016100.3016174
}
\author{
Ze Vinicius and Daniel Palomar
}
