% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\docType{data}
\name{comets}
\alias{comets}
\title{Comet orbits}
\format{
A data frame with 3633 rows and 8 variables:
\describe{
  \item{id}{database ID.}
  \item{full_name}{full name/designation following the
  \href{https://www.iau.org/public/themes/naming/#comets}{
  IUA naming convention}.}
  \item{i}{inclination; the orbit angle with respect to the ecliptic plane,
  in radians in \eqn{[0, \pi]}.}
  \item{om}{longitude of the ascending node; the angle between the normal
  vector of the orbit and the normal vector of the ecliptic plane, in
  radians in \eqn{[0, 2\pi)}.}
  \item{per_y}{sidereal orbital period (in years).}
  \item{class}{orbit classification. A factor with levels given below.}
  \item{diameter}{diameter from equivalent sphere (in km).}
  \item{ccf2009}{flag indicating if the comet was considered in the data
  application in Cuesta-Albertos et al. (2009); see details below.}
}
}
\source{
\url{https://ssd.jpl.nasa.gov/sbdb_query.cgi}
}
\usage{
comets
}
\description{
Comet orbits data from the
\href{https://ssd.jpl.nasa.gov/sbdb_query.cgi}{
JPL Small-Body Database Search Engine}. The normal vector of a comet orbit
represents is a vector on \eqn{S^2}.
}
\details{
The normal vector to the ecliptic plane of the comet with inclination
\eqn{i} and longitude of the ascending node \eqn{\omega} is
\deqn{(\sin(i) \sin(\omega), -\sin(i) \cos(\omega), \cos(i))'.}{
(sin(i) sin(\omega), -sin(i) cos(\omega), cos(i))'.}

A prograde comet has positive \eqn{\cos(i)}{cos(i)}, negative
\eqn{\cos(i)}{cos(i)} represents a retrograde comet.

\code{class} has the following levels:
\itemize{
 \item \code{COM}: comet orbit not matching any defined orbit class.
 \item \code{CTc}: Chiron-type comet, as defined by Levison and Duncan
 (T_Jupiter > 3; a > a_Jupiter).
 \item \code{ETc}: Encke-type comet, as defined by Levison and Duncan
 (T_Jupiter > 3; a < a_Jupiter).
 \item \code{HTC}: Halley-type comet, classical definition (20y < P < 200y).
 \item \code{HYP}: comets on hyperbolic orbits.
 \item \code{JFc}: Jupiter-family comet, as defined by Levison and Duncan
 (2 < T_Jupiter < 3).
 \item \code{JFC}: Jupiter-family comet, classical definition (P < 20y).
 \item \code{PAR}: comets on parabolic orbits.
}
Hyperbolic and parabolic comets are not periodic; only elliptical comets
are periodic.

The \code{ccf2009} variable gives the observations considered in
Cuesta-Albertos et al. (2009) after fetching in the database in 2007-12-14
for the comets such that
\code{!(class \%in\% c("HYP", "PAR")) & per_y >= 200 & !numbered}. A
periodic comet is \code{numbered} by the IUA only after its second
perihelion passage, and then its \code{id} starts with \code{c}. Due to the
dynamic nature of the data, more comets were added to the database since
2007 and also some observations were updated.

The script performing the data preprocessing is available at
\href{https://github.com/egarpor/sphunif/blob/master/data-raw/comets.R}{
\code{comets.R}}. The data was retrieved on 2020-05-07.
}
\examples{
# Load data
data("comets")

# Add normal vectors
comets$normal <- cbind(sin(comets$i) * sin(comets$om),
                       -sin(comets$i) * cos(comets$om),
                       cos(comets$i))

# Add numbered information
comets$numbered <- substr(comets$id, 1, 1) == "c"

# Tests to be performed
type_tests <- c("PCvM", "PAD", "PRt")

# Excluding the C/1882 R1-X (Great September comet) records with X = B, C, D
comets_ccf2009 <- comets[comets$ccf2009, ][-c(13:15), ]

# Sample size
nrow(comets_ccf2009)

# Tests for the data in Cuesta-Albertos et al. (2009)
tests_ccf2009 <- unif_test(data = comets_ccf2009$normal, type = type_tests,
                           p_value = "asymp")
tests_ccf2009
}
\references{
Cuesta-Albertos, J. A., Cuevas, A., Fraiman, R. (2009) On projection-based
tests for directional and compositional data. \emph{Statistics and
Computing}, 19:367--380. \doi{10.1007/s11222-008-9098-3}
}
\keyword{datasets}
