\name{sim.sample}
\alias{sim.sample}
\title{Simulated survival data generator}
\description{Generates a simulated sample of clustered survival data, allowing
for flexible baseline hazard and frailty distributions.}
\usage{
sim.sample(m = 10, Ji = rep(5,10), params=NULL)
}
\arguments{
    \item{m}{an integer, number of clusters in the sample.}
    \item{Ji}{a vector of length \code{m} containing the number of subjects
        in each cluster.}
    \item{params}{a list, optionally containing the parameters that specify the covariate 
        distribution, baseline hazard, frailty distribution, and censoring time,
        as follows:
        \describe{
            \item{\code{beta}}{the \dQuote{true} value of the single covariate.}
            \item{\code{haz.type}}{a string specifying the type of hazard to be used.}
            \item{\code{haz.params}}{a list with parameters corresponding to the hazard type.}
            \item{\code{frail.type}}{a string specifying the type of frailty to be used.}
            \item{\code{frail.params}}{a list with parameters corresponding to the frailty type.}
            \item{\code{Z.type}}{a string specifying the type of covariate to be used.}
            \item{\code{Z.params}}{a list with parameters corresponding to the covariate type.}
            \item{\code{C.type}}{a string specifying the type of censoring to be used.}
            \item{\code{C.params}}{a list with parameters corresponding to the censoring type.}
        }
        The default corresponds to standard normal covariates, a Weibull hazard,
        gamma frailties and a Weibull censoring time. See details.}
}
\details{
    The possible values for \code{frail.type}, \code{Z.type}, \code{C.type} and the
    corresponding entries in \code{frail.params}, \code{Z.params} and \code{C.params}
    are as follows:
    \describe{
        \item{\code{*.type="fixed"}}{\code{*.params} is a list with component \code{value}, containing
            the fixed value.}
        \item{\code{*.type="weibull"}}{\code{*.params} is a list with components \code{lambda0} and
            \code{gamweib}, for components \eqn{\lambda} and \eqn{\gamma} of the Weibull
            distribution.}
        \item{\code{*.type="gamma", *.type="normal", *.type="lognormal"}}{
            \code{*.params} is a list with components \code{mu} and \code{sigma2}, giving the
            mean and variance of the gamma, normal or lognormal distribution desired.}
        \item{\code{*.type="normmix", *.type="lognormmix"}}{\code{*.params} is a list of with components \code{mu}, \code{sigma2}
            and \code{w}, all vectors of the same length, giving the mean, variance and weight of each
            component of the normal or lognormal mixture.}
    }
    For \code{haz.type}, the possible values and parameters are:
    \describe{
        \item{\code{haz.type="weibull"}}{\code{haz.params} is a list with components \code{lambda0} and
            \code{gamweib}, for components \eqn{\lambda} and \eqn{\gamma} of the Weibull
            distribution.}
        \item{\code{haz.type="stepfunction"}}{\code{haz.params} is a list with vectors \code{breaks} giving
            the positions of jumps in the stepfunction, and \code{haz} giving the initial value of the hazard
            and its value after each jump.}
        \item{\code{haz.type="bspline"}}{\code{haz.params} is a list with components \code{b}, a basis matrix
            generated by \code{\link{bs}} specifying the B-spline basis, and \code{w}, a vector of weights
            for each basis element.}
    }
}
\value{
    A list with three components:
    \item{agdata}{a data.frame in with columns
    \describe{
        \item{i}{cluster id}
        \item{j}{patient id within the cluster}
        \item{time}{time of event}
        \item{delta}{censoring indicator}
        \item{Z}{value of the covariate}
    }
    }
    \item{Ui}{a vector of frailties used in the simulation}
    \item{params}{the set of parameters in the format of the input \code{params}.
        See details.}
}
\seealso{
\code{\link{splinesurv}}
}
\examples{
    # Generate a simulated sample with default parameters
    # (Weibull baseline and gamma frailty)
    s <- sim.sample(m = 10, Ji = rep(10,10))
    plot(survfit(coxph(Surv(time,delta) ~ Z + frailty(i), data = s$agdata)))
    
    # Generate a sample with piecewise baseline hazard and
    # lognormal mixture frailties
    s2 <- sim.sample(m = 10, Ji = rep(10,10), params = list(
        haz.type = "stepfunction", haz.params = list(breaks = c(.25, .5), 
        haz = c(1, 0 , 2)), frail.type = "lognormmix", 
        frail.params = list(mu = c(.5, 2), sigma2 = c(.2, .2), 
        w = c(.666, .333))))
    plot(survfit(coxph(Surv(time,delta) ~ Z + frailty(i), data = s2$agdata)))   
}
\keyword{survival}
