% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmidwpred.R
\name{glmidwpred}
\alias{glmidwpred}
\title{Generate spatial predictions using the hybrid method of generalised
linear models  ('glm') and inverse distance weighted ('IDW') ('glmidw')}
\usage{
glmidwpred(
  formula = NULL,
  longlat,
  trainxy,
  y,
  longlatpredx,
  predx,
  family = "gaussian",
  idp = 2,
  nmaxidw = 12,
  ...
)
}
\arguments{
\item{formula}{a formula defining the response variable and predictive variables
for 'glm'.}

\item{longlat}{a dataframe contains longitude and latitude of point samples. The
location information must be named as 'long' and 'lat'.}

\item{trainxy}{a dataframe contains longitude (long), latitude (lat),
predictive variables and the response variable of point samples. That is,
the location information must be named as 'long' and 'lat'.}

\item{y}{a vector of the response variable in the formula, that is, the left
part of the formula.}

\item{longlatpredx}{a dataframe contains longitude and latitude of point locations
(i.e., the centers of grids) to be predicted.}

\item{predx}{a dataframe or matrix contains columns of predictive variables
for the grids to be predicted.}

\item{family}{a description of the error distribution and link function to
be used in the model. See '?glm' for details.}

\item{idp}{a numeric number specifying the inverse distance weighting power.}

\item{nmaxidw}{for a local predicting: the number of nearest observations that
should be used for a prediction or simulation, where nearest is defined in
terms of the space of the spatial locations. By default, 12 observations
are used.}

\item{...}{other arguments passed on to 'glm'.}
}
\value{
A dataframe of longitude, latitude, and predictions.
}
\description{
This function is for generating spatial predictions using the hybrid
method of 'glm' and 'idw' ('glmidw') (see reference #1).
}
\examples{
\donttest{
library(spm)
data(petrel)
data(petrel.grid)

gravel <- petrel[, c(1, 2, 6:9, 5)]
longlat <- petrel[, c(1, 2)]
model <- log(gravel + 1) ~  lat +  bathy + I(long^3) + I(lat^2) + I(lat^3)
y <- log(gravel[, 7] +1)

glmidwpred1 <- glmidwpred(formula = model, longlat = longlat, trainxy =  gravel,
y = y, longlatpredx = petrel.grid[, c(1:2)], predx = petrel.grid, idp = 2,
 nmaxidw = 12)
 # Since the default 'family' is used, actually a 'lm' model is used.

names(glmidwpred1)

# Back transform 'glmidwpred$predictions' to generate the final predictions
glmidwpred1$predictions.bt <- exp(glmidwpred1$predictions) - 1
range(glmidwpred1$predictions.bt)
}

}
\references{
Li, J., Alvarez, B., Siwabessy, J., Tran, M., Huang, Z.,
Przeslawski, R., Radke, L., Howard, F. and Nichol, S. (2017). "Application
of random forest, generalised linear model and their hybrid methods with
geostatistical techniques to count data: Predicting sponge species richness."
Environmental Modelling & Software 97: 112-129.

Pebesma, E.J., 2004. Multivariable geostatistics in S: the gstat package.
Computers & Geosciences, 30: 683-691.
}
\author{
Jin Li
}
