#' Get Current User's Saved Albums
#'
#' Get a list of the albums saved in the current Spotify user’s ‘Your Music’ library.
#'
#' @param limit Optional. \cr
#' Maximum number of albums to return. \cr
#' Default: 20 \cr
#' Minimum: 1 \cr
#' Maximum: 50 \cr
#' @param offset Optional. \cr
#' The index of the first albums to return. \cr
#' Default: 0 (the first object). Maximum offset: 100,000. Use with
#' \code{limit} to get the next set of albums.
#' @param market Optional. \cr
#' An \href{https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2}{ISO 3166-1 alpha-2 country code} or the string \code{"from_token"}. Provide this parameter if you want to apply \href{https://developer.spotify.com/documentation/general/guides/track-relinking-guide/}{Track Relinking}
#' @param authorization Required. A valid access token from the Spotify Accounts service. See the \href{https://developer.spotify.com/documentation/general/guides/authorization-guide/}{Web API authorization Guide} for more details. Defaults to \code{spotifyr::get_spotify_authorization_code()}. The access token must have been issued on behalf of the current user.
#' @param include_meta_info Optional. Boolean indicating whether to include full result, with meta information such as \code{"total"}, and \code{"limit"}. Defaults to \code{FALSE}.
#' @return
#' Returns a data frame of results containing user profile information. See
#' \url{https://developer.spotify.com/documentation/web-api/reference/library/get-users-saved-albums/} for more information.
#' @export

get_my_saved_albums <- function(limit = 20,
                                offset = 0,
                                market = NULL,
                                authorization = get_spotify_authorization_code(),
                                include_meta_info = FALSE) {

   validate_parameters(limit=limit,
                       offset=offset,
                       market=market,
                       include_meta_info = include_meta_info )

    base_url <- 'https://api.spotify.com/v1/me/albums'
    if (!is.null(market)) {
        if (str_detect(market, '^[[:alpha:]]{2}$')) {
            stop('"market" must be an ISO 3166-1 alpha-2 country code')
        }
    }

    params <- list(
        limit = limit,
        offset = offset,
        market = market
    )

    res <- RETRY('GET', base_url,
                 query = params,
                 config(token = authorization), encode = 'json')

    stop_for_status(res)

    res <- fromJSON(content(res, as = 'text', encoding = 'UTF-8'), flatten = TRUE)

    if (!include_meta_info) {
        res <- res$items
    }

   res
}

#' Get User's Saved Tracks
#'
#' Get a list of the songs saved in the current Spotify user’s ‘Your Music’ library.
#'
#' @param limit Optional. \cr
#' Maximum number of tracks to return. \cr
#' Default: 20 \cr
#' Minimum: 1 \cr
#' Maximum: 50 \cr
#' @param offset Optional. \cr
#' The index of the first track to return. \cr
#' Default: 0 (the first object). Maximum offset: 100,000. Use with \code{limit} to get the next set of tracks.
#' @param market Optional. \cr
#' An \href{https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2}{ISO 3166-1 alpha-2 country code} or the string \code{"from_token"}. Provide this parameter if you want to apply \href{https://developer.spotify.com/documentation/general/guides/track-relinking-guide/}{Track Relinking}
#' @param authorization Required. A valid access token from the Spotify Accounts service.
#' See the \href{https://developer.spotify.com/documentation/general/guides/authorization-guide/}{Web API authorization Guide} for more details. Defaults to \code{spotifyr::get_spotify_authorization_code()}. The access token must have been issued on behalf of the current user.
#' @param include_meta_info Optional. Boolean indicating whether to include full result,
#' with meta information such as \code{"total"}, and \code{"limit"}. Defaults to \code{FALSE}.
#' @return
#' Returns a data frame of results containing user profile information.
#' See \url{https://developer.spotify.com/documentation/web-api/reference/users-profile/get-current-users-profile/} for more information.
#' @export

get_my_saved_tracks <- function(limit = 20,
                                offset = 0,
                                market = NULL,
                                authorization = get_spotify_authorization_code(),
                                include_meta_info = FALSE) {

    validate_parameters(market=market, limit=limit, offset = offset, include_meta_info)

    base_url <- 'https://api.spotify.com/v1/me/tracks'


    params <- list(
        limit = limit,
        offset = offset,
        market = market
    )

    res <- RETRY('GET', base_url, query = params,
                 config(token = authorization),
                 encode = 'json')

    stop_for_status(res)

    res <- fromJSON(content(res, as = 'text', encoding = 'UTF-8'),
                    flatten = TRUE)

    if (!include_meta_info) {
        res <- res$items
    }

    res
}
