\name{3. Mechanistic models}
\alias{sraCstvar}
\alias{sraDrift}
\alias{sraMutation}
\alias{sraCanalization}
\alias{sraCanalizationOpt}
\alias{sraSelection}
\alias{sraDirepistasis}
\title{ Descriptive models of artificial-selection responses: quantitative genetics models }
\description{
  The sra functions for mechanistic model are wrappers for the maximum-likelihood optimization routine \code{\link{mle}}. They implement classical quantitative genetics models, fit them to artificial-selection time series, and provide estimates of e.g. the effective population size, the mutational variance, the strength of genetic / environmental canalization, the directionality and strength of epistasis, etc., given some assumptions about the properties of the genetic architecture. 
}
\usage{
sraCstvar(sradata, start=NULL, fixed=NULL, macroE=FALSE, Bulmer=TRUE, ...)
sraDrift(sradata, start=NULL, fixed=NULL, macroE=FALSE, Bulmer=TRUE, ...)
sraMutation(sradata, start=NULL, fixed=NULL, macroE=FALSE, Bulmer=TRUE, ...)
sraCanalization(sradata, start=NULL, fixed=NULL, macroE=FALSE, Bulmer=TRUE, ...)
sraCanalizationOpt(sradata, start=NULL, fixed=NULL, macroE=FALSE, Bulmer=TRUE, ...)
sraSelection(sradata, start=NULL, fixed=NULL, macroE=FALSE, Bulmer=TRUE, ...)
sraDirepistasis(sradata, start=NULL, fixed=NULL, macroE=FALSE, ...)
}

\arguments{
  \item{sradata}{ A data object generated by \code{\link{sraData}}. }
  \item{start}{ A named list of starting values for the convergence algorithm. \code{NA} is allowed and lets the function \code{\link{sraStartingvalues}} calculating a (hopefully) educated guess on the starting values. Parameters are described below. }
  \item{fixed}{ A named list of the parameters that have to be kept constant. \code{NA} is not allowed. }
  \item{macroE}{ Whether or not macro-environmental effects (random deviation of the phenotypic mean each generation) should be included. This might have some side effects. }
  \item{Bulmer}{ Whether or not the impact of linkage disequilibrium (Bulmer effect) due to selection on variance should be accounted for. }
  \item{\dots}{ Additional parameters to be passed to \code{\link{mle}}, and thus to \code{\link{optim}}. }
}
\details{
  All functions (except \code{sraDirepistasis}) rely on the same underlying model, and thus simply provide convenient shortcuts for different sets of parameters to fit. 
  
  \code{mu0} is the initial phenotype of the population.
  
  \code{logvarA0} is the logarithm of the initial additive variance in the population.
  
  \code{logvarE0} is the logarithm of the initial environmental variance in the population.
  
  \code{logNe} is the logarithm of the effective population size.
  
  \code{logn} is the logarithm of the effective number of loci.
  
  \code{logvarM} is the logarithm of the mutational variance.
  
  \code{kc} and \code{kg} are the strength of environmental and genetic canalization, respectively. 
  
  \code{o} corresponds to the 'optimum' phenotype. When fixed and set to \code{NA}, \code{o} is identical to \code{mu0}. For convenience, the same optimum is used for environmental canalization, genetic canalization and natural stabilizing selection. 
  
  \code{s} corresponds to the strength of natural selection.
  
  \code{logvarepsilon} is the logarithm of the variance of the epistatic coefficient (\eqn{\varepsilon}{epsilon}). This parameter is fixed by default, since it is unlikely that realistic data sets contain enough information to estimate it properly. 
  
  The dynamic model that is fitted (except of the directional epistasis, detailed in the next paragraph) is:
  
  \deqn{\mu(t+1) = \mu(t) + VarA(t) * (\beta(t) + s \delta(t)) }{mu(t+1) = mu(t) + VarA(t) * (beta(t) + s * delta(t)) }
  \deqn{VarA(t+1) = VarM + VarA(t) * (1 - 1/(2*N_e)) * e^{kg*(|\delta(t+1)| - |\delta(t)|)}}{VarA(t+1) = VarM + VarA(t) * (1 - 1/(2*Ne)) * exp(kg*(|delta(t+1)| - |delta(t)|))}
  \deqn{VarE(t+1) = VarE(t) * e^{kc*|\delta(t)|}}{VarE(t+1) = VarE(t) * exp(kc*|delta(t)|)}
  
  \eqn{\mu(1)}, \eqn{VarA(1)} and \eqn{varE(1)} are parameters of the model, \eqn{\beta(t)} is the selection gradient, calculated for each generation from the data set, and \eqn{\delta(t) = \mu(t) - o}{delta(t) = mu(t) - o}. 
  
  The directional epistasis model has its own setting:
  
  \deqn{\mu(t+1) = \mu(t) + varA(t) * \beta(t)}{mu(t+1) = mu(t) + varA(t) * beta(t)}
  \deqn{VarA(t+1) = VarA(t) + 2*\beta(t)*\varepsilon*VarA(t)^2}{VarA(t+1) = VarA(t) + 2*beta(t)*epsilon*VarA(t)^2}
  \deqn{VarE(t+1) = VarE(1) + (\varepsilon^2 + Var\varepsilon)*VarA(t)^2}{VarE(t+1) = VarE(1) + (epsilon^2 + varepsilon)*VarA(t)^2}
  
  Where \code{epsilon} is a key parameter, representing the directionality of epistasis (Hansen \& Wagner 2001, Carter et al. 2005). The properties of the likelihood function when epsilon varies makes numerical convergence tricky, and the function \code{SRAdirepistasis} actually performs two model fits: one for positive epsilons (the estimated parameter being \code{logepsilon}) and one for negative epsilons (estimating \code{logminusepsilon}). The likelihood of both models is then compared, and the best model is returned, providing either \code{logepsilon} or \code{logminusepsilon}. Although part of the model, the parameter \code{logvarepsilon} appeared to affect environmental variance only weakly, and its estimation is problematic in most cases. 
  
  These models are extensively described in le Rouzic et al 2010. 
}
\value{
The functions return objects of class \code{srafit}, a list containing information about the model, the data, and the parameter estimates. Some standard \R functions can be applied to the object, including \code{AIC} (\code{\link{AIC.srafit}}), \code{logLik} (\code{\link{logLik.srafit}}), \code{vcov} (\code{\link{vcov.srafit}}), \code{coef} (\code{\link{coef.srafit}}) \code{confint} (\code{\link{confint.srafit}}), and \code{plot} (\code{\link{plot.srafit}}).
}
\references{ 

Carter, A.J.R., Hermisson, J. and Hansen, T.F. (2005) The role of epistatic genetic interactions in the response to selection and the evolution of evolvability. Theor. Pop. Biol. 68, 179-196. 

Hansen, T.F. and Wagner, G.P. (2001) Modelling genetic architecture: a multilinear theory of gene interaction. Theor. Pop. biol. 59, 61-86.

Le Rouzic, A., Houle, D., and Hansen, T.F. (2010) A modelling framework for the analysis of artificial selection-response time series. in prep.
 }
\author{ Arnaud Le Rouzic }

\seealso{ \code{\link{sraAutoreg}}, \code{\link{sraAutoregHerit}} and \code{\link{sraAutoregEvolv}} for phenomenological models, \code{\link{sraAutoregTsMinuslogL}} and \code{\link{sraAutoregTimeseries}} for some details about the internal functions, \code{\link{AIC.srafit}}, \code{\link{logLik.srafit}},\code{\link{vcov.srafit}}, \code{\link{coef.srafit}}, \code{\link{confint.srafit}}, \code{\link{plot.srafit}} for the analysis of the results. }
\examples{
########### Generating a dummy dataset ################

m <- c(12,11,12,14,18,17,19,22,20,19)
v <- c(53,47,97,155,150,102,65,144,179,126)
s <- c(15,14,14,17,21,20,22,25,24,NA)
n <- c(100,80,120,60,100,90,110,80,60,100)

########## Making a sra data set #######################
data <- sraData(phen.mean=m, phen.var=v, phen.sel=s, N=n)

#################### Data Analysis ####################

cstvar <- sraCstvar(data)
drift <- sraDrift(data)
direpi <- sraDirepistasis(data)

# In case of convergence problems, better starting values can be provided:
direpi <- sraDirepistasis(data, start=list(mu0=10, logvarA0=log(20), logvarE0=NA), 
fixed=list(logNe=log(50)))

plot(cstvar)

AIC(direpi)
}

\keyword{ models }
\keyword{ nonlinear }
\keyword{ ts }

