#' @title Making expression containing parametric ANOVA results
#' @name expr_anova_parametric
#' @author \href{https://github.com/IndrajeetPatil}{Indrajeet Patil}
#'
#' @return For more details, see-
#' \url{https://indrajeetpatil.github.io/statsExpressions/articles/stats_details.html}
#'
#' @note For repeated measures designs (`paired = TRUE`), only omega-squared and
#'   partial eta-squared effect sizes are supported.
#'
#' @inheritParams t1way_ci
#' @param paired Logical that decides whether the experimental design is
#'   repeated measures/within-subjects or between-subjects. The default is
#'   `FALSE`.
#' @param effsize.type Type of effect size needed for *parametric* tests. The
#'   argument can be `"biased"` (equivalent to `"d"` for Cohen's *d* for
#'   **t-test**; `"partial_eta"` for partial eta-squared for **anova**) or
#'   `"unbiased"` (equivalent to `"g"` Hedge's *g* for **t-test**;
#'   `"partial_omega"` for partial omega-squared for **anova**)).
#' @param sphericity.correction Logical that decides whether to apply correction
#'   to account for violation of sphericity in a repeated measures design ANOVA
#'   (Default: `TRUE`).
#' @param messages Decides whether messages references, notes, and warnings are
#'   to be displayed (Default: `TRUE`).
#' @inheritParams expr_template
#' @param ... Additional arguments (currently ignored).
#' @inheritParams stats::oneway.test
#' @inheritParams groupedstats::lm_effsize_standardizer
#'
#'
#' @importFrom dplyr select
#' @importFrom rlang !! enquo eval_tidy expr ensym
#' @importFrom stats lm oneway.test na.omit
#' @importFrom ez ezANOVA
#' @importFrom groupedstats lm_effsize_standardizer
#'
#' @examples
#' \donttest{
#' # for reproducibility
#' set.seed(123)
#' library(statsExpressions)
#'
#' # -------------------- between-subjects ------------------------------
#'
#' # with defaults
#' statsExpressions::expr_anova_parametric(
#'   data = ggplot2::msleep,
#'   x = vore,
#'   y = sleep_rem,
#'   paired = FALSE,
#'   k = 3
#' )
#'
#' # modifying the defaults
#' statsExpressions::expr_anova_parametric(
#'   data = ggplot2::msleep,
#'   x = vore,
#'   y = sleep_rem,
#'   paired = FALSE,
#'   effsize.type = "biased",
#'   partial = FALSE,
#'   var.equal = TRUE,
#'   nboot = 10
#' )
#'
#' # -------------------- repeated measures ------------------------------
#'
#' statsExpressions::expr_anova_parametric(
#'   data = iris_long,
#'   x = condition,
#'   y = value,
#'   paired = TRUE,
#'   k = 4,
#'   nboot = 10
#' )
#' }
#' @export

# function body
expr_anova_parametric <- function(data,
                                  x,
                                  y,
                                  paired = FALSE,
                                  effsize.type = "unbiased",
                                  partial = TRUE,
                                  conf.level = 0.95,
                                  nboot = 100,
                                  var.equal = FALSE,
                                  sphericity.correction = TRUE,
                                  k = 2,
                                  stat.title = NULL,
                                  messages = TRUE,
                                  ...) {

  # make sure both quoted and unquoted arguments are allowed
  x <- rlang::ensym(x)
  y <- rlang::ensym(y)

  # for paired designs, variance is going to be equal across grouping levels
  if (isTRUE(paired)) {
    var.equal <- TRUE
  } else {
    sphericity.correction <- FALSE
  }

  # number of decimal places for degree of freedom
  if (isTRUE(var.equal) && isFALSE(sphericity.correction)) {
    k.df2 <- 0L
  } else {
    k.df2 <- k
  }

  # denominator degrees of freedom
  if (isTRUE(paired) && isTRUE(sphericity.correction)) {
    k.df1 <- k
  } else {
    k.df1 <- 0L
  }

  # figuring out which effect size to use
  effsize.type <- effsize_type_switch(effsize.type)

  # some of the effect sizes don't work properly for paired designs
  if (isTRUE(paired)) {
    if (effsize.type == "unbiased") {
      partial <- FALSE
    } else {
      partial <- TRUE
    }
  }

  # omega
  if (effsize.type == "unbiased") {
    effsize <- "omega"
    if (isTRUE(partial)) {
      effsize.text <- quote(omega["p"]^2)
    } else {
      effsize.text <- quote(omega^2)
    }
  }

  # eta
  if (effsize.type == "biased") {
    effsize <- "eta"
    if (isTRUE(partial)) {
      effsize.text <- quote(eta["p"]^2)
    } else {
      effsize.text <- quote(eta^2)
    }
  }

  # creating a dataframe
  data %<>%
    dplyr::select(.data = ., {{ x }}, {{ y }}) %>%
    dplyr::mutate(.data = ., {{ x }} := droplevels(as.factor({{ x }}))) %>%
    tibble::as_tibble(x = .)

  # -------------- within-subjects design --------------------------------

  # properly removing NAs if it's a paired design
  if (isTRUE(paired)) {
    # converting to long format and then getting it back in wide so that the
    # rowid variable can be used as the block variable
    data %<>% df_cleanup_paired(data = ., x = {{ x }}, y = {{ y }})

    # sample size
    sample_size <- length(unique(data$rowid))
    n.text <- quote(italic("n")["pairs"])

    # warn the user if
    if (sample_size < nlevels(as.factor(data %>% dplyr::pull({{ x }})))) {
      # no sphericity correction applied
      sphericity.correction <- FALSE
      k.df1 <- 0L
      k.df2 <- 0L

      # inform the user
      message(cat(
        crayon::red("Warning: "),
        crayon::blue("No. of factor levels is greater than no. of observations per cell.\n"),
        crayon::blue("No sphericity correction applied. Interpret the results with caution.\n")
      ),
      sep = ""
      )
    }

    # run the ANOVA
    ez_df <-
      rlang::eval_tidy(rlang::expr(
        ez::ezANOVA(
          data = data %>%
            dplyr::mutate_if(
              .tbl = .,
              .predicate = is.character,
              .funs = as.factor
            ) %>%
            dplyr::mutate(.data = ., rowid = as.factor(rowid)),
          dv = !!rlang::ensym(y),
          wid = rowid,
          within = !!rlang::ensym(x),
          detailed = TRUE,
          return_aov = TRUE
        )
      ))

    # list with results
    if (isTRUE(sphericity.correction)) {
      e_corr <- ez_df$`Sphericity Corrections`$GGe
      stats_df <-
        list(
          statistic = ez_df$ANOVA$F[2],
          parameter = c(e_corr * ez_df$ANOVA$DFn[2], e_corr * ez_df$ANOVA$DFd[2]),
          p.value = ez_df$`Sphericity Corrections`$`p[GG]`[[1]]
        )
    } else {
      stats_df <-
        list(
          statistic = ez_df$ANOVA$F[2],
          parameter = c(ez_df$ANOVA$DFn[2], ez_df$ANOVA$DFd[2]),
          p.value = ez_df$ANOVA$p[2]
        )
    }

    # creating a standardized dataframe with effect size and its CIs
    effsize_object <- ez_df$aov
  }

  # ------------------- between-subjects design ------------------------------

  if (isFALSE(paired)) {
    # remove NAs listwise for between-subjects design
    data %<>% tidyr::drop_na(data = .)

    # sample size
    sample_size <- nrow(data)
    n.text <- quote(italic("n")["obs"])

    # Welch's ANOVA run by default
    stats_df <-
      stats::oneway.test(
        formula = rlang::new_formula({{ y }}, {{ x }}),
        data = data,
        subset = NULL,
        na.action = na.omit,
        var.equal = var.equal
      )

    # creating a standardized dataframe with effect size and its CIs
    effsize_object <-
      stats::lm(
        formula = rlang::new_formula({{ y }}, {{ x }}),
        data = data,
        na.action = na.omit
      )
  }

  # creating a standardized dataframe with effect size and its CIs
  effsize_df <-
    groupedstats::lm_effsize_standardizer(
      object = effsize_object,
      effsize = effsize,
      partial = partial,
      conf.level = conf.level,
      nboot = nboot
    )

  # preparing subtitle
  subtitle <-
    expr_template(
      no.parameters = 2L,
      stat.title = stat.title,
      statistic.text = quote(italic("F")),
      statistic = stats_df$statistic[[1]],
      parameter = stats_df$parameter[[1]],
      parameter2 = stats_df$parameter[[2]],
      p.value = stats_df$p.value[[1]],
      effsize.text = effsize.text,
      effsize.estimate = effsize_df$estimate[[1]],
      effsize.LL = effsize_df$conf.low[[1]],
      effsize.UL = effsize_df$conf.high[[1]],
      n = sample_size,
      n.text = n.text,
      conf.level = conf.level,
      k = k,
      k.parameter = k.df1,
      k.parameter2 = k.df2
    )

  # message about effect size measure
  if (isTRUE(messages)) effsize_ci_message(nboot, conf.level)

  # return the subtitle
  return(subtitle)
}

#' @title Making text subtitle for nonparametric ANOVA.
#' @name expr_anova_nonparametric
#' @author \href{https://github.com/IndrajeetPatil}{Indrajeet Patil}
#'
#' @details For paired designs, the effect size is Kendall's coefficient of
#'   concordance (*W*), while for between-subjects designs, the effect size is
#'   epsilon-squared (for more, see `?rcompanion::epsilonSquared` and
#'   `?rcompanion::kendallW`).
#'
#' @return For more details, see-
#' \url{https://indrajeetpatil.github.io/statsExpressions/articles/stats_details.html}
#'
#' @inheritParams t1way_ci
#' @inheritParams expr_anova_parametric
#' @inheritParams expr_template
#'
#' @importFrom dplyr select
#' @importFrom rlang !! enquo
#' @importFrom stats friedman.test kruskal.test
#' @importFrom broomExtra tidy
#' @importFrom rcompanion epsilonSquared kendallW
#'
#' @examples
#' \donttest{
#' # setup
#' set.seed(123)
#' library(statsExpressions)
#'
#' # -------------- within-subjects design --------------------------------
#'
#' # creating the subtitle
#' statsExpressions::expr_anova_nonparametric(
#'   data = bugs_long,
#'   x = condition,
#'   y = desire,
#'   paired = TRUE,
#'   conf.level = 0.99,
#'   k = 2
#' )
#'
#' # -------------- between-subjects design --------------------------------
#'
#' statsExpressions::expr_anova_nonparametric(
#'   data = ggplot2::msleep,
#'   x = vore,
#'   y = sleep_rem,
#'   paired = FALSE,
#'   conf.level = 0.99,
#'   conf.type = "perc"
#' )
#' }
#' @export

# function body
expr_anova_nonparametric <- function(data,
                                     x,
                                     y,
                                     paired = FALSE,
                                     conf.type = "perc",
                                     conf.level = 0.95,
                                     k = 2,
                                     nboot = 100,
                                     stat.title = NULL,
                                     messages = TRUE,
                                     ...) {

  # make sure both quoted and unquoted arguments are allowed
  x <- rlang::ensym(x)
  y <- rlang::ensym(y)

  # creating a dataframe
  data %<>%
    dplyr::select(.data = ., {{ x }}, {{ y }}) %>%
    dplyr::mutate(.data = ., {{ x }} := droplevels(as.factor({{ x }}))) %>%
    tibble::as_tibble(x = .)

  # ------------------- within-subjects design ------------------------------

  # properly removing NAs if it's a paired design
  if (isTRUE(paired)) {
    # calculating Kendall's W and its CI
    effsize_df <-
      rcompanion::kendallW(
        x = dplyr::select(long_to_wide_converter(data, {{ x }}, {{ y }}), -rowid),
        correct = TRUE,
        na.rm = TRUE,
        ci = TRUE,
        conf = conf.level,
        type = conf.type,
        R = nboot,
        histogram = FALSE,
        digits = 5
      ) %>%
      rcompanion_cleaner(object = ., estimate.col = "W")

    # converting to long format and then getting it back in wide so that the
    # rowid variable can be used as the block variable
    data %<>% df_cleanup_paired(data = ., x = {{ x }}, y = {{ y }})

    # sample size
    sample_size <- length(unique(data$rowid))
    n.text <- quote(italic("n")["pairs"])

    # setting up the anova model (`y ~ x | id`) and getting its summary
    stats_df <-
      broomExtra::tidy(stats::friedman.test(
        formula = rlang::new_formula(
          {{ rlang::enexpr(y) }}, rlang::expr(!!rlang::enexpr(x) | rowid)
        ),
        data = data,
        na.action = na.omit
      ))

    # text for effect size
    effsize.text <- quote(italic("W")["Kendall"])
  }

  # ------------------- between-subjects design ------------------------------

  if (isFALSE(paired)) {
    # remove NAs listwise for between-subjects design
    data %<>% tidyr::drop_na(data = .)

    # sample size
    sample_size <- nrow(data)
    n.text <- quote(italic("n")["obs"])

    # setting up the anova model and getting its summary
    stats_df <-
      broomExtra::tidy(stats::kruskal.test(
        formula = rlang::new_formula({{ y }}, {{ x }}),
        data = data,
        na.action = na.omit
      ))

    # getting partial eta-squared based on H-statistic
    effsize_df <-
      rcompanion::epsilonSquared(
        x = data %>% dplyr::pull({{ y }}),
        g = data %>% dplyr::pull({{ x }}),
        group = "row",
        ci = TRUE,
        conf = conf.level,
        type = conf.type,
        R = nboot,
        histogram = FALSE,
        digits = 5
      ) %>%
      rcompanion_cleaner(object = ., estimate.col = "epsilon.squared")

    # text for effect size
    effsize.text <- quote(epsilon^2)
  }

  # message about effect size measure
  if (isTRUE(messages)) effsize_ci_message(nboot, conf.level)

  # preparing subtitle
  subtitle <-
    expr_template(
      no.parameters = 1L,
      stat.title = stat.title,
      statistic.text = quote(italic(chi)^2),
      statistic = stats_df$statistic[[1]],
      parameter = stats_df$parameter[[1]],
      p.value = stats_df$p.value[[1]],
      effsize.text = effsize.text,
      effsize.estimate = effsize_df$estimate[[1]],
      effsize.LL = effsize_df$conf.low[[1]],
      effsize.UL = effsize_df$conf.high[[1]],
      n = sample_size,
      n.text = n.text,
      conf.level = conf.level,
      k = k,
      k.parameter = 0L
    )

  # return the subtitle
  return(subtitle)
}

#' @title Expression containing results from heteroscedastic one-way ANOVA for
#'   trimmed means
#' @name expr_anova_robust
#' @author \href{https://github.com/IndrajeetPatil}{Indrajeet Patil}
#'
#' @return For more details, see-
#' \url{https://indrajeetpatil.github.io/statsExpressions/articles/stats_details.html}
#'
#' @inheritParams t1way_ci
#' @inheritParams expr_anova_nonparametric
#' @inheritParams expr_template
#'
#' @importFrom dplyr select
#' @importFrom rlang !! enquo ensym as_name
#' @importFrom WRS2 rmanova
#'
#' @examples
#'
#' \donttest{
#' # for reproducibility
#' set.seed(123)
#'
#' # ------------------------ between-subjects -----------------------------
#'
#' # going with the defaults
#' statsExpressions::expr_anova_robust(
#'   data = ggplot2::midwest,
#'   x = state,
#'   y = percbelowpoverty,
#'   paired = FALSE,
#'   nboot = 10
#' )
#'
#' # changing defaults
#' expr_anova_robust(
#'   data = ggplot2::midwest,
#'   x = state,
#'   y = percollege,
#'   paired = FALSE,
#'   conf.level = 0.99,
#'   tr = 0.2,
#'   nboot = 10
#' )
#'
#' # ------------------------ within-subjects -----------------------------
#'
#' statsExpressions::expr_anova_robust(
#'   data = iris_long,
#'   x = condition,
#'   y = value,
#'   paired = TRUE,
#'   tr = 0.2,
#'   k = 3
#' )
#' }
#' @export

# function body
expr_anova_robust <- function(data,
                              x,
                              y,
                              paired = FALSE,
                              tr = 0.1,
                              nboot = 100,
                              conf.level = 0.95,
                              conf.type = "norm",
                              k = 2,
                              stat.title = NULL,
                              messages = TRUE,
                              ...) {

  # make sure both quoted and unquoted arguments are allowed
  x <- rlang::ensym(x)
  y <- rlang::ensym(y)

  # creating a dataframe
  data %<>%
    dplyr::select(.data = ., {{ x }}, {{ y }}) %>%
    dplyr::mutate(.data = ., {{ x }} := droplevels(as.factor({{ x }}))) %>%
    tibble::as_tibble(x = .)

  # -------------- within-subjects design --------------------------------

  # properly removing NAs if it's a paired design
  if (isTRUE(paired)) {
    # converting to long format and then getting it back in wide so that the
    # rowid variable can be used as the block variable
    data %<>% df_cleanup_paired(data = ., x = {{ x }}, y = {{ y }})

    # sample size
    sample_size <- length(unique(data$rowid))

    # test
    stats_df <-
      WRS2::rmanova(
        y = data[[rlang::as_name(y)]],
        groups = data[[rlang::as_name(x)]],
        blocks = data[["rowid"]],
        tr = tr
      )

    # preparing the subtitle
    subtitle <-
      substitute(
        expr = paste(
          italic("F"),
          "(",
          df1,
          ",",
          df2,
          ") = ",
          estimate,
          ", ",
          italic("p"),
          " = ",
          p.value,
          ", ",
          italic("n")["pairs"],
          " = ",
          n
        ),
        env = list(
          estimate = specify_decimal_p(x = stats_df$test[[1]], k = k),
          df1 = specify_decimal_p(x = stats_df$df1[[1]], k = k),
          df2 = specify_decimal_p(x = stats_df$df2[[1]], k = k),
          p.value = specify_decimal_p(x = stats_df$p.value[[1]], k = k, p.value = TRUE),
          n = sample_size
        )
      )
  }

  # -------------- between-subjects design --------------------------------

  if (isFALSE(paired)) {
    # remove NAs listwise for between-subjects design
    data %<>% tidyr::drop_na(data = .)

    # sample size
    sample_size <- nrow(data)
    n.text <- quote(italic("n")["obs"])

    # setting up the Bootstrap version of the heteroscedastic one-way ANOVA for
    # trimmed means
    stats_df <-
      t1way_ci(
        data = data,
        x = {{ x }},
        y = {{ y }},
        tr = tr,
        nboot = nboot,
        conf.level = conf.level,
        conf.type = conf.type
      )

    # preparing subtitle
    subtitle <-
      expr_template(
        no.parameters = 2L,
        stat.title = stat.title,
        statistic.text = quote(italic("F")),
        statistic = stats_df$F.value[[1]],
        parameter = stats_df$df1[[1]],
        parameter2 = stats_df$df2[[1]],
        p.value = stats_df$p.value[[1]],
        effsize.text = quote(italic(xi)),
        effsize.estimate = stats_df$xi[[1]][[1]],
        effsize.LL = stats_df$conf.low[[1]],
        effsize.UL = stats_df$conf.high[[1]],
        n = sample_size,
        n.text = n.text,
        conf.level = conf.level,
        k = k,
        k.parameter = 0L,
        k.parameter2 = k
      )

    # message about effect size measure
    if (isTRUE(messages)) effsize_ci_message(nboot, conf.level)
  }

  # return the subtitle
  return(subtitle)
}


#' @title Making expression containing Bayesian one-way ANOVA results.
#' @name expr_anova_bayes
#' @author \href{https://github.com/IndrajeetPatil}{Indrajeet Patil}
#'
#' @return For more details, see-
#' \url{https://indrajeetpatil.github.io/statsExpressions/articles/stats_details.html}
#'
#' @inheritParams expr_anova_parametric
#' @inheritParams expr_t_bayes
#'
#' @importFrom dplyr select
#' @importFrom rlang !! enquo
#' @importFrom stats lm oneway.test na.omit
#'
#' @examples
#' \donttest{
#' set.seed(123)
#'
#' # between-subjects ---------------------------------------
#' # with defaults
#' statsExpressions::expr_anova_bayes(
#'   data = ggplot2::msleep,
#'   x = vore,
#'   y = sleep_rem
#' )
#'
#' # modifying the defaults
#' statsExpressions::expr_anova_bayes(
#'   data = ggplot2::msleep,
#'   x = vore,
#'   y = sleep_rem,
#'   k = 3,
#'   bf.prior = 0.8
#' )
#'
#' # repeated measures ---------------------------------------
#' statsExpressions::expr_anova_bayes(
#'   data = WRS2::WineTasting,
#'   x = Wine,
#'   y = Taste,
#'   paired = TRUE,
#'   k = 4
#' )
#' }
#' @export

# function body
expr_anova_bayes <- function(data,
                             x,
                             y,
                             paired = FALSE,
                             bf.prior = 0.707,
                             k = 2,
                             ...) {

  # make sure both quoted and unquoted arguments are allowed
  x <- rlang::ensym(x)
  y <- rlang::ensym(y)

  # creating a dataframe
  data %<>%
    dplyr::select(.data = ., {{ x }}, {{ y }}) %>%
    dplyr::mutate(.data = ., {{ x }} := droplevels(as.factor({{ x }}))) %>%
    tibble::as_tibble(x = .)

  # properly removing NAs if it's a paired design
  # converting to long format and then getting it back in wide so that the
  # rowid variable can be used as the block variable
  if (isTRUE(paired)) data %<>% df_cleanup_paired(data = ., x = {{ x }}, y = {{ y }})

  # remove NAs listwise for between-subjects design
  if (isFALSE(paired)) data %<>% tidyr::drop_na(.)

  # bayes factor results
  subtitle <-
    bf_oneway_anova(
      data = data,
      x = {{ x }},
      y = {{ y }},
      paired = paired,
      bf.prior = bf.prior,
      k = k,
      caption = NULL,
      output = "h1"
    )

  # return the subtitle
  return(subtitle)
}
