#' @title Expression for one sample *t*-test and its non-parametric and
#'   robust equivalents
#' @name expr_t_onesample
#'
#' @param x A numeric variable from the dataframe `data`.
#' @param test.value A number specifying the value of the null hypothesis
#'   (Default: `0`).
#' @param type Type of statistic expected (`"parametric"` or `"nonparametric"`
#'   or `"robust"` or `"bayes"`).Corresponding abbreviations are also accepted:
#'   `"p"` (for parametric), `"np"` (nonparametric), `"r"` (robust), or
#'   `"bf"`resp.
#' @param robust.estimator If `type = "robust"`, a robust estimator to be used
#'   (`"onestep"` (Default), `"mom"`, or `"median"`). For more, see
#'   `?WRS2::onesampb`.
#' @param ... Additional arguments (currently ignored).
#' @inheritParams expr_t_parametric
#' @inheritParams tidyBF::bf_corr_test
#' @inheritParams expr_anova_parametric
#' @inheritParams expr_anova_nonparametric
#'
#' @return Expression containing results from a one-sample test. The exact test
#'   and the effect size details contained will be dependent on the `type`
#'   argument.
#'
#' @references For more details, see-
#' \url{https://indrajeetpatil.github.io/statsExpressions/articles/stats_details.html}
#'
#' @importFrom dplyr select mutate pull rename_all recode
#' @importFrom WRS2 onesampb
#' @importFrom rcompanion wilcoxonOneSampleR
#' @importFrom ipmisc stats_type_switch
#' @importFrom effectsize cohens_d hedges_g
#' @importFrom stats t.test wilcox.test
#' @importFrom rlang !! ensym new_formula exec
#' @importFrom broomExtra easystats_to_tidy_names
#'
#' @examples
#' \donttest{
#' # for reproducibility
#' set.seed(123)
#'
#' # ----------------------- parametric ---------------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "parametric"
#' )
#'
#' # ----------------------- non-parametric -----------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "nonparametric"
#' )
#'
#' # ----------------------- robust --------------------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "robust"
#' )
#'
#' # ----------------------- Bayes Factor -----------------------------------
#'
#' statsExpressions::expr_t_onesample(
#'   data = ggplot2::msleep,
#'   x = brainwt,
#'   test.value = 0.275,
#'   type = "bayes",
#'   bf.prior = 0.8
#' )
#' }
#' @export

expr_t_onesample <- function(data,
                             x,
                             type = "parametric",
                             test.value = 0,
                             k = 2L,
                             conf.level = 0.95,
                             conf.type = "norm",
                             bf.prior = 0.707,
                             robust.estimator = "onestep",
                             effsize.type = "g",
                             nboot = 100L,
                             stat.title = NULL,
                             ...) {

  # ====================== dataframe ========================================

  # preparing a dataframe out of provided inputs
  data %<>%
    dplyr::select(.data = ., {{ x }}) %>%
    tidyr::drop_na(data = .) %>%
    as_tibble(.)

  # sample size
  sample_size <- nrow(data)

  # standardize the type of statistics
  stats.type <- ipmisc::stats_type_switch(type)

  # ========================= parametric ====================================

  if (stats.type == "parametric") {
    # deciding which effect size to use (Hedge's g or Cohen's d)
    if (effsize.type %in% c("unbiased", "g")) {
      effsize.text <- quote(widehat(italic("g"))["Hedge"])
      .f <- effectsize::hedges_g
    } else {
      effsize.text <- quote(widehat(italic("d"))["Cohen"])
      .f <- effectsize::cohens_d
    }

    # setting up the t-test model and getting its summary
    stats_df <-
      broomExtra::tidy(stats::t.test(
        x = data %>% dplyr::pull({{ x }}),
        mu = test.value,
        conf.level = conf.level,
        alternative = "two.sided",
        na.action = na.omit
      ))

    # creating effect size info
    effsize_df <-
      rlang::exec(
        .fn = .f,
        x = data %>% dplyr::pull({{ x }}) - test.value,
        correction = FALSE,
        ci = conf.level
      ) %>%
      broomExtra::easystats_to_tidy_names(.)

    # preparing subtitle parameters
    statistic.text <- quote(italic("t")["Student"])
    no.parameters <- 1L
  }

  # ========================== non-parametric ==============================

  if (stats.type == "nonparametric") {
    # setting up the Mann-Whitney U-test and getting its summary
    stats_df <-
      broomExtra::tidy(
        stats::wilcox.test(
          x = data %>% dplyr::pull({{ x }}),
          alternative = "two.sided",
          na.action = na.omit,
          mu = test.value,
          exact = FALSE
        )
      ) %>%
      dplyr::mutate(.data = ., statistic = log(statistic))

    # effect size dataframe
    effsize_df <-
      rcompanion::wilcoxonOneSampleR(
        x = data %>% dplyr::pull({{ x }}),
        mu = test.value,
        ci = TRUE,
        conf = conf.level,
        type = conf.type,
        R = nboot,
        histogram = FALSE,
        digits = k,
        reportIncomplete = TRUE
      ) %>%
      rcompanion_cleaner(.)

    # preparing subtitle parameters
    statistic.text <- quote("log"["e"](italic("V")["Wilcoxon"]))
    no.parameters <- 0L
    effsize.text <- quote(widehat(italic("r")))
  }

  # preparing subtitle
  if (stats.type %in% c("parametric", "nonparametric")) {
    subtitle <-
      expr_template(
        stat.title = stat.title,
        no.parameters = no.parameters,
        stats.df = stats_df,
        effsize.df = effsize_df,
        statistic.text = statistic.text,
        effsize.text = effsize.text,
        n = sample_size,
        n.text = quote(italic("n")["obs"]),
        conf.level = conf.level,
        k = k
      )
  }

  # ======================= robust =========================================

  if (stats.type == "robust") {
    # running one-sample percentile bootstrap
    stats_df <-
      WRS2::onesampb(
        x = data %>% dplyr::pull({{ x }}),
        est = robust.estimator,
        nboot = nboot,
        nv = test.value,
        alpha = 1 - conf.level
      )

    # preparing the subtitle
    subtitle <-
      substitute(
        expr = paste(
          italic("M")["robust"],
          " = ",
          estimate,
          ", CI"[conf.level],
          " [",
          LL,
          ", ",
          UL,
          "], ",
          italic("p"),
          " = ",
          p.value,
          ", ",
          italic("n")["obs"],
          " = ",
          n
        ),
        env = list(
          estimate = specify_decimal_p(x = stats_df$estimate[[1]], k = k),
          conf.level = paste(conf.level * 100, "%", sep = ""),
          LL = specify_decimal_p(x = stats_df$ci[[1]], k = k),
          UL = specify_decimal_p(x = stats_df$ci[[2]], k = k),
          p.value = specify_decimal_p(x = stats_df$p.value[[1]], k = k, p.value = TRUE),
          n = sample_size
        )
      )
  }

  # ======================== bayes ============================================

  # running Bayesian one-sample t-test
  if (stats.type == "bayes") {
    subtitle <-
      tidyBF::bf_ttest(
        data = data,
        x = {{ x }},
        test.value = test.value,
        bf.prior = bf.prior,
        output = "h1",
        k = k
      )$expr
  }

  # return the subtitle
  return(subtitle)
}
