% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parfrailty_methods.R
\name{standardize_parfrailty}
\alias{standardize_parfrailty}
\title{Regression standardization in shared frailty gamma-Weibull models}
\usage{
standardize_parfrailty(
  formula,
  data,
  values,
  times,
  clusterid,
  ci_level = 0.95,
  ci_type = "plain",
  contrasts = NULL,
  family = "gaussian",
  reference = NULL,
  transforms = NULL
)
}
\arguments{
\item{formula}{The formula which is used to fit the model for the outcome.}

\item{data}{The data.}

\item{values}{A named list or data.frame specifying the variables and values
at which marginal means of the outcome will be estimated.}

\item{times}{A vector containing the specific values of \eqn{T} at
which to estimate the standardized survival function.}

\item{clusterid}{An optional string containing the name of a cluster identification variable
when data are clustered.}

\item{ci_level}{Coverage probability of confidence intervals.}

\item{ci_type}{A string, indicating the type of confidence intervals.
Either "plain", which gives untransformed intervals, or "log", which gives
log-transformed intervals.}

\item{contrasts}{A vector of contrasts in the following format:
If set to \code{"difference"} or \code{"ratio"}, then \eqn{\psi(x)-\psi(x_0)}
or \eqn{\psi(x) / \psi(x_0)} are constructed, where \eqn{x_0} is a reference
level specified by the \code{reference} argument. Has to be \code{NULL}
if no references are specified.}

\item{family}{The family argument which is used to fit the glm model for the outcome.}

\item{reference}{A vector of reference levels in the following format:
If \code{contrasts} is not \code{NULL}, the desired reference level(s). This
must be a vector or list the same length as \code{contrasts}, and if not named,
it is assumed that the order is as specified in contrasts.}

\item{transforms}{A vector of transforms in the following format:
If set to \code{"log"}, \code{"logit"}, or \code{"odds"}, the standardized
mean \eqn{\theta(x)} is transformed into \eqn{\psi(x)=\log\{\theta(x)\}},
\eqn{\psi(x)=\log[\theta(x)/\{1-\theta(x)\}]}, or
\eqn{\psi(x)=\theta(x)/\{1-\theta(x)\}}, respectively.
If the vector is \code{NULL}, then \eqn{\psi(x)=\theta(x)}.}
}
\value{
An object of class \code{std_surv}.
This is basically a list with components estimates and covariance estimates in \code{res}
Results for transformations, contrasts, references are stored in \code{res_contrasts}.
The output contains estimates for contrasts and confidence intervals for all
combinations of transforms and reference levels.
Obtain numeric results in a data frame with the \link{tidy} function.
}
\description{
\code{standardize_parfrailty} performs regression standardization in shared frailty
gamma-Weibull models, at specified values of the exposure, over the sample
covariate distribution. Let \eqn{T}, \eqn{X}, and \eqn{Z} be the survival
outcome, the exposure, and a vector of covariates, respectively.
\code{standardize_parfrailty} fits a parametric frailty model to
estimate the standardized survival function
\eqn{\theta(t,x)=E\{S(t|X=x,Z)\}}, where \eqn{t} is a specific value of
\eqn{T}, \eqn{x} is a specific value of \eqn{X}, and the expectation is over
the marginal distribution of \eqn{Z}.
}
\details{
\code{standardize_parfrailty} fits a shared frailty gamma-Weibull model
\deqn{\lambda(t_{ij}|X_{ij},Z_{ij})=\lambda(t_{ij};\alpha,\eta)U_iexp\{h(X_{ij},Z_{ij};\beta)\}}
, with parameterization as described in the help section for
\link{parfrailty}. Integrating out the gamma frailty gives the survival
function
\deqn{S(t|X,Z)=[1+\phi\Lambda_0(t;\alpha,\eta)\exp\{h(X,Z;\beta)\}]^{-1/\phi},}
where \eqn{\Lambda_0(t;\alpha,\eta)} is the cumulative baseline hazard
\deqn{(t/\alpha)^{\eta}.} The ML estimates of \eqn{(\alpha,\eta,\phi,\beta)}
are used to obtain estimates of the survival function \eqn{S(t|X=x,Z)}:
\deqn{\hat{S}(t|X=x,Z)=[1+\hat{\phi}\Lambda_0(t;\hat{\alpha},\hat{\eta})\exp\{h(X,Z;\hat{\beta})\}]^{-1/\hat{\phi}}.}
For each \eqn{t} in the \code{t} argument and for each \eqn{x} in the
\code{x} argument, these estimates are averaged across all subjects (i.e.
all observed values of \eqn{Z}) to produce estimates
\deqn{\hat{\theta}(t,x)=\sum_{i=1}^n \hat{S}(t|X=x,Z_i)/n.} The variance for
\eqn{\hat{\theta}(t,x)} is obtained by the sandwich formula.
}
\note{
Standardized survival functions are sometimes referred to as (direct)
adjusted survival functions in the literature.

\code{standardize_coxph/standardize_parfrailty} does not currently handle time-varying exposures or
covariates.

\code{standardize_coxph/standardize_parfrailty} internally loops over all values in the \code{t} argument.
Therefore, the function will usually be considerably faster if
\code{length(t)} is small.

The variance calculation performed by \code{standardize_coxph} does not condition on
the observed covariates \eqn{\bar{Z}=(Z_1,...,Z_n)}. To see how this
matters, note that
\deqn{var\{\hat{\theta}(t,x)\}=E[var\{\hat{\theta}(t,x)|\bar{Z}\}]+var[E\{\hat{\theta}(t,x)|\bar{Z}\}].}
The usual parameter \eqn{\beta} in a Cox proportional hazards model does not
depend on \eqn{\bar{Z}}. Thus, \eqn{E(\hat{\beta}|\bar{Z})} is independent
of \eqn{\bar{Z}} as well (since \eqn{E(\hat{\beta}|\bar{Z})=\beta}), so that
the term \eqn{var[E\{\hat{\beta}|\bar{Z}\}]} in the corresponding variance
decomposition for \eqn{var(\hat{\beta})} becomes equal to 0. However,
\eqn{\theta(t,x)} depends on \eqn{\bar{Z}} through the average over the
sample distribution for \eqn{Z}, and thus the term
\eqn{var[E\{\hat{\theta}(t,x)|\bar{Z}\}]} is not 0, unless one conditions on
\eqn{\bar{Z}}. The variance calculation by Gail and Byar (1986) ignores this
term, and thus effectively conditions on \eqn{\bar{Z}}.
}
\examples{



require(survival)

# simulate data
set.seed(6)
n <- 300
m <- 3
alpha <- 1.5
eta <- 1
phi <- 0.5
beta <- 1
id <- rep(1:n, each = m)
U <- rep(rgamma(n, shape = 1 / phi, scale = phi), each = m)
X <- rnorm(n * m)
# reparameterize scale as in rweibull function
weibull.scale <- alpha / (U * exp(beta * X))^(1 / eta)
T <- rweibull(n * m, shape = eta, scale = weibull.scale)

# right censoring
C <- runif(n * m, 0, 10)
D <- as.numeric(T < C)
T <- pmin(T, C)

# strong left-truncation
L <- runif(n * m, 0, 2)
incl <- T > L
incl <- ave(x = incl, id, FUN = sum) == m
dd <- data.frame(L, T, D, X, id)
dd <- dd[incl, ]

fit.std <- standardize_parfrailty(
  formula = Surv(L, T, D) ~ X,
  data = dd,
  values = list(X = seq(-1, 1, 0.5)),
  times = 1:5,
  clusterid = "id"
)
print(fit.std)
plot(fit.std)

}
\references{
Chang I.M., Gelman G., Pagano M. (1982). Corrected group prognostic curves
and summary statistics. \emph{Journal of Chronic Diseases} \bold{35},
669-674.

Dahlqwist E., Pawitan Y., Sjölander A. (2019). Regression standardization
and attributable fraction estimation with between-within frailty models for
clustered survival data. \emph{Statistical Methods in Medical Research}
\bold{28}(2), 462-485.

Gail M.H. and Byar D.P. (1986). Variance calculations for direct adjusted
survival curves, with applications to testing for no treatment effect.
\emph{Biometrical Journal} \bold{28}(5), 587-599.

Makuch R.W. (1982). Adjusted survival curve estimation using covariates.
\emph{Journal of Chronic Diseases} \bold{35}, 437-443.
}
\author{
Arvid Sjölander
}
