% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spm_projection.R
\name{spm_projection}
\alias{spm_projection}
\title{A data projection with previously estimated or user-defined parameters. 
Projections are constructed for a cohort with fixed or 
normally distributed initial covariates.}
\usage{
spm_projection(x, N = 100, ystart = 80, model = "discrete", tstart = 30,
  tend = 105, dt = 1, sd0 = 1, nobs = NULL, gomp = TRUE,
  format = "short")
}
\arguments{
\item{x}{A list of parameters from output of the \code{spm(...)} function.}

\item{N}{A number of individuals to simulate, N=100 by default.}

\item{ystart}{A vector of starting values of covariates (variables), ystart=80 by default.}

\item{model}{A model type. Choices are: "discrete", "continuous" or "time-dependent".}

\item{tstart}{Start time (age), default=30. Can be an interval: c(a, b) - in this case,
the starting time is sumulated via \code{runif(1, a, b)}.}

\item{tend}{End time (age), default=105.}

\item{dt}{A time interval between observations, dt=1 by default.}

\item{sd0}{A standard deviation value for simulation of the next value of variable.
sd0=1 by default.}

\item{nobs}{A number of observations (lines) for i-th individual.}

\item{gomp}{A flag (FALSE by default). 
When it is set, then time-dependent exponential form of mu0 and Q are used:
mu0 = mu0*exp(theta*t), Q = Q*exp(theta*t). 
Only for continous-time SPM.}

\item{format}{Data format: "short" (default), "long".}
}
\value{
An object of 'spm.projection' class with two elements. 
(1) A simulated data set.
(2) A summary statistics which includes (i) age-specific means of state variables and
(ii) Survival probabilities.
}
\description{
A data projection with previously estimated or user-defined parameters. 
Projections are constructed for a cohort with fixed or 
normally distributed initial covariates.
}
\examples{
\dontrun{ 
library(stpm)
# Setting up the model
model.par <- list()
model.par$a <- matrix(c(-0.05, 1e-3, 2e-3, -0.05), nrow=2, ncol=2, byrow=TRUE)
model.par$f1 <- matrix(c(90, 35), nrow=1, ncol=2)
model.par$Q <- matrix(c(1e-8, 1e-9, 1e-9, 1e-8), nrow=2, ncol=2, byrow=TRUE)
model.par$f <- matrix(c(80, 27), nrow=1, ncol=2)
model.par$b <- matrix(c(6, 2), nrow=2, ncol=2)
model.par$mu0 <- 1e-6
model.par$theta <- 0.09
# Projection
# Discrete-time model
data.proj.discrete <- spm_projection(model.par, N=5000, ystart=c(80, 27))
plot(data.proj.discrete$stat$srv.prob)
# Continuous-time model
data.proj.continuous <- spm_projection(model.par, N=5000, 
ystart=c(80, 27), model="continuous")
plot(data.proj.continuous$stat$srv.prob)
# Time-dependent model
model.par <- list(at = "-0.05", f1t = "80", Qt = "2e-8", 
ft= "80", bt = "5", mu0t = "1e-5*exp(0.11*t)")
data.proj.time_dependent <- spm_projection(model.par, N=500, 
ystart=80, model="time-dependent")
plot(data.proj.time_dependent$stat$srv.prob, xlim = c(30,105))
}
}
\references{
Yashin, A. et al (2007), 
Stochastic model for analysis of longitudinal data on aging 
and mortality. Mathematical Biosciences, 208(2), 538-551.

Akushevich I., Kulminski A. and Manton K. (2005). 
Life tables with covariates: Dynamic model 
for Nonlinear Analysis of Longitudinal Data. 
Mathematical Popu-lation Studies, 12(2), pp.: 51-80.
<DOI: 10.1080/08898480590932296>.

Yashin, A. et al (2007), Health decline, 
aging and mortality: how are they related? 
Biogerontology, 8(3), 291-302.<DOI:10.1007/s10522-006-9073-3>.
}
