% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/opt.R
\name{opt}
\alias{opt}
\title{Optimum Sample Allocation in Stratified Sampling}
\usage{
opt(n, a, m = NULL, M = NULL, M_algorithm = "rna")
}
\arguments{
\item{n}{(\code{number})\cr total sample size. A strictly positive scalar.
If \code{m} is not \code{NULL}, it is then is required that \code{n >= sum(m)}.
If \code{M} is not \code{NULL}, it is then is required that \code{n <= sum(m)}.}

\item{a}{(\code{numeric})\cr population constants \eqn{A_1,\ldots,A_H}. Strictly
positive numbers.}

\item{m}{(\code{numeric} or \code{NULL})\cr lower bounds \eqn{m_1,\ldots,m_H},
optionally imposed on sample sizes in strata. If no lower bounds should be
imposed, then \code{m} must be set to \code{NULL}.}

\item{M}{(\code{numeric} or \code{NULL})\cr upper bounds \eqn{M_1,\ldots,M_H},
optionally imposed on sample sizes in strata. If no upper bounds should be
imposed, then \code{M} must be set to \code{NULL}.}

\item{M_algorithm}{(\code{string})\cr the name of the underlying algorithm to be
used for computing sample allocation under one-sided upper-bounds
constraints.
It must be one of the following: \code{rna} (default), \code{sga}, \code{sgaplus}, \code{coma}.
This parameter is used only in case when \code{m} argument is \code{NULL} and \code{M} is
not \code{NULL} and number of strata \eqn{H > 1} and \code{n < sum(M)}.}
}
\value{
Numeric vector with optimal sample allocations in strata.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

A classical problem in survey methodology in stratified sampling is optimum
sample allocation. This problem is formulated as determination of strata
sample sizes that minimize the variance of the
\emph{stratified \eqn{\pi} estimator} of the population total (or mean) of a
given study variable, under certain constraints on sample sizes in strata.

The \code{opt()} user function solves the following optimum sample allocation
problem, formulated below in the language of mathematical optimization.

Minimize
\deqn{f(x_1,\ldots,x_H) = \sum_{h=1}^H \frac{A^2_h}{x_h}}
subject to
\deqn{\sum_{h=1}^H x_h = n}
\deqn{m_h \leq x_h \leq M_h, \quad h = 1,\ldots,H,}
where \eqn{n > 0,\, A_h > 0,\, m_h > 0,\, M_h > 0,\, h = 1, \ldots ,H}, such
that \eqn{\sum_{h=1}^H m_h \leq n \leq \sum_{h=1}^H M_h} are given numbers.
The minimization is on \eqn{\mathbb R_+^H}.

The inequality constraints are optional and user can choose whether and how
they are to be added to the optimization problem. This is achieved by the
proper use of \code{m} and \code{M} arguments of this function, according to the
following rules:
\itemize{
\item no inequality constraints imposed: both \code{m} and \code{M} must be both set to
\code{NULL} (default).
\item one-sided lower bounds \eqn{m_h,\, h = 1,\ldots,H}, imposed:
lower bounds are specified with \code{m}, while \code{M} is set to \code{NULL}.
\item one-sided upper bounds \eqn{M_h,\, h = 1,\ldots,H}, imposed:
upper bounds are specified with \code{M}, while \code{m} is set to \code{NULL}.
\item box-constraints imposed: lower and upper bounds must be specified with \code{m}
and \code{M}, respectively.
}
}
\details{
The \code{opt()} function makes use of several allocation algorithms, depending
on which of the inequality constraints should be taken into account in the
optimization problem. Each algorithm is implemented in a separate R
function that in general should not be used directly by the end user.
The following is the list with the algorithms that are used along with the
name of the function that implements a given algorithm. See the description
of a specific function to find out more about the corresponding algorithm.
\itemize{
\item one-sided lower-bounds \eqn{m_h,\, h = 1,\ldots,H}:
\itemize{
\item \code{LRNA} - \code{\link[=rna]{rna()}}
}
\item one-sided upper-bounds \eqn{M_h,\, h = 1,\ldots,H}:
\itemize{
\item \code{RNA} - \code{\link[=rna]{rna()}},
\item \code{SGA} - \code{\link[=sga]{sga()}},
\item \code{SGAPLUS} - \code{\link[=sgaplus]{sgaplus()}},
\item \code{COMA} - \code{\link[=coma]{coma()}}
}
\item box constraints \eqn{m_h, M_h,\, h = 1,\ldots,H}:
\itemize{
\item \code{RNABOX} - \code{\link[=rnabox]{rnabox()}}
}
}
}
\note{
If no inequality constraints are added, the allocation is given by the
Neyman allocation as:
\deqn{x_h = A_h \frac{n}{\sum_{i=1}^H A_i}, \quad h = 1,\ldots,H.}
For \emph{stratified \eqn{\pi} estimator} of the population total and for
\emph{stratified simple random sampling without replacement} design, the
parameters of the objective function \eqn{f} are
\deqn{A_h = N_h S_h, \quad h = 1,\ldots,H,}
where \eqn{N_h} is the size of stratum \eqn{h} and \eqn{S_h} denotes
standard deviation of a given study variable in stratum \eqn{h}.
}
\examples{
a <- c(3000, 4000, 5000, 2000)
m <- c(100, 90, 70, 50)
M <- c(300, 400, 200, 90)

# One-sided lower bounds.
opt(n = 340, a = a, m = m)
opt(n = 400, a = a, m = m)
opt(n = 700, a = a, m = m)

# One-sided upper bounds.
opt(n = 190, a = a, M = M)
opt(n = 700, a = a, M = M)

# Box-constraints.
opt(n = 340, a = a, m = m, M = M)
opt(n = 500, a = a, m = m, M = M)
x <- opt(n = 800, a = a, m = m, M = M)
x
var_st(x = x, a = a, a0 = 45000) # Value of the variance for allocation x.

# Execution-time comparisons of different algorithms with `microbenchmark` R package.
\dontrun{
N <- pop969[, "N"]
S <- pop969[, "S"]
a <- N * S
nfrac <- c(0.005, seq(0.05, 0.95, 0.05))
n <- setNames(as.integer(nfrac * sum(N)), nfrac)
lapply(
  n,
  function(ni) {
    microbenchmark::microbenchmark(
      RNA = opt(ni, a, M = N, M_algorithm = "rna"),
      SGA = opt(ni, a, M = N, M_algorithm = "sga"),
      SGAPLUS = opt(ni, a, M = N, M_algorithm = "sgaplus"),
      COMA = opt(ni, a, M = N, M_algorithm = "coma"),
      times = 200,
      unit = "us"
    )
  }
)
}
}
\references{
Wesołowski, J., Wieczorkowski, R., Wójciak, W. (2021).
Optimality of the Recursive Neyman Allocation.
\emph{Journal of Survey Statistics and Methodology}, 10(5), pp. 1263–1275.
\doi{10.1093/jssam/smab018},
\doi{10.48550/arXiv.2105.14486} \cr

Särndal, C.-E., Swensson, B. and Wretman, J. (1992).
\emph{Model Assisted Survey Sampling}, Springer, New York.
}
\seealso{
\code{\link[=optcost]{optcost()}}, \code{\link[=rna]{rna()}}, \code{\link[=sga]{sga()}}, \code{\link[=sgaplus]{sgaplus()}}, \code{\link[=coma]{coma()}}, \code{\link[=rnabox]{rnabox()}}.
}
