% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/publish_DSD_via_WebService.R
\name{publish_DSD_via_WebService}
\alias{publish_DSD_via_WebService}
\title{Publish a Data Stream via a Web Service}
\usage{
publish_DSD_via_WebService(
  dsd,
  port,
  task_file = NULL,
  serializer = "csv",
  serve = TRUE,
  background = TRUE,
  debug = FALSE
)
}
\arguments{
\item{dsd}{A character string that creates a DSD.}

\item{port}{port used to serve the DSD.}

\item{task_file}{name of the plumber task script file.}

\item{serializer}{method used to serialize the data. By default \code{csv} (comma separated values)
is used. Other methods are \code{json} and \code{rds} (see \link[plumber:serializers]{plumber::serializer_csv}).}

\item{serve}{if \code{TRUE}, then a task file is written and a server started, otherwise,
only a plumber task file is written.}

\item{background}{logical; start a background process?}

\item{debug}{if \code{TRUE}, then the service is started locally and a web client is started to explore the interface.}
}
\value{
a \link[processx:process]{processx::process} object created with \code{\link[callr:r_bg]{callr::r_bg()}} which runs the plumber server
in the background. The process can be stopped with \code{rp$kill()} or by killing the process
using the operating system with the appropriate PID. \code{rp$get_result()} can
be used to check for errors in the server process (e.g., when it terminates
unexpectedly).
}
\description{
Uses the package plumber to publish a data stream as a web service.
}
\details{
The function writes a plumber task script file and starts the web server to serve
the content of the stream using the endpoints
\itemize{
\item \verb{http://localhost:port/get_points?n=100} and
\item \verb{http://localhost:port/info}.
}

APIs generated using plumber can be easily deployed. See: \href{https://www.rplumber.io/articles/hosting.html}{Hosting}. By setting a \code{task_file} and \code{serve = FALSE} a plumber
task script file is generated that can be deployment.

A convenient reader for stream data over web services is available as \link{DSD_ReadWebService}.
}
\examples{
# find a free port
port <- httpuv::randomPort()
port

# create a background DSD process sending data to the port
rp1 <- publish_DSD_via_WebService("DSD_Gaussians(k = 3, d = 3)", port = port)
rp1

# connect to the port and read manually. See DSD_ReadWebService for
# a more convenient way to connect to the WebService in R.
library("httr")

# we use RETRY to give the server time to spin up
resp <- RETRY("GET", paste0("http://localhost:", port, "/info"))
d <- content(resp, show_col_types = FALSE)
d

# example: Get 100 points and plot them
resp <- GET(paste0("http://localhost:", port, "/get_points?n=100"))
d <- content(resp, show_col_types = FALSE)
head(d)

dsd <- DSD_Memory(d)
dsd
plot(dsd, n = -1)

# end the DSD process. Note: that closing the connection above
# may already kill the process.
rp1$kill()
rp1

# Publish using json

rp2 <- publish_DSD_via_WebService("DSD_Gaussians(k = 3, d = 3)", 
           port = port, serializer = "json")
rp2

# connect to the port and read
# we use RETRY to give the server time to spin up
resp <- RETRY("GET", paste0("http://localhost:", port, "/info"))
content(resp, as = "text")

resp <- GET(paste0("http://localhost:", port, "/get_points?n=5"))
content(resp, as = "text")

# cleanup
rp2$kill()
rp2

# Debug the interface (run the service and start a web interface)
if (interactive())
  publish_DSD_via_WebService("DSD_Gaussians(k = 3, d = 3)", port = port, 
         debug = TRUE)
}
\seealso{
Other WebService: 
\code{\link{DSC_WebService}()},
\code{\link{DSD_ReadWebService}()},
\code{\link{publish_DSC_via_WebService}()}

Other dsd: 
\code{\link{DSD_ReadSocket}()},
\code{\link{DSD_ReadWebService}()},
\code{\link{publish_DSD_via_Socket}()}
}
\concept{WebService}
\concept{dsd}
