% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate.R
\name{simulate_clc}
\alias{simulate_clc}
\title{Simulate closed-loop control using Bayesian updates}
\usage{
simulate_clc(
  pkmod_prior,
  pkmod_true,
  target_vals,
  target_tms,
  obs_tms,
  update_tms,
  type = c("effect", "plasma"),
  custom_alg = NULL,
  resp_bounds = NULL,
  delay = 0,
  seed = NULL,
  verbose = TRUE
)
}
\arguments{
\item{pkmod_prior}{`pkmod` or `poppkmod` object describing a PK/PK-PD model that is used to calculate
TCI infusion rates and is updated as data are simulated and incorporated. Must have an
associated Omega matrix.}

\item{pkmod_true}{`pkmod` or `poppkmod` object describing the patient's "true" response. This model
will be used to simulate observations.}

\item{target_vals}{A vector of numeric values indicating PK or PD targets for TCI algorithm.}

\item{target_tms}{A vector of numeric values indicating times at which the TCI algorithm should
begin targeting each value.}

\item{obs_tms}{Times at which data values should be simulated from `pkmod_true`.}

\item{update_tms}{Times at which `pkmod_prior` should be updated using all available
simulated observations.}

\item{type}{Type of TCI algorithm to be used. Options are "plasma" and "effect".
Defaults to "effect". Will be overwritten if `custom_alg` is non-null.}

\item{custom_alg}{Custom TCI algorithm to overwrite default plasma- or effect-site targeting.}

\item{resp_bounds}{Optional vector of two values indicating minimum and maximum values possible for the response.}

\item{delay}{Optional numeric value indicating a temporal delay between when observations
are simulated and when they should be made available for updating `pkmod_prior`. For example,
a delay should be set to account for a processing time delay in Bispectral Index measurements
or the time required to measure drug concentrations from collected samples.}

\item{seed}{An integer used to initialize the random number generator.}

\item{verbose}{Logical. Print progress as simulation is run.}
}
\description{
Simulate closed-loop control using Bayesian updates for `pkmod` or `poppkmod` objects.
Infusion rates are calculated using `pkmod_prior` to reach `target_vals` at
`target_tms`. Data values are simulated using `pkmod_true` at `obs_tms`.
`pkmod_prior` and `pkmod_true` do not need to have the same structure. Model
parameters are updated at each update time using all available simulated observations.
Processing delays can be added through the `delay` argument, such that observations
aren't made available to the update mechanism until `update_tms >= obs_tms + delay`.
}
\examples{
data <- data.frame(ID = 1:3, AGE = c(20,30,40), TBW = c(60,70,80),
HGT = c(150,160,170), MALE = c(TRUE,FALSE,TRUE))
pkmod_prior <- poppkmod(data, drug = "ppf", model = "eleveld")
pkmod_true  <- poppkmod(data, drug = "ppf", model = "eleveld", sample = TRUE)
obs_tms <- seq(1/6,10,1/6)
update_tms <- c(2,4,6,8)
target_vals = c(75,60,50,50)
target_tms = c(0,3,6,10)
\dontrun{
sim <- simulate_clc(pkmod_prior, pkmod_true, target_vals, target_tms, obs_tms,
update_tms, seed = 200)
len <- 500
tms <- seq(0,10,length.out = len)
resp <- data.frame(rbind(predict(pkmod_true, sim$inf, tms),
predict(pkmod_prior, sim$inf, tms),
predict(sim$pkmod_post, sim$inf, tms)))
resp$type = c(rep("true",len*3),rep("prior",len*3),rep("posterior",len*3))
library(ggplot2)
ggplot(resp) + geom_line(aes(x = time, y = pdresp, color = factor(id))) +
facet_wrap(~type) + labs(x = "Hours", y = "Bispectral Index") +
  geom_step(data = data.frame(time = target_tms, value = target_vals),
  aes(x = time, y = value), inherit.aes = FALSE)
}
}
