% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/decorate_grob.R
\name{decorate_grob}
\alias{decorate_grob}
\title{Add Titles, Footnotes, Page Number, and a Bounding Box to a Grid Grob}
\usage{
decorate_grob(
  grob,
  titles,
  footnotes,
  page = "",
  width_titles = grid::unit(1, "npc") - grid::unit(1.5, "cm"),
  width_footnotes = grid::unit(1, "npc") - grid::unit(1.5, "cm"),
  border = TRUE,
  margins = grid::unit(c(1, 0, 1, 0), "lines"),
  padding = grid::unit(rep(1, 4), "lines"),
  outer_margins = grid::unit(c(2, 1.5, 3, 1.5), "cm"),
  gp_titles = grid::gpar(),
  gp_footnotes = grid::gpar(fontsize = 8),
  name = NULL,
  gp = grid::gpar(),
  vp = NULL
)
}
\arguments{
\item{grob}{a grid grob object, optionally \code{NULL} if only a \code{grob} with the decoration should be shown.}

\item{titles}{vector of character strings. Vector elements are separated by a newline and strings are wrapped
according to the page width.}

\item{footnotes}{vector of character string. Same rules as for \code{titles}.}

\item{page}{string with page numeration, if \code{NULL} then no page number is displayed.}

\item{width_titles}{unit object}

\item{width_footnotes}{unit object}

\item{border}{boolean, whether a a border should be drawn around the plot or not.}

\item{margins}{unit object of length 4}

\item{padding}{unit object of length 4}

\item{outer_margins}{unit object of length 4}

\item{gp_titles}{a \code{gpar} object}

\item{gp_footnotes}{a \code{gpar} object}

\item{name}{a character identifier for the grob.  Used to find the
    grob on the display list and/or as a child of another grob. }

\item{gp}{A \code{"gpar"} object, typically the output
    from a call to the function \code{\link[grid]{gpar}}.  This is basically
    a list of graphical parameter settings.}

\item{vp}{a \code{\link[grid]{viewport}} object (or \code{NULL}).}
}
\value{
A grid grob (\code{gTree}).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

This function is useful to label grid grobs (also \code{ggplot2}, and \code{lattice} plots)
with title, footnote, and page numbers.
}
\details{
The titles and footnotes will be ragged, i.e. each title will be wrapped individually.
}
\examples{
library(grid)

titles <- c(
  "Edgar Anderson's Iris Data",
  paste(
    "This famous (Fisher's or Anderson's) iris data set gives the measurements",
    "in centimeters of the variables sepal length and width and petal length",
    "and width, respectively, for 50 flowers from each of 3 species of iris."
  )
)

footnotes <- c(
  "The species are Iris setosa, versicolor, and virginica.",
  paste(
    "iris is a data frame with 150 cases (rows) and 5 variables (columns) named",
    "Sepal.Length, Sepal.Width, Petal.Length, Petal.Width, and Species."
  )
)

## empty plot
grid.newpage()

grid.draw(
  decorate_grob(
    NULL,
    titles = titles,
    footnotes = footnotes,
    page = "Page 4 of 10"
  )
)

# grid
p <- gTree(
  children = gList(
    rectGrob(),
    xaxisGrob(),
    yaxisGrob(),
    textGrob("Sepal.Length", y = unit(-4, "lines")),
    textGrob("Petal.Length", x = unit(-3.5, "lines"), rot = 90),
    pointsGrob(iris$Sepal.Length, iris$Petal.Length, gp = gpar(col = iris$Species), pch = 16)
  ),
  vp = vpStack(plotViewport(), dataViewport(xData = iris$Sepal.Length, yData = iris$Petal.Length))
)
grid.newpage()
grid.draw(p)

grid.newpage()
grid.draw(
  decorate_grob(
    grob = p,
    titles = titles,
    footnotes = footnotes,
    page = "Page 6 of 129"
  )
)

## with ggplot2
library(ggplot2)

p_gg <- ggplot2::ggplot(iris, aes(Sepal.Length, Sepal.Width, col = Species)) +
  ggplot2::geom_point()
p_gg
p <- ggplotGrob(p_gg)
grid.newpage()
grid.draw(
  decorate_grob(
    grob = p,
    titles = titles,
    footnotes = footnotes,
    page = "Page 6 of 129"
  )
)

## with lattice
library(lattice)

xyplot(Sepal.Length ~ Petal.Length, data = iris, col = iris$Species)
p <- grid.grab()
grid.newpage()
grid.draw(
  decorate_grob(
    grob = p,
    titles = titles,
    footnotes = footnotes,
    page = "Page 6 of 129"
  )
)

# with gridExtra - no borders
library(gridExtra)
grid.newpage()
grid.draw(
  decorate_grob(
    tableGrob(
      head(mtcars)
    ),
    titles = "title",
    footnotes = "footnote",
    border = FALSE
  )
)

}
