# Normal usage of 3D plots


#' 3D Scatter Plot
#' @param x,y,z numeric vectors with the same length \code{n}.
#' @param size size for each point.
#' @param col color vector/matrix, can be either numeric or factor.
#'            Its length (vector) or nrow (matrix) must be either \code{n} or 1.
#' @param label text label of each observation.
#' @param group categorical group names of each points.
#' @param timestamp numeric vector, length of 0 or \code{ncol(col)}.
#' @param pal color palette, vector of colors, can be integers.
#' @param scale 'auto', \code{NULL}, or numeric, rescale the final coordinates.
#'              Default 1, no re-scale.
#' @param axis logical, draw axis.
#' @param show_legend logical, show legend.
#' @param control_panel logical, show sidebar (control panel).
#' @param control_presets if control_panel is true, which widgets to show.
#' @param camera_pos initial camera position, auto assign if missing.
#' @param ... other arguments passing to \code{threejs_brain}.
#' @examples
#'
#' #' Continuous color example:
#'
#' data("iris")
#' three_scatter(x = iris$Sepal.Length, y = iris$Sepal.Width,
#'               z = iris$Petal.Length, size = 0.1,
#'               col = iris$Petal.Width, group = iris$Species,
#'               pal = c('orange', 'blue3', 'darkgreen'),
#'               start_zoom = 12, axis = FALSE)
#'
#'
#' #' Discrete example:
#'
#' three_scatter(x = rnorm(26, c(20, 50, -20)), y = rnorm(26, c(30, -60, 30)),
#'               z = rnorm(26, c(10, 40, -40)), size = 1,
#'               col = sample(letters[1:3], 20, TRUE),
#'               pal = c('orange', 'blue3', 'darkgreen'))
#'
#' @export
three_scatter <- function(
  x, y, z, size = 1, col = 1, label = NULL, group = 1, timestamp = NULL, pal = NULL,
  scale = 1, axis = TRUE, show_legend = TRUE, control_panel = TRUE,
  control_presets = c("animation"), camera_pos, ...
){
  maxl = max(length(x),length(y),length(z))
  rec = function(d, max_len = maxl){
    if(length(d) == 0){
      return(seq_len(max_len))
    }
    if(is.matrix(d)){
      if(nrow(d) > max_len){
        return(d[seq_len(max_len),,drop=FALSE])
      }
      nrep = ceiling(max_len / nrow(d))
      if(nrep > 1){
        d = apply(d, 2, function(dd){
          rep(dd, nrep)[seq_len(max_len)]
        })
      }
    }else{
      if(length(d) > max_len){
        return(d[seq_len(max_len)])
      }
      nrep = ceiling(max_len / length(d))
      if(nrep > 1){
        d = rep(d, nrep)[seq_len(max_len)]
      }
    }

    d

  }

  if(length(scale)){
    scale = scale[[1]]
    if(scale == 'auto'){
      scale = 50 / max(abs(range(x,y,z)))
    }
  }else{
    scale = 1
  }

  x = rec(x)
  y = rec(y)
  z = rec(z)
  size = rec(size)
  label = rec(label)
  group = rec(as.character(group))

  groups = sapply(sort(unique(group)), function(gname){
    GeomGroup$new(name = gname)
  }, USE.NAMES = TRUE, simplify = FALSE)


  if(!is.numeric(col)){
    col_type = 'discrete'
    col = as.factor(col)
    col_names = levels(col)
    col = as.numeric(col)
    pal = rec(pal, length(col_names))
  }else{
    col_type = 'continuous'
    col_names = NULL
    if(length(pal) == 0){
      pal = 1:2
    }
    stopifnot2(length(pal) >= 2, msg = 'Continuous mode, pal must have length >= 2.')
  }

  col = as.matrix(rec(col))
  timestamp = rec(timestamp, ncol(col))


  rg = range(x,y,z, na.rm = TRUE)

  geoms = lapply(seq_len(maxl), function(ii){
    if(length(groups) > 1){
      nm = sprintf('%s (%s)', label[ii], group[[ii]])
    }else{
      nm = label[ii]
    }
    g = SphereGeom$new(name = nm, position = c(x[ii], y[ii], z[ii]) * scale,
                       radius = size[ii], group = groups[[group[[ii]]]])
    g$set_value(value = col[ii,], time_stamp = timestamp)

    if(scale != 1){
      g$custom_info = sprintf('Rescale: %.2f x', 1/scale)
    }


    g
  })

  time_range = range(timestamp)
  if(time_range[2] == time_range[1]){
    time_range[2] = time_range[1] + 1
  }else{
    steps = (time_range[2] - time_range[1]) / length(timestamp)
    time_range[2] = time_range[2] + steps
  }

  value_range = range(col)
  if(col_type == 'continuous' && value_range[1] == value_range[2]){
    value_range[1] = -value_range[1]
  }

  camera_center = c(mean(range(x)),mean(range(y)),mean(range(z))) * scale

  span = c(max(x), max(y), max(z))
  span = sqrt(max(sum(span^2), sum((span - camera_center)^2)))

  if(isTRUE(axis)){
    coords = c(span, span, span)
  }else{
    coords = NULL
  }


  if(missing(camera_pos)){
    camera_pos = camera_center + c(0,0,2*span)
  }


  threejs_brain(.list = geoms, time_range = time_range, value_range = value_range,
                control_panel = control_panel, control_presets = control_presets,
                color_ramp = pal, color_type = col_type, color_names = col_names,
                camera_center = camera_center, show_legend = show_legend,
                coords = coords, camera_pos = camera_pos* scale,
                ...)

}



