% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coersion-tk_tbl.R
\name{tk_tbl}
\alias{tk_tbl}
\title{Coerce time-series objects to tibble.}
\usage{
tk_tbl(
  data,
  preserve_index = TRUE,
  rename_index = "index",
  timetk_idx = FALSE,
  silent = FALSE,
  ...
)
}
\arguments{
\item{data}{A time-series object.}

\item{preserve_index}{Attempts to preserve a time series index. Default is \code{TRUE}.}

\item{rename_index}{Enables the index column to be renamed.}

\item{timetk_idx}{Used to return a date / datetime index for
regularized objects that contain a timetk "index" attribute.
Refer to \code{\link[=tk_index]{tk_index()}} for more information on returning index information
from regularized timeseries objects (i.e. \code{ts}).}

\item{silent}{Used to toggle printing of messages and warnings.}

\item{...}{Additional parameters passed to the \code{\link[tibble:as_tibble]{tibble::as_tibble()}} function.}
}
\value{
Returns a \code{tibble} object.
}
\description{
Coerce time-series objects to tibble.
}
\details{
\code{tk_tbl} is designed
to coerce time series objects (e.g. \code{xts}, \code{zoo}, \code{ts}, \code{timeSeries}, etc)
to \code{tibble} objects. The main advantage is that the function keeps the
date / date-time information from the underlying time-series object.

When \code{preserve_index = TRUE} is specified, a new column,
\code{index}, is created during object coercion, and the function attempts to preserve
the date or date-time information. The date / date-time column name
can be changed using the \code{rename_index} argument.

The \code{timetk_idx} argument is applicable when coercing \code{ts} objects that were
created using \code{tk_ts()} from an object that had a time base
(e.g. \code{tbl}, \code{xts}, \code{zoo}).
Setting \code{timetk_idx = TRUE} enables returning the timetk "index" attribute if present,
which is the original (non-regularized) time-based index.
}
\examples{
library(tibble)
library(dplyr)
library(timetk)

data_tbl <- tibble(
    date = seq.Date(from = as.Date("2010-01-01"), by = 1, length.out = 5),
    x    = seq(100, 120, by = 5)
)


### ts to tibble: Comparison between as.data.frame() and tk_tbl()
data_ts <- tk_ts(data_tbl, start = c(2010,1), freq = 365)

# No index
as.data.frame(data_ts)

# Defualt index returned is regularized numeric index
tk_tbl(data_ts)

# Original date index returned (Only possible if original data has time-based index)
tk_tbl(data_ts, timetk_idx = TRUE)


### xts to tibble: Comparison between as.data.frame() and tk_tbl()
data_xts <- tk_xts(data_tbl)

# Dates are character class stored in row names
as.data.frame(data_xts)

# Dates are appropriate date class and within the data frame
tk_tbl(data_xts)


### zooreg to tibble: Comparison between as.data.frame() and tk_tbl()
data_zooreg <- tk_zooreg(1:8, start = zoo::yearqtr(2000), frequency = 4)

# Dates are character class stored in row names
as.data.frame(data_zooreg)

# Dates are appropriate zoo yearqtr class within the data frame
tk_tbl(data_zooreg)


### zoo to tibble: Comparison between as.data.frame() and tk_tbl()
data_zoo <- zoo::zoo(1:12, zoo::yearmon(2016 + seq(0, 11)/12))

# Dates are character class stored in row names
as.data.frame(data_zoo)

# Dates are appropriate zoo yearmon class within the data frame
tk_tbl(data_zoo)




}
\seealso{
\code{\link[=tk_xts]{tk_xts()}}, \code{\link[=tk_zoo]{tk_zoo()}}, \code{\link[=tk_zooreg]{tk_zooreg()}}, \code{\link[=tk_ts]{tk_ts()}}
}
