% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tt.R
\name{tt}
\alias{tt}
\title{Draw a Tiny Table}
\usage{
tt(
  x,
  digits = get_option("tinytable_tt_digits", default = NULL),
  caption = get_option("tinytable_tt_caption", default = NULL),
  notes = get_option("tinytable_tt_notes", default = NULL),
  width = get_option("tinytable_tt_width", default = NULL),
  theme = get_option("tinytable_tt_theme", default = "default"),
  rownames = get_option("tinytable_tt_rownames", default = FALSE),
  escape = get_option("tinytable_tt_escape", default = FALSE),
  ...
)
}
\arguments{
\item{x}{A data frame or data table to be rendered as a table.}

\item{digits}{Number of significant digits to keep for numeric variables. When \code{digits} is an integer, \code{tt()} calls \code{format_tt(x, digits = digits)} before proceeding to draw the table. Note that this will apply all default argument values of \code{format_tt()}, such as replacing \code{NA} by "". Users who need more control can use the \code{format_tt()} function instead.}

\item{caption}{A string that will be used as the caption of the table. This argument should \emph{not} be used in Quarto or Rmarkdown documents. In that context, please use the appropriate chunk options.}

\item{notes}{Notes to append to the bottom of the table. This argument accepts several different inputs:
\itemize{
\item Single string insert a single note: \code{"blah blah"}
\item Multiple strings insert multiple notes sequentially: \code{list("Hello world", "Foo bar")}
\item A named list inserts a list with the name as superscript: \code{list("a" = list("Hello World"))}
\item A named list with positions inserts markers as superscripts inside table cells: \code{list("a" = list(i = 0:1, j = 2, text = "Hello World"))}
}}

\item{width}{Table or column width.
\itemize{
\item Single numeric value smaller than or equal to 1 determines the full table width, in proportion of line width.
\item Numeric vector of length equal to the number of columns in \code{x} determines the width of each column, in proportion of line width. If the sum of \code{width} exceeds 1, each element is divided by \code{sum(width)}. This makes the table full-width with relative column sizes.
}}

\item{theme}{Function or string.
\itemize{
\item String: bootstrap, grid, multipage, placement, revealjs, resize, rotate, spacing, striped, tabular, void
\item Function: Applied to the \code{tinytable} object.
}}

\item{rownames}{Logical. If \code{TRUE}, rownames are included as the first column}

\item{escape}{Logical. If \code{TRUE}, escape special characters in the table. Equivalent to \code{format_tt(tt(x), escape = TRUE)}.}

\item{...}{Additional arguments are ignored}
}
\value{
An object of class \code{tt} representing the table.

The table object has S4 slots which hold information about the structure of the table. Relying on or modifying the contents of these slots is strongly discouraged. Their names and contents could change at any time, and the \code{tinytable} developers do not consider changes to the internal structure of the output object to be a "breaking  change" for versioning or changelog purposes.
}
\description{
The \code{tt} function renders a table in different formats with various styling options: HTML, Markdown, LaTeX, Word, PDF, PNG, or Typst. The table can be customized with additional functions:
\itemize{
\item \code{style_tt()}: style fonts, colors, alignment, etc.
\item \code{format_tt()}: format numbers, dates, strings, etc.
\item \code{group_tt()}: row or column group labels.
\item \code{theme_tt()}: apply a collection of transformations to a \code{tinytable.}
\item \code{save_tt()}: save the table to a file or return the table as a string.
\item \code{print()}: print to a specific format, ex: \code{print(x, "latex")}
}

\code{tinytable} attempts to determine the appropriate way to print the table based on interactive use, RStudio availability, and output format in RMarkdown or Quarto documents. Users can call \code{print(x, output="markdown")} to print the table in a specific format. Alternatively, they can set a global option: \code{options("tinytable_print_output"="markdown")}
}
\section{LaTeX preamble}{


\code{tinytable} uses the \code{tabularray} package from your LaTeX distribution to draw tables. \code{tabularray}, in turn, uses the special \code{tblr}, \code{talltblr}, and \code{longtblr} environments.

When rendering a document from Quarto or Rmarkdown directly to PDF, \code{tinytable} will populate the LaTeX preamble automatically with all the required packages. For standalone LaTeX documents, these commands should be inserted in the preamble manually:

Note: Your document will fail to compile to PDF in Quarto if you enable caching and you use tinytable due to missing LaTeX headers. To avoid this problem, set the option \verb{#| cache: false} for the chunk(s) where you use tinytable.

\if{html}{\out{<div class="sourceCode latex">}}\preformatted{\\usepackage\{tabularray\}
\\usepackage\{float\}
\\usepackage\{graphicx\}
\\usepackage\{rotating\}
\\usepackage[normalem]\{ulem\}
\\UseTblrLibrary\{booktabs\}
\\UseTblrLibrary\{siunitx\}
\newcommand{\tinytableTabularrayUnderline}[1]\{\\underline\{#1\}\}
\newcommand{\tinytableTabularrayStrikeout}[1]\{\\sout\{#1\}\}
\\NewTableCommand\{\\tinytableDefineColor\}[3]\{\\definecolor\{#1\}\{#2\}\{#3\}\}
}\if{html}{\out{</div>}}
}

\section{Global options}{


Many global options can be used to set the default argument values of \code{tinytable} functions.
For a full list, see:

https://vincentarelbundock.github.io/tinytable/vignettes/options.html
\subsection{Quarto}{
\subsection{Figure environment}{
\itemize{
\item \code{options("tinytable_quarto_figure" = FALSE)}: Typst only. Normally, it is best to allow Quarto to define the figure environment, so the default behavior is to not include one.
\item \code{options(tinytable_print_rstudio_notebook = "inline")}: Display tables "inline" or in the "viewer" in RStudio notebooks.
}
}

\subsection{Data Processing}{

The \code{format_tt(quarto=TRUE)} argument activates Quarto data processing for specific cells. This funcationality comes with  a few warnings:
\enumerate{
\item Currently, Quarto provides a \verb{\\QuartoMarkdownBase64\{\}} LaTeX macro, but it does not appear to do anything with it. References and markdown codes may not be processed as expected in LaTeX.
\item Quarto data processing can enter in conflict with \code{tinytable} styling or formatting options. See below for how to disable it.
}

\code{options(tinytable_quarto_disable_processing = TRUE)}

Disable Quarto processing of cell content. Setting this global option to \code{FALSE} may lead to conflicts with some \code{tinytable} features, but it also allows use of markdown and Quarto-specific code in table cells, such as cross-references.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- data.frame(Math = "x^2^", Citation = "@Lovelace1842")
fn <- function(z) sprintf("<span data-qmd='\%s'></span>", z)
tt(x) |> format_tt(i = 1, fn = fn)
}\if{html}{\out{</div>}}

See this link for more details: https://quarto.org/docs/authoring/tables.html#disabling-quarto-table-processing
}

}

\subsection{HTML}{
\itemize{
\item \code{options(tinytable_html_mathjax = TRUE)}
\itemize{
\item insert MathJax scripts in the HTML document. Warning: This may conflict with other elements of the page if MathJax is otherwise loaded.
}
\item \code{options(tinytable_html_portable = TRUE)}
\itemize{
\item \code{plot_tt()} inserts base 64 encoded images directly in the HTML file rather than use external links.
}
}
}

\subsection{PDF}{
\itemize{
\item \code{options(tinytable_pdf_clean = TRUE)}
\itemize{
\item deletes temporary and log files.
}
\item \code{options(tinytable_pdf_engine = "xelatex")}
\itemize{
\item "xelatex", "pdflatex", "lualatex"
}
}
}
}

\examples{
library(tinytable)
x <- mtcars[1:4, 1:5]

tt(x)

tt(x,
   theme = "striped",
   width = 0.5,
   caption = "Data about cars.")

tt(x, notes = "Hello World!")

fn <- list(i = 0:1, j = 2, text = "Hello World!")
tab <- tt(x, notes = list("*" = fn))
print(tab, "latex")

k <- data.frame(x = c(0.000123456789, 12.4356789))
tt(k, digits=2)

}
