% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smooth_map.R
\name{smooth_map}
\alias{smooth_map}
\title{Create a smooth map in various formats: smooth raster, contour lines, and polygons.}
\usage{
smooth_map(shp, var = NULL, nrow = NA, ncol = NA, N = 250000,
  unit = "km", unit.size = 1000, smooth.raster = TRUE, nlevels = 5,
  style = ifelse(is.null(breaks), "pretty", "fixed"), breaks = NULL,
  bandwidth = NA, cover.type = NA, cover = NULL, cover.threshold = 0.6,
  weight = 1, extracting.method = "full", buffer.width = NA,
  to.Raster = FALSE)
}
\arguments{
\item{shp}{shape object of class \code{\link[sp:Spatial]{Spatial}}, \code{\link[raster:Raster-class]{Raster}}, or \code{sf}. Spatial points, polygons, and grids are supported. Spatial lines are not.}

\item{var}{variable name. Not needed for \code{\link[sp:SpatialPoints]{SpatialPoints}}. If missing, the first variable name is taken.}

\item{nrow}{number of rows in the raster that is used to smooth the shape object. Only applicable if shp is not a \code{\link[sp:SpatialGridDataFrame]{SpatialGrid(DataFrame)}} or \code{\link[raster:Raster-class]{Raster}}}

\item{ncol}{number of rows in the raster that is used to smooth the shape object. Only applicable if shp is not a \code{\link[sp:SpatialGridDataFrame]{SpatialGrid(DataFrame)}} or \code{\link[raster:Raster-class]{Raster}}}

\item{N}{preferred number of points in the raster that is used to smooth the shape object. Only applicable if shp is not a \code{\link[sp:SpatialGridDataFrame]{SpatialGrid(DataFrame)}} or \code{\link[raster:Raster-class]{Raster}}}

\item{unit}{unit specification. Needed when calculating density values. When set to \code{NA}, the densities values are based on the dimensions of the raster (defined by \code{nrow} and \code{ncol}). See also \code{unit.size}.}

\item{unit.size}{size of the unit in terms of coordinate units. The coordinate system of many projections is approximately in meters while thematic maps typically range many kilometers, so by default \code{unit="km"} and \code{unit.size=1000} (meaning 1 kilometer equals 1000 coordinate units).}

\item{smooth.raster}{logical that determines whether 2D kernel density smoothing is applied to the raster shape object. Not applicable when \code{shp} is a \code{\link[sp:SpatialPoints]{SpatialPoints}} object.}

\item{nlevels}{preferred number of levels}

\item{style}{method to cut the color scale: e.g. "fixed", "equal", "pretty", "quantile", or "kmeans". See the details in \code{\link[classInt:classIntervals]{classIntervals}}.}

\item{breaks}{in case \code{style=="fixed"}, breaks should be specified}

\item{bandwidth}{single numeric value or vector of two numeric values that specifiy the bandwidth of the kernal density estimator. By default, it is 1/50th of the shortest side in units (specified with \code{unit.size}).}

\item{cover.type}{character value that specifies the type of raster cover, in other words, how the boundaries are specified. Options: \code{"original"} uses the same boundaries as \code{shp} (default for polygons), \code{"smooth"} calculates a smooth boundary based on the 2D kernal density (determined by \code{\link{smooth_raster_cover}}), \code{"rect"} uses the bounding box of \code{shp} as boundaries (default for spatial points and grids).}

\item{cover}{\code{\link[sp:SpatialPolygons]{SpatialPolygons}} shape that determines the covered area in which the contour lines are placed. If specified, \code{cover.type} is ignored.}

\item{cover.threshold}{numeric value between 0 and 1 that determines which part of the estimated 2D kernal density is returned as cover. Only applicable when \code{cover.type="smooth"}.}

\item{weight}{single number that specifies the weight of a single point. Only applicable if \code{shp} is a \code{\link[sp:SpatialPoints]{SpatialPoints}} object.}

\item{extracting.method}{Method of how coordinates are extracted from the kernel density polygons. Options are: \code{"full"} (default), \code{"grid"}, and \code{"single"}. See details. For the slowest method \code{"full"}, \code{\link[raster:extract]{extract}} is used. For \code{"grid"}, points on a grid layout are selected that intersect with the polygon. For \code{"simple"}, a simple point is generated with \code{\link[rgeos:gPointOnSurface]{gPointOnSurface}}.}

\item{buffer.width}{Buffer width of the iso lines to cut kernel density polygons. Should be small enough to let the polygons touch each other without space in between. However, too low values may cause geometric errors.}

\item{to.Raster}{should the "raster" output (see \code{output}) be a \code{\link[raster:Raster-class]{RasterLayer}}? By default, it is returned as a \code{\link[sp:SpatialGridDataFrame]{SpatialGridDataFrame}}}
}
\value{
List with the following items:
\describe{
\item{\code{"raster"}}{A smooth raster, which is either a \code{\link[sp:SpatialGridDataFrame]{SpatialGridDataFrame}} or a \code{\link[raster:Raster-class]{RasterLayer}} (see \code{to.Raster})}
\item{\code{"iso"}}{Contour lines, which is a \code{\link[sp:SpatialLinesDataFrame]{SpatialLinesDataFrame}} (or an \code{sf} object if \code{shp} is an \code{sf})}
\item{\code{"polygons"}}{Kernel density polygons, which is a \code{\link[sp:SpatialPolygonsDataFrame]{SpatialPolygonsDataFrame}} (or an \code{sf} object if \code{shp} is an \code{sf})}
\item{\code{"bbox"}}{Bounding box of the used raster}
\item{\code{"ncol"}}{Number of rows in the raster}
\item{\code{"nrow"}}{Number of columns in the raster}
}
}
\description{
Create a smooth map from a shape object. A 2D kernel density estimator is applied to the shape, which can be a spatial points, polygons, or raster object. Various format are returned: a smooth raster, contour lines, and polygons. The covered area can be specified, i.e., the area outside of it is extracted from the output.
}
\details{
For the estimation of the 2D kernal density, code is borrowed from \code{\link[KernSmooth:bkde2D]{bkde2D}}. This implemention is slightly different: \code{\link[KernSmooth:bkde2D]{bkde2D}} takes point coordinates and applies linear binning, whereas in this function, the data is already binned, with values 1 if the values of \code{var} are not missing and 0 if values of \code{var} are missing.
}
\examples{
\dontrun{
if (require(tmap)) {
    # set mode to plotting mode
    current.mode <- tmap_mode("plot")

    ####################################
    ## Already smoothed raster
    ####################################
    vol <- raster::raster(t(volcano[, ncol(volcano):1]), xmn=0, xmx=870, ymn=0, ymx=610)
    vol_smooth <- smooth_map(vol, smooth.raster = FALSE, nlevels = 10)

    tm_shape(vol_smooth$polygons) +
    	tm_fill(palette=terrain.colors(11), title="Elevation") +
    	tm_shape(vol_smooth$iso) +
    	tm_iso(col = "black", size = .7, fontcolor="black") +
    	tm_layout("Maunga Whau volcano (Auckland)", title.position=c("left", "bottom"),
    	    inner.margins=0) +
    	tm_legend(width=.13, position=c("right", "top"), bg.color="gray80", frame = TRUE)


    ####################################
    ## Smooth polygons
    ####################################
    data(NLD_muni)

    NLD_muni$population_dens <- calc_densities(NLD_muni, "population")

    qtm(NLD_muni, fill="population_dens")

    NLD_smooth <- smooth_map(NLD_muni, var = "population_dens")

    qtm(NLD_smooth$raster, style="grey")
    qtm(NLD_smooth$polygons, format="NLD")

    ####################################
    ## Smooth points
    ####################################

    # Approximate world population density as spatial points, one for each 1 million people,
    # in the following way. Each metropolitan area of x million people will be represented
    # by x dots. The remaining population per country will be represented by dots that are
    # sampled across the country.
    create_dot_per_1mln_people <- function() {
    	data(World, metro)
    	metro_eck <- set_projection(metro, projection = "eck4")

    	# aggregate metropolitan population per country
    	metro_per_country <- tapply(metro_eck$pop2010, INDEX = list(metro_eck$iso_a3), FUN=sum)
    	metro_per_country_in_World <- metro_per_country[names(metro_per_country) \%in\% World$iso_a3]

    	# assign to World shape
    	World$pop_metro <- 0
    	World$pop_metro[match(names(metro_per_country_in_World), World$iso_a3)] <-
    		metro_per_country_in_World

    	# define population density other than metropolitan areas
    	World$pop_est_dens_non_metro <- (World$pop_est - World$pop_metro) / World$area

    	# generate dots for metropolitan areas (1 dot = 1mln people)
    	metro_dots <- do.call("sbind", lapply(1:length(metro_eck), function(i) {
    		m <- metro_eck[i,]
    		m[rep(1, max(1, m$pop2010 \%/\% 1e6)),]
    	}))

    	# sample population dots from non-metropolitan areas (1 dot = 1mln people)
    	World_pop <- sample_dots(World, vars="pop_est_dens_non_metro", w = 1e6,
    							 npop = 7.3e9 - length(metro_dots)*1e6)

    	# combine
    	sbind(as(World_pop, "SpatialPoints"), as(metro_dots, "SpatialPoints"))
    }

    World_1mln_dots <- create_dot_per_1mln_people()


    # dot map
    tm_shape(World_1mln_dots) + tm_dots()

    # create smooth map
    World_list <- smooth_map(World_1mln_dots, cover = World, weight=1e6)

    # plot smooth raster map
    qtm(World_list$raster, style="grey")

    # plot smooth raster map
    qtm(World, bbox="India") + qtm(World_list$iso)

    # plot kernel density map
    qtm(World_list$polygons, style="grey", format="World")

    ####################################
    ## Smooth raster
    ####################################
    data(land)

    land_smooth <- smooth_map(land, var="trees", cover.type = "smooth")

    qtm(land, raster="trees")
    qtm(land_smooth$raster)
    qtm(land_smooth$polygons, format="World", style="grey")

    # reset current mode
    tmap_mode(current.mode)
}
}

}

