% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visualizations.R
\name{ggPanelplot}
\alias{ggPanelplot}
\title{Create a tmod panel plot using ggplot}
\usage{
ggPanelplot(
  res,
  sgenes = NULL,
  auc_thr = 0.5,
  q_thr = 0.05,
  filter_row_q = 0.01,
  filter_row_auc = 0.65,
  q_cutoff = 1e-12,
  cluster = TRUE,
  id_order = NULL,
  effect_size = "auto",
  colors = c("red", "grey", "blue"),
  label_angle = 45,
  add_ids = TRUE,
  mset = NULL
)
}
\arguments{
\item{res}{a list with tmod results (each element of the list is a data
frame returned by a tmod test function)}

\item{sgenes}{a list with summaries of significantly DE genes by gene set.
Each a element of the list is a matrix returned by tmodDecideTests. If
NULL, the bars on the plot will be monochromatic.}

\item{auc_thr}{gene sets enrichments with AUC (or other effect size) below `auc_thr` will not be shown}

\item{q_thr}{gene sets enrichments with q (adjusted P value) above `q_thr` will not be shown}

\item{filter_row_q}{Gene sets will only be shown if at least in one
contrast q is smaller than `filter_row_q`}

\item{filter_row_auc}{Gene sets will only be shown if at least in one
contrast AUC (or other effect size if specified) is larger than `filter_row_auc`}

\item{q_cutoff}{Any q value below `q_cutoff` will be considered equal to
`q_cutoff`}

\item{cluster}{whether to cluster the IDs by similarity}

\item{id_order}{character vector specifying the order of IDs to be
shown. This needs not to contain all IDs shown, but whatever IDs are in this
vector, they will be shown on top in the given order.}

\item{effect_size}{name of the column which contains the effect sizes;
by default, the name of this column is taken from the "effect_size"
attribute of the first result table.}

\item{colors}{character vector with at least 1 (when sgenes is NULL) or 3
(when sgenes is not NULL) elements}

\item{label_angle}{The angle at which column labels are shown}

\item{add_ids}{add IDs of gene sets to their titles on the plot}

\item{mset}{an object of the type 'tmodGS'. If the option `cluster` is
TRUE, the mset object is used to cluster the gene sets. By default, the
built-in transcription modules are used. See details.}
}
\value{
The object returned is a ggplot2 object which can be further
        modified the usual way.
}
\description{
Create a tmod panel plot using ggplot
}
\details{
Panel plot is a kind of a heatmap. This is the most compact way of
representing the results of several gene set enrichment analyses. Each row of a
panel plot shows the result for one gene set, and each column shows
corresponds to one analysis. For example, if one tests gene set enrichment
for a number of different contrasts, then each contrast will be represented
in a separate column.

Each cell of a panel plot shows both the effect size and the p-value.
The p-value is encoded as transparency: the enrichments with a lower
p-value have stronger colors. The size of the bar corresponds to the effect
size, however it is defined. For example, in case of the tmodCERNOtest,
tmodZtest or tmodUtest it is the area under curve, AUC.

In addition, the bars may also encode information about the number of
up- or down-regulated genes. For this, an object must be created using the
function tmodDecideTests. This object provides  information about which
genes in a particular gene set are regulated in which direction.

The order of the gene sets displayed is, by default, determined by
clustering the gene sets based on their overlaps. For this to work,
ggPanelplot must know about what genes are contained in which gene sets.
This is provided by the parameter `mset`. By default (when mset is NULL)
this is the built-in list of gene sets. If, however, the results of gene
set enrichment come from a different set of gene sets, you need to specify
it with the mset parameter. See Examples.
}
\examples{
## prepare a set of results
data(Egambia)
genes <- Egambia$GENE_SYMBOL
exprs <- Egambia[ , -1:-4 ]
group <- gsub("\\\\..*", "", colnames(exprs))
## test differential expression using limma
design <- cbind(Intercept=rep(1, 30), TB=rep(c(0,1), each= 15))
\dontrun{
library(limma)
fit <- eBayes( lmFit(Egambia[,-c(1:3)], design))
tt <- topTable(fit, coef=2, number=Inf, genelist=Egambia[,1:3] )
res <- tmodCERNOtest(tt$GENE_SYMBOL)
## show the results using a panel plot
ggPanelplot(list(limma=res))
## add information about the significant genes
sgenes <- tmodDecideTests(tt$GENE_SYMBOL, lfc=tt$logFC, pval=tt$adj.P.Val)
names(sgenes) <- "limma"
ggPanelplot(list(limma=res), sgenes=sgenes)
## we will now compare the results of enrichments for different types of
## differential expression tests on the data
res_utest <- apply(exprs, 1, function(x) wilcox.test(x ~ group)$p.value)
res_ttest <- apply(exprs, 1, function(x) t.test(x ~ group)$p.value)
## Calculate the gene set enrichment analysis results for each of the
## different types of tests
res_tmod <- list()
res_tmod$limma <- res
res_tmod$utest <- tmodCERNOtest(genes[order(res_utest)])
res_tmod$ttest <- tmodCERNOtest(genes[order(res_ttest)])
ggPanelplot(res_tmod)
## Using the `mset` parameter
## First, we generate results using a different set of gene sets
data(cell_signatures)
res_cs <- tmodCERNOtest(tt$GENE_SYMBOL, mset=cell_signatures)
## the following will triger a warning that no clustering is possible
## because ggPanelplot doesn't have the information about the gene set
## contents
ggPanelplot(list(res=res_cs))
## if we use the mset parameter, clustering is available
ggPanelplot(list(res=res_cs), mset=cell_signatures)
}
}
\seealso{
[tmodDecideTests()], [tmodPanelPlot()]
}
