## ----mtram-pkgs, echo = FALSE, results = "hide", message = FALSE, warning = FALSE----
set.seed(290875)

pkgs <- c("colorspace", "survival", "lme4", "tram", "gridExtra",
          "lattice", "latticeExtra", "mvtnorm", "ordinalCont", "tramME")
pkgs <- pkgs %in% installed.packages()


## ----mtram-citation, echo = FALSE---------------------------------------------
year <- substr(packageDescription("tram")$Date, 1, 4)
version <- packageDescription("tram")$Version


## ----fail, results = "asis", echo = FALSE-------------------------------------
if (any(!pkgs))
{
    cat(paste("Package(s)", paste(names(pkgs)[!pkgs], collapse = ", "), 
        "not available, stop processing.",
        "\\end{document}\n"))
    knitr::knit_exit()
}


## ----mtram-setup, echo = FALSE, results = "hide", message = FALSE, warning = FALSE----
library("lattice")
library("latticeExtra")
library("gridExtra")
trellis.par.set(list(plot.symbol = list(col=1,pch=20, cex=0.7),
                     box.rectangle = list(col=1),
                     box.umbrella = list(lty=1, col=1),
                     strip.background = list(col = "white")))
ltheme <- canonical.theme(color = FALSE)     ## in-built B&W theme
ltheme$strip.background$col <- "transparent" ## change strip bg
lattice.options(default.theme = ltheme)

knitr::opts_chunk$set(echo = TRUE, results = 'markup', error = FALSE,
                      warning = FALSE, message = FALSE,
                      tidy = FALSE, cache = FALSE, size = "small",
                      fig.width = 6, fig.height = 4, fig.align = "center",
                      out.width = NULL, ###'.6\\linewidth', 
                      out.height = NULL,
                      fig.scap = NA)
knitr::render_sweave()  # use Sweave environments
knitr::set_header(highlight = '')  # do not \usepackage{Sweave}
## R settings
options(prompt = "R> ", continue = "+  ", useFancyQuotes = FALSE)  # JSS style
options(width = 75)

### ecdf plots
myprepanel <- function (x, y, f.value = NULL, ...) 
{
    ans <- prepanel.default.qqmath(x, f.value = f.value, distribution = qunif)
    with(ans, list(xlim = ylim, ylim = c(0, 1), dx = dy, dy = dx))
}


mypanel <- function (x, y, f.value = NULL, type = "s", groups = NULL, qtype = 7, 
    ref = TRUE, ...) 
{
    if (ref) {
        reference.line <- trellis.par.get("reference.line")
        do.call(panel.abline, c(list(h = c(0, 1)), reference.line))
    }
    x <- as.numeric(x)
    distribution <- qunif
    nobs <- sum(!is.na(x))
    if (!is.null(groups)) {
        panel.superpose(x, y = NULL, f.value = f.value, type = type, 
            distribution = distribution, qtype = qtype, groups = groups, 
            panel.groups = panel.ecdfplot, ...)
    }
    else if (nobs) {
        if (is.null(f.value)) {
            panel.xyplot(x = sort(x), y = cumsum(y[order(x)]) / sum(y),
                type = type, ...)
        }
        else {
            p <- if (is.numeric(f.value)) 
                f.value
            else f.value(nobs)
            panel.xyplot(x = quantile(x, p, names = FALSE, type = qtype, 
                na.rm = TRUE), y = distribution(p), type = type, 
                ...)
        }
    }
}
library("colorspace")
col <- diverge_hcl(2, h = c(246, 40), c = 120, l = c(65, 90), alpha = .75)


## ----mtram-vignette, eval = FALSE---------------------------------------------
## install.packages("tram")
## demo("mtram", package = "tram")


## ----mtram-sleep-plot, echo = FALSE-------------------------------------------
library("lme4")
xyplot(Reaction ~ Days | Subject, data = sleepstudy, 
       xlab = "Days of sleep deprivation", ylab = "Average reaction time (in ms)")


## ----mtram-sleep_lmer, cache = TRUE-------------------------------------------
sleep_lmer <- lmer(Reaction ~ Days + (Days | Subject), 
                   data = sleepstudy, REML = FALSE)


## ----mtram-tram---------------------------------------------------------------
library("tram")

## ----mtram-sleep_mtram, cache = TRUE------------------------------------------
sleep_LM <- Lm(Reaction ~ Days, data = sleepstudy)
sleep_LMmer <- mtram(sleep_LM, ~ (Days | Subject), data = sleepstudy)


## ----mtram-sleep_cmp----------------------------------------------------------
logLik(sleep_lmer)
logLik(sleep_LMmer)


## ----mtram-sleep_sd-----------------------------------------------------------
(sdinv <- 1 / summary(sleep_lmer)$sigma)
coef(sleep_LMmer)["Reaction"]


## ----mtram-sleep_beta---------------------------------------------------------
fixef(sleep_lmer) * c(-1, 1) * sdinv
coef(sleep_LMmer)[c("(Intercept)", "Days")]


## ----mtram-sleep_varparm------------------------------------------------------
sleep_lmer@theta
coef(sleep_LMmer)[-(1:3)]


## ----mtram-sleep_varcorr------------------------------------------------------
sleep_LMmer$G * (1 / sdinv)^2
cov2cor(sleep_LMmer$G * (1 / sdinv)^2)
unclass(VarCorr(sleep_lmer))$Subject


## ----mtram-sleep-Surv---------------------------------------------------------
library("survival")
sleepstudy$Reaction_I <- with(sleepstudy, Surv(Reaction - 20, Reaction + 20, 
                                               type = "interval2"))
sleepstudy$Reaction_I[1:5]


## ----mtram-sleep-interval, cache = TRUE---------------------------------------
sleep_LM_I <- Lm(Reaction_I ~ Days, data = sleepstudy)
sleep_LMmer_I <- mtram(sleep_LM_I, ~ (Days | Subject), data = sleepstudy)


## ----mtram-sleep-interval-results---------------------------------------------
logLik(sleep_LMmer_I)
coef(sleep_LMmer_I)
coef(sleep_LMmer)


## ----mtram-sleep_BoxCox, cache = TRUE-----------------------------------------
sleep_BC <- BoxCox(Reaction ~ Days, data = sleepstudy)
sleep_BCmer <- mtram(sleep_BC, ~ (Days | Subject), data = sleepstudy, 
                     Hessian = TRUE)
logLik(sleep_BCmer)


## ----mtram-sleep_BoxCoxPlot, echo = FALSE, fig.height = 5---------------------
tmp <- as.mlt(sleep_BC)
cf <- coef(tmp)
coef(tmp) <- coef(sleep_BCmer)[names(cf)]
plot(tmp, newdata = data.frame(Days = 0), type = "trafo", col = "black",
     xlab = "Average reaction time (in ms)", ylab = expression(h(y)))
rug(sleepstudy$Reaction, col = rgb(.1, .1, .1, .1))


## ----mtram-sleep_marginal, fig.height = 5, fig.width = 7, echo = FALSE--------
days <- 0:9
q <- seq(from = min(sleepstudy$Reaction), to = max(sleepstudy$Reaction), 
         length.out = 100)
meth <- c("Normal linear mixed-effects model", "Non-normal linear transformation model")
ex <- expand.grid(Reaction = q, Days = days,
                  Method = factor(meth, levels = meth, labels = meth))
U <- cbind(1, days)
### Linear
tmp <- as.mlt(sleep_LM)
cf <- coef(tmp)
coef(tmp) <- coef(sleep_LMmer)[names(cf)]
SLM <- tcrossprod(U %*% as(sleep_BCmer$G, "matrix"), U) + diag(nrow(U))
sd <- sqrt(diag(SLM))
h <- predict(tmp, newdata = data.frame(Days = days), q = q, type = "trafo")
prob_LM <- pnorm(t(t(h) / sd ))
### BoxCox
tmp <- as.mlt(sleep_BC)
cf <- coef(tmp)
coef(tmp) <- coef(sleep_BCmer)[names(cf)]
SBC <- tcrossprod(U %*% as(sleep_BCmer$G, "matrix"), U) + diag(nrow(U))
sd <- sqrt(diag(SBC))
h <- predict(tmp, newdata = data.frame(Days = days), q = q, type = "trafo")
prob_BC <- pnorm(t(t(h) / sd ))
ex$prob <- c(prob_LM, prob_BC)
plotfun <- function(prob, data, ...) {
    fm <- as.formula(paste(prob, "~ Reaction | Days"))
    xyplot(fm, data = data, type = "l", 
        panel = function(x, y, subscripts, ...) {
            tmp <- subset(sleepstudy, Days == unique(nd[subscripts, "Days"]))
            mypanel(tmp$Reaction, rep(1, nrow(tmp)), lwd = 3, col = grey)
            panel.xyplot(x, y, subscripts = subscripts, ...)
    }, col = col,  xlab = "Average reaction time (in ms)", 
    ylab = "Marginal distribution function", lwd = 2, groups = Method, ...)
}
grey <- rgb(.75, .75, .75)
nd <- ex
plotfun("prob", ex, layout = c(5, 2), par.settings = simpleTheme(col=c(grey, col), lwd = 3),
  auto.key = list(text = c("Empirical cumulative distribution function", levels(nd$Method)), 
                  points = FALSE, lines = TRUE))


## ----mtram-sleep_corr---------------------------------------------------------
cov2cor(sleep_BCmer$G)


## ----mtram-sleep_vcov---------------------------------------------------------
library("mvtnorm")
VC <- solve(sleep_BCmer$Hessian)
idx <- (nrow(VC) - 2):nrow(VC)
Rcoef <- rmvnorm(1000, mean = coef(sleep_BCmer), sigma = VC)[,idx]
ret <- apply(Rcoef, 1, function(gamma) {
    L <- matrix(c(gamma[1:2], 0, gamma[3]), nrow = 2)
    V <- tcrossprod(L)
    c(diag(V), cov2cor(V)[1,2])
})


## ----mtram-sleep_ci-----------------------------------------------------------
### variance random intercept
quantile(ret[1,], c(.025, .5, .975))
### variance random slope
quantile(ret[2,], c(.025, .5, .975))
### correlation random intercept / random slope
quantile(ret[3,], c(.025, .5, .975))


## ----mtram-toenail-plot, echo = FALSE, cache = TRUE---------------------------
data("toenail", package = "HSAUR3")
rlev <- levels(toenail$patientID)[xtabs(~ patientID, 
                                        data = toenail) == 1]
toenail <- subset(toenail, !patientID %in% rlev)
toenail$patientID <- toenail$patientID[, drop = TRUE]
layout(matrix(1:2, ncol = 2))
trt <- levels(toenail$treatment)
cdplot(outcome ~ time, data = subset(toenail, treatment == trt[1]),
       main = trt[1], xlab = "Time", ylab = "Toe nail infection")
cdplot(outcome ~ time, data = subset(toenail, treatment == trt[2]),
       main = trt[2], xlab = "Time", ylab = "")


## ----mtram-toenail_glmer_RI, cache = TRUE-------------------------------------
### Laplace
toenail_glmer_RI_1 <- 
    glmer(outcome ~ treatment * time + (1 | patientID),
          data = toenail, family = binomial(link = "probit"), 
          nAGQ = 1)
summary(toenail_glmer_RI_1)
toenail_glmer_RI_1@theta

### Adaptive Gaussian Quadrature
toenail_glmer_RI_2 <- 
    glmer(outcome ~ treatment * time + (1 | patientID),
          data = toenail, family = binomial(link = "probit"), 
          nAGQ = 20)
summary(toenail_glmer_RI_2)
toenail_glmer_RI_2@theta


## ----mtram-glmmsr-------------------------------------------------------------
library("glmmsr")

## ----mtram-toenail_glmmsr_RI, cache = TRUE------------------------------------
toenail_glmm_RI_3 <- 
    glmm(outcome ~ treatment * time + (1 | patientID),
         data = toenail, family = binomial(link = "probit"), 
         method = "SR", control = list(nSL = 3))
summary(toenail_glmm_RI_3)


## ----mtram-toenail_mtram_RI, cache = TRUE-------------------------------------
m <- ctm(as.basis(~ outcome, data = toenail), 
         shifting = ~ treatment * time, 
         data = toenail, todistr = "Normal")
toenail_probit <- mlt(m, data = toenail, 
                      fixed = c("outcomemoderate or severe" = 0))
toenail_mtram_RI <- 
    mtram(toenail_probit, ~ (1 | patientID), 
          data = toenail, Hessian = TRUE)
coef(toenail_mtram_RI)


## ----mtram-toenail-hessian----------------------------------------------------
vcov(toenail_glmer_RI_2)
solve(toenail_mtram_RI$Hessian)[1:4, 1:4]


## ----mtram-toenail_glmer_RS, cache = TRUE-------------------------------------
toenail_glmer_RS <- 
    glmer(outcome ~ treatment * time + (1 + time | patientID),
          data = toenail, family = binomial(link = "probit"))
summary(toenail_glmer_RS)
toenail_glmer_RS@theta

## ----mtram-lme4-detach, echo = FALSE------------------------------------------
### coef<- gets overwritten
detach(package:lme4)
#detach(package:tram)
#detach(package:mlt)
#library("tram")


## ----mtram-toenail_glmmsr_RS, cache = TRUE------------------------------------
toenail_glmm_RS_1 <- 
    glmm(outcome ~ treatment * time + (1 + time | patientID),
         data = toenail, family = binomial(link = "probit"), 
         method = "Laplace")
toenail_glmm_RS_1$estim[1:3]
toenail_glmm_RS_1$estim[-(1:3)]


## ----mtram-toenail_mtram_RS, cache = TRUE-------------------------------------
toenail_mtram_RS <- 
    mtram(toenail_probit, ~ (1 + time | patientID), 
          data = toenail)
logLik(toenail_mtram_RS)
coef(toenail_mtram_RS)


## ----mtram-toenail_logit, cache = TRUE----------------------------------------
m <- ctm(as.basis(~ outcome, data = toenail), 
         shifting = ~ treatment * time, 
         data = toenail, todistr = "Logistic")
toenail_logit <- mlt(m, data = toenail, 
                     fixed = c("outcomemoderate or severe" = 0))
toenail_mtram_logit <- mtram(toenail_logit, ~ (1 | patientID), 
                             data = toenail, Hessian = TRUE)


## ----mtram-toenail_marginal_logit_s-------------------------------------------
tmp <- toenail_logit
cf <- coef(tmp)
cf <- cf[names(cf) != "outcomemoderate or severe"]
sdrf <- rev(coef(toenail_mtram_logit))[1]
cf <- coef(toenail_mtram_logit)[names(cf)] / sqrt(sdrf^2 + 1)
cf <- c(cf[1], "outcomemoderate or severe" = 0, cf[-1])
coef(tmp) <- cf
time <- 0:180/10
treatment <- sort(unique(toenail$treatment))
nd <- expand.grid(time = time, treatment = treatment)
nd$prob_logit <- predict(tmp, newdata = nd, type = "distribution")[1,]
nd$odds <- exp(predict(tmp, newdata = nd, type = "trafo")[1,])


## ----mtram-toenail_OR_2, dev = "png", cache = TRUE, echo = FALSE, dpi = 300----
X <- model.matrix(~ treatment * time, data = nd)
rbeta <- rmvnorm(10000, mean = coef(toenail_mtram_logit), 
                 sigma = solve(toenail_mtram_logit$Hessian))
s <- rbeta[,ncol(rbeta)]
rbeta <- rbeta[,-ncol(rbeta)] / sqrt(s^2 + 1)
odds <- exp(X %*% t(rbeta))
OR <- odds[1:length(time),] / odds[-(1:length(time)),]
plot(time, rep(0, length(time)), ylim = range(OR), type = "n", 
     xlab = "Time", ylab = "Odds ratio")
colgrey <- rgb(.1, .1, .1, .01)
out <- apply(OR, 2, function(x) lines(time, x, col = colgrey))
ORest <- nd$odds[1:length(time)] / nd$odds[-(1:length(time))]
lines(time, ORest, col = col[1], lwd = 2)


## ----mtram-toenail_marginal_logit---------------------------------------------
tmp <- toenail_logit
cf <- coef(tmp)
cf <- cf[names(cf) != "outcomemoderate or severe"]
sdrf <- rev(coef(toenail_mtram_logit))[1]
cf <- coef(toenail_mtram_logit)[names(cf)] 
cf <- c(cf[1], "outcomemoderate or severe" = 0, cf[-1])
coef(tmp) <- cf
pr <- predict(tmp, newdata = nd, type = "distribution")[1,]
nd$prob_logit <- pnorm(qnorm(pr) / sdrf)


## ----mtram-toenail_marginal_probit--------------------------------------------
tmp <- toenail_probit
cf <- coef(tmp)
cf <- cf[names(cf) != "outcomemoderate or severe"]
sdrf <- rev(coef(toenail_mtram_RI))[1]
cf <- coef(toenail_mtram_RI)[names(cf)] / sqrt(sdrf^2 + 1)
cf <- c(cf[1], "outcomemoderate or severe" = 0, cf[-1])
coef(tmp) <- cf
nd$prob_probit <- predict(tmp, newdata = nd, type = "distribution")[1,]


## ----mtram-toenail_probplot, echo = FALSE-------------------------------------
# nd2 <- nd[rep(1:nrow(nd), 3),]
# nd2$prob <- c(nd$prob_probit, nd$prob_logit_s, nd$prob_logit)
# lev <- c("Probit (M1) = (M2)", "Logit (M1)", "Logit (M2)")
nd2 <- nd[rep(1:nrow(nd), 2),]
nd2$prob <- c(nd$prob_probit, nd$prob_logit)
lev <- c("Probit", "Logit")
nd2$model <- rep(factor(lev, labels = lev, levels = lev), each = nrow(nd))

xyplot(prob ~ time | model, data = nd2, group = treatment, ylim = c(0, 1), 
       xlab = "Time", 
       par.settings = simpleTheme(col = col),
       auto.key = list(text = levels(nd2$treatment), 
                       points = FALSE, lines = TRUE), 
       col = col, type = "l", ylab = "Probability (none or mild)")


## ----toenail-comparisons, cache = TRUE, echo = FALSE, results = "hide"--------
t1 <- system.time(toenail_glmer_RI_1 <- 
    glmer(outcome ~ treatment * time + (1 | patientID),
          data = toenail, family = binomial(link = "probit"), 
          nAGQ = 1))

t2 <- system.time(toenail_glmer_RI_2 <- 
    glmer(outcome ~ treatment * time + (1 | patientID),
          data = toenail, family = binomial(link = "probit"), 
          nAGQ = 20))

t3 <- system.time(toenail_glmm_RI_3 <- 
    glmm(outcome ~ treatment * time + (1 | patientID),
         data = toenail, family = binomial(link = "probit"), 
         method = "SR", control = list(nSL = 3)))

library("hglm")
a <-  hglm(fixed = outcome ~ treatment * time, random = ~ 1 | patientID,
         data = toenail, family = binomial(link = "probit"))

m <- ctm(as.basis(~ outcome, data = toenail), 
         shifting = ~ treatment * time, 
         data = toenail, todistr = "Normal")
toenail_probit <- mlt(m, data = toenail, 
                      fixed = c("outcomemoderate or severe" = 0))
t4 <- system.time(toenail_mtram_RI <- 
    mtram(toenail_probit, ~ (1 | patientID), 
           data = toenail, Hessian = TRUE))

t5 <- system.time(toenail_glmer_RS <- 
    glmer(outcome ~ treatment * time + (1 + time | patientID),
          data = toenail, family = binomial(link = "probit")))

t6 <- system.time(toenail_glmm_RS_1 <- 
    glmm(outcome ~ treatment * time + (1 + time | patientID),
         data = toenail, family = binomial(link = "probit"), 
         method = "Laplace"))

t7 <- system.time(toenail_mtram_RS <- 
    mtram(toenail_probit, ~ (1 + time | patientID), 
           data = toenail))

## ----output, echo = FALSE------------------------------------------------
tn_RI_glmer_L <- c(fixef(toenail_glmer_RI_1), toenail_glmer_RI_1@theta, 0, 0)
tn_RI_glmer_A <- c(fixef(toenail_glmer_RI_2), toenail_glmer_RI_2@theta, 0, 0)
tn_RI_glmmsr <- c(toenail_glmm_RI_3$estim[-1], toenail_glmm_RI_3$estim[1], 0, 0)
tn_RI_mlt <- c(-coef(toenail_mtram_RI), 0, 0)
tn_RI_mlt["gamma1"] <- -tn_RI_mlt["gamma1"]
tn_RS_glmer <- c(fixef(toenail_glmer_RS), toenail_glmer_RS@theta)
tn_RS_glmmsr <- c(toenail_glmm_RS_1$estim[-(1:3)], toenail_glmm_RS_1$estim[1:3])
tn_RS_mlt <- coef(toenail_mtram_RS) * rep(c(-1, 1), c(4, 3))
tn <- cbind(tn_RI_glmer_L, tn_RI_glmer_A , tn_RI_glmmsr, tn_RI_mlt ,
            tn_RS_glmer, tn_RS_glmmsr, tn_RS_mlt)
ll <- c(
### logLik of transformation model for glmer (Laplace) parameters
logLik(toenail_mtram_RI, c(-fixef(toenail_glmer_RI_1), 
                            toenail_glmer_RI_1@theta)),
### logLik of transformation model for glmer (AGQ) parameters
logLik(toenail_mtram_RI, c(-fixef(toenail_glmer_RI_2), 
                            toenail_glmer_RI_2@theta)),
### logLik of transformation model for glmmsr (SR) parameters
logLik(toenail_mtram_RI, parm = c(-toenail_glmm_RI_3$estim[-1], 
                                   toenail_glmm_RI_3$estim[1])),
### logLik of transformation model
logLik(toenail_mtram_RI),
### logLik of transformation model for glmer (Laplace) parameters
logLik(toenail_mtram_RS, c(-fixef(toenail_glmer_RS), 
                            toenail_glmer_RS@theta)),
### logLik of transformation model for glmmsr (Laplace) parameters
logLik(toenail_mtram_RS, parm = c(-toenail_glmm_RS_1$estim[-(1:3)], 
                                   toenail_glmm_RS_1$estim[1:3])),
### logLik of transformation model
logLik(toenail_mtram_RS))

tm <- c(t1["user.self"],
        t2["user.self"],
        t3["user.self"],
        t4["user.self"],
        t5["user.self"],
        t6["user.self"],
        t7["user.self"])
tm <- formatC(tm, format = "f", digits = 2, width = 5)

tn <- formatC(tn, format = "f", digits = 2, width = 5)
ll <- formatC(ll, format = "f", digits = 2, width = 6)
tn <- cbind(c("$\\alpha$", "$\\eshiftparm_1$", "$\\eshiftparm_2$", "$\\eshiftparm_3$", "$\\gamma_1$", "$\\gamma_2$", "$\\gamma_3$"), tn)
ret <- c("
\\begin{tabular}{lrrrr|rrr} \\\\ \\hline
& \\multicolumn{4}{c|}{RI} & \\multicolumn{3}{c}{RI + RS} \\\\
& \\texttt{glmer} & \\texttt{glmer} & \\texttt{glmm} &  & \\texttt{glmer} & \\texttt{glmm} & \\\\
& L               & AGQ             & SR & (7) & L & L & (7) \\\\ \\hline")
ret <- c(ret, apply(tn, 1, function(x) c(paste(x, collapse = " & "), "\\\\")))
ret <- c(ret, "\\hline")
ret <- c(ret, 
         paste("LogLik &", paste(ll, collapse = "&"), "\\\\ "), 
         paste("Time (sec)   &", paste(tm, collapse = "&"), "\\\\ \\hline"), 
         "\\end{tabular}")

## ----table, echo = FALSE, results = "asis"------------------------------------
cat(ret, sep = "\n")


## ----mtram-neck_plot, echo = FALSE, fig.height = 4, fig.width = 7-------------
data("neck_pain", package = "ordinalCont")
pain_df <- neck_pain
idt <- xtabs(~ id, data = pain_df)
miss <- names(idt)[idt < 3]
pain_df <- subset(pain_df, !id %in% miss)
pain_df$id <- factor(pain_df$id)
levels(pain_df$laser) <- c("Placebo", "Active")
levels(pain_df$time) <- c("Baseline", "7 weeks", "12 weeks")
pain <- rbind(subset(pain_df, laser == levels(pain_df$laser)[1]),
              subset(pain_df, laser == levels(pain_df$laser)[2]))
p1 <- xyplot(vas ~ time | laser, data = pain, 
       groups = id, type = "l", col = rgb(.1, .1, .1, .1),
       lwd = 2, layout = c(2, 1),
       ylab = "Neck pain (on visual analog scale)", xlab = "Examinations")
plot(p1)


## ----mtram-ordinalCont--------------------------------------------------------
library("ordinalCont")

## ----mtram-neck_ocm, cache = TRUE, results = "hide"---------------------------
neck_ocm <- ocm(vas ~ laser * time + (1 | id), data = pain_df, 
                scale = c(0, 1))


## ----mtram-neck_ocm_summary---------------------------------------------------
summary(neck_ocm)


## ----mtram-neck_ocm_ci--------------------------------------------------------
exp(cbind(coef(neck_ocm)[2:6], confint(neck_ocm)[2:6,]))


## ----tramME-neck--------------------------------------------------------------
library("tramME")
neck_ColrME <- ColrME(vas ~ laser * time + (1 | id), data = pain_df, 
                      bounds = c(0, 1), support = c(0, 1))


## ----tramME-neck_ci-----------------------------------------------------------
exp(coef(neck_ColrME))


## ----mtram-neck_Colr, cache = TRUE--------------------------------------------
neck_Colr <- Colr(vas ~ laser * time, data = pain_df, 
                  bounds = c(0, 1), support = c(0, 1),
                  extrapolate = TRUE)
neck_Colrmer <- mtram(neck_Colr, ~ (1 | id), data = pain_df, 
                      Hessian = TRUE)


## ----mtram-neck_Colr_distr, echo = FALSE, fig.height = 4, fig.width = 7-------
nd <- expand.grid(laser = unique(pain_df$laser),
                  time = unique(pain_df$time))
tmp <- as.mlt(neck_Colr)
coef(tmp)[] <- coef(neck_Colrmer)[1:12]
q <- 1:99/100
nd2 <- expand.grid(vas = q, laser = unique(pain_df$laser),
                   time = unique(pain_df$time))
# tmp <- as.mlt(neck_Colr) 
sd <- sqrt(coef(neck_Colrmer)[13]^2 + 1)
prb <- predict(tmp, newdata = nd, type = "distribution", q = q)
nd2$prob <- c(pnorm(qnorm(prb) / sd))
p2 <- xyplot(prob ~ vas | time, data = nd2, groups = laser, type = "l", 
             col = col, 
             layout = c(3, 1),
             xlab = "Neck pain (on visual analog scale)", 
             ylab = "Marginal distribution", 
             par.settings = simpleTheme(col=col),
             auto.key = list(text = levels(nd2$laser), 
                             points = FALSE, lines = TRUE))
plot(p2)

## M1
# neck_Colrmer <- mtram(neck_Colr, ~ (1 | id), data = pain_df, 
#                       Hessian = TRUE, standardise = FALSE)
# logLik(neck_Colrmer)
# 
# nd <- expand.grid(laser = unique(pain_df$laser),
#                   time = unique(pain_df$time))
# q <- 1:99/100
# nd2 <- expand.grid(vas = q, laser = unique(pain_df$laser),
#                    time = unique(pain_df$time))
# tmp <- as.mlt(neck_Colr)
# coef(tmp)[] <- coef(neck_Colrmer)[1:12]
# sd <- sqrt(coef(neck_Colrmer)[13]^2 + 1)
# prb <- predict(tmp, newdata = nd, type = "distribution", q = q)
# nd2$prob <- c(pnorm(qnorm(prb) / sd))
# p2 <- xyplot(prob ~ vas | time, data = nd2, groups = laser, type = "l", 
#              col = col, ylim = c(-0.05, 1.05),
#              layout = c(3, 1),
#              xlab = "Neck pain (on visual analog scale)", 
#              ylab = "Marginal distribution", 
#              par.settings = simpleTheme(col=col),
#              auto.key = list(text = levels(nd2$laser), 
#                              points = FALSE, lines = TRUE))
# plot(p2)


## ----mtram-neck_Colr-CI, echo = TRUE, eval=TRUE-------------------------------
S <- solve(neck_Colrmer$Hessian)
rbeta <- rmvnorm(10000, mean = coef(neck_Colrmer), sigma = S)
s <- rbeta[, ncol(rbeta)]
rbeta <- rbeta[,-ncol(rbeta)] / sqrt(s^2 + 1)
t(apply(rbeta[, 8:12], 2, function(x) {
  quantile(exp(x),prob = c(.025, .5, .975))}))


## ----mtram-neck_Colr-std_beta-------------------------------------------------
beta <- coef(neck_Colrmer)[8:12]
alpha <- coef(neck_Colrmer)[13]
(std_beta <- cbind(beta / sqrt(1 + alpha^2)))


## ----mtram-neck_Colr-ctr_mat--------------------------------------------------
ctr_mat <- matrix(c(1, 0, 0, 0, 0,
                    1, 0, 0, 1, 0,
                    1, 0, 0, 0, 1), nrow = 3, byrow = TRUE)
ctr_mat %*% std_beta


## ----mtram-neck_PImanual, eval=TRUE-------------------------------------------
(ci_emp <- t(apply(ctr_mat %*% t(rbeta[, 8:12]), 1, function(x) {
  quantile(x, prob = c(.025, .5, .975))})))

PI(-ci_emp, link = "logistic")


## ----neck_Colr-PI, echo = TRUE------------------------------------------------
nd <- expand.grid(time = unique(pain_df$time),
                  laser = unique(pain_df$laser))
neck_Colr_marg <- neck_Colr
neck_Colr_marg$coef <- coef(neck_Colrmer)[1:12] / sqrt(coef(neck_Colrmer)[13]^2 + 1)
(neck_Colr_PI <- PI(neck_Colr_marg, newdata = nd[1:3, ], reference = nd[4:6, ],
                    one2one = TRUE, conf.level = .95))[1:3, 1:3]


## ----mtram-CAO, echo = FALSE--------------------------------------------------
dir <- system.file("rda", package = "TH.data")
load(file.path(dir, "Primary_endpoint_data.rda"))


## ----mtram-CAO-plot, cache = TRUE, echo = FALSE-------------------------------
ra <- sort(unique(CAOsurv$randarm))
st <- sort(unique(CAOsurv$strat_t))
sn <- sort(unique(CAOsurv$strat_n))
su <- c(1, 1700)
add <- c(0,  max(CAOsurv$iDFS[, "time2"]) - su[2])
ylim <- c(-.05, 1.05)
tmp <- as.mlt(Coxph(iDFS | 0 + strat_n:strat_t:randarm ~ 1, data = CAOsurv, 
                    support = su, add = add, log_first = TRUE))
nd <- expand.grid(strat_n = sn, strat_t = st, randarm = ra)
q <- mkgrid(tmp, 100)[[1]]
surv <- predict(tmp, newdata = nd, type = "survivor", q = q)
nd <- nd[rep(1:nrow(nd), each = nrow(surv)),]
nd$time <- q
nd$surv <- c(surv)
xyplot(surv ~ time | strat_t + strat_n, data = nd, groups = randarm, 
       type = "l", ylim = c(0, 1), col = col, ylab = "Probability",
       xlab = "Time (in days)",
       par.settings = simpleTheme(col=col),
       auto.key = list(text = levels(nd$randarm), 
                       points = FALSE, lines = TRUE))


## ----mtram-CAO_DFS------------------------------------------------------------
### convert "exact" event dates to interval-censoring (+/- one day)
tmp <- CAOsurv$iDFS
exact <- tmp[,3] == 1 
tmp[exact,2] <- tmp[exact,1] + 2
tmp[exact,1] <- pmax(tmp[exact,1] - 2, 0)
tmp[exact,3] <- 3
CAOsurv$iDFS2 <- tmp


## ----mtram-CAO_SR, cache = TRUE-----------------------------------------------
CAO_SR <- Survreg(iDFS2 ~ randarm, data = CAOsurv)
CAO_SR_mtram <- mtram(CAO_SR, ~ (1 | Block), data = CAOsurv,
                      Hessian = TRUE)
logLik(CAO_SR_mtram)
(cf <- coef(CAO_SR_mtram))
(OR <- exp(-cf["randarm5-FU + Oxaliplatin"] / sqrt(cf["gamma1"]^2 + 1)))


## ----mtram-CAO-CI-------------------------------------------------------------
S <- solve(CAO_SR_mtram$Hessian)
# sqrt(diag(S))
rbeta <- rmvnorm(10000, mean = coef(CAO_SR_mtram), 
                 sigma = S)
s <- rbeta[, ncol(rbeta)]
rbeta <- rbeta[, -ncol(rbeta)] / sqrt(s^2 + 1)
quantile(exp(-rbeta[, ncol(rbeta)]), prob = c(.025, .5, .975))


## ----mtram-CAO_SR_2, cache = TRUE---------------------------------------------
CAO_SR_2 <- Survreg(iDFS2 | 0 + strat_n:strat_t ~ randarm, data = CAOsurv)
CAO_SR_2_mtram <- mtram(CAO_SR_2, ~ (1 | Block), data = CAOsurv,
                        Hessian  = TRUE)
logLik(CAO_SR_2_mtram)
(cf <- coef(CAO_SR_2_mtram))
(OR_2 <- exp(-cf["randarm5-FU + Oxaliplatin"] / sqrt(cf["gamma1"]^2 + 1)))


## ----mtram-CAO-CI-2, echo = FALSE---------------------------------------------
H <- CAO_SR_2_mtram$Hessian
S <- solve(H + .01 * diag(nrow(H)))
rbeta <- rmvnorm(10000, mean = coef(CAO_SR_2_mtram), 
                 sigma = S)
s <- rbeta[, ncol(rbeta)]
rbeta <- rbeta[, -ncol(rbeta)] / sqrt(s^2 + 1)
quantile(exp(-rbeta[, ncol(rbeta)]), prob = c(.025, .5, .975))


## ----mtram-CAO_Cox_2, cache = TRUE--------------------------------------------
CAO_Cox_2 <- Coxph(iDFS2 | 0 + strat_n:strat_t ~ randarm, data = CAOsurv, 
                   support = c(1, 1700), log_first = TRUE, order = 4)
logLik(CAO_Cox_2)
CAO_Cox_2_mtram <- mtram(CAO_Cox_2, ~ (1 | Block), data = CAOsurv, 
                         Hessian = TRUE)
logLik(CAO_Cox_2_mtram)
coef(CAO_Cox_2_mtram)


## ----mtram-CAO-CI-3, echo = FALSE---------------------------------------------
H <- CAO_Cox_2_mtram$Hessian
S <- solve(H + .1 * diag(nrow(H)))
rbeta <- rmvnorm(10000, mean = coef(CAO_Cox_2_mtram), 
                 sigma = S)
s <- rbeta[,ncol(rbeta)]
rbeta <- rbeta[,-ncol(rbeta)] / sqrt(s^2 + 1)
quantile(exp(rbeta[, ncol(rbeta)]), prob = c(.025, .5, .975))


## ----mtram-CAO_PI-------------------------------------------------------------
nd <- CAOsurv[1:2, ]
tmp <- CAO_Cox_2
tmp$coef <- coef(CAO_Cox_2_mtram)[-22] / sqrt(coef(CAO_Cox_2_mtram)[22]^2 + 1)
(CAO_Cox_PI <- PI(tmp, newdata = nd[2, ], reference = nd[1, ],
                  one2one = TRUE, conf.level = .95))[1, ]


## ----mtram-CAO_PI_man---------------------------------------------------------
ci_man <- quantile(-rbeta[, ncol(rbeta)], prob = c(.025, .5, .975))
(CAO_Cox_PIm <- PI(ci_man, link = "minimum extreme value"))


## ----tramME-CAO_SR, cache = TRUE----------------------------------------------
CAO_Cox_2_tramME <- CoxphME(iDFS2 | 0 + strat_n:strat_t ~ randarm + (1 | Block), 
                            data = CAOsurv, log_first = TRUE)


## ----tramME-CAO_SR-hr, cache = TRUE-------------------------------------------
exp(coef(CAO_Cox_2_tramME))
exp(confint(CAO_Cox_2_tramME, parm = "randarm5-FU + Oxaliplatin", 
            estimate = TRUE))


## ----echo=FALSE, eval=FALSE---------------------------------------------------
## sqrt(VarCorr(CAO_Cox_2_tramME)$Block$var)
## coef(CAO_Cox_2_mtram)["gamma1"]


## ----mtram-CAO-coxme, echo = FALSE, eval = FALSE------------------------------
## library("coxme")
## m <- coxme(DFS ~ randarm + (1 | Block), data = CAOsurv)
## summary(m)
## sd <- sqrt(diag(vcov(m)))
## exp(coef(m) + c(-1, 0, 1) * qnorm(.975) * sd)


## ----mtram-CHFLS, echo = FALSE------------------------------------------------
library("TH.data")
load(file.path(path.package(package="TH.data"), "rda", "CHFLS.rda"))

### choose neccessary variables
org <- chfls1[, c("REGION6", "ZJ05", "ZJ06", "A35", "ZJ07", "ZJ16M", "INCRM",
                  "JK01", "JK02", "JK20", "HY04", "HY07", "A02", "AGEGAPM", 
                  "A07M", "A14", "A21", "A22M", "A23", "AX16", "INCAM", "SEXNOW", "ZW04")]

names(org) <- c("Region",
                "Rgender",               ### gender of respondent
                "Rage",                  ### age of respondent
                "RagestartA",            ### age of respondent at beginning of relationship with partner A
                "Redu",                  ### education of respondent
                "RincomeM",              ### rounded monthly income of respondent
                "RincomeComp",           ### inputed monthly income of respondent
                "Rhealth",               ### health condition respondent
                "Rheight",               ### respondent's height
                "Rhappy",                ### respondent's happiness
                "Rmartial",              ### respondent's marital status
                "RhasA",                 ### R has current A partner
                "Agender",               ### gender of partner A
                "RAagegap",              ### age gap
                "RAstartage",            ### age at marriage
                "Aheight",               ### height of partner A
                "Aedu",                  ### education of partner A
                "AincomeM",              ### rounded partner A income
                "AincomeEst",            ### estimated partner A income
                "orgasm",                ### orgasm frequency
                "AincomeComp",           ### imputed partner A income
                "Rsexnow",               ### has sex last year
                "Rhomosexual")           ### R is homosexual

### duration of partnership 
org$RAduration <- org$Rage - org$RagestartA

### code missing values
org$AincomeM[org$AincomeM < 0] <- NA
org$RincomeM[org$RincomeM < 0] <- NA
org$Aheight[org$Aheight < 0] <- NA

olevels <- c("never", "rarely", "sometimes", "often", "always")
orgA <- subset(org, Rgender == "female" & Rhomosexual != "yes" & orgasm %in% olevels)

orgA$orgasm <- ordered(as.character(orgA$orgasm),
        levels = c("never", "rarely", "sometimes", "often", "always"))

orgA$Redu <- factor(as.character(orgA$Redu),
        levels = c("univ/grad", "j col", "up mid", "low mid", "primary", "no school"))
levels(orgA$Redu) <-  c("univ", "jcol", "upmid", "lowmid", "primary", "noschool")

orgA$Aedu <- factor(as.character(orgA$Aedu),
        levels = c("univ/grad", "j col", "up mid", "low mid", "primary", "no school"))

orgA$Rhappy <- factor(as.character(orgA$Rhappy),
        levels = c("v unhappy", "not too", "relatively", "very"))

orgA$Rhealth <- factor(as.character(orgA$Rhealth),
        levels = c("poor", "not good", "fair", "good", "excellent"))

orgA$Region <- factor(as.character(orgA$Region),
        levels = c("CentralW", "Northeast", "North", "InlandS", "CoastalE", "CoastalS"))

orgA$AincomeSD <- orgA$AincomeComp/sd(orgA$AincomeComp)
orgA$AheightSD <- orgA$Aheight/sd(orgA$Aheight)
orgA$RageSD <- orgA$Rage/sd(orgA$Rage)
orgA$edudiff <- as.numeric(orgA$Aedu) - as.numeric(orgA$Redu)
orgA$edudiffSD <- orgA$edudiff/sd(orgA$edudiff, na.rm=TRUE)
orgA$wealthdiff <- orgA$RincomeComp - orgA$AincomeComp
orgA$wealthdiffSD <- orgA$wealthdiff/sd(orgA$wealthdiff, na.rm=TRUE)
orgA$RAdurationSD <- orgA$RAduration/sd(orgA$RAduration, na.rm=TRUE)
orgAtmp <- orgA[, c("orgasm", "AincomeSD", "AheightSD", "RAdurationSD",
                 "RageSD", "edudiffSD", "wealthdiffSD", "Redu", "Rhealth",
                 "Rhappy", "Region")]
cc <- complete.cases(orgAtmp)
orgAcc <- subset(orgA, cc)


## ----mtram-CHFLS-Polr, cache = TRUE-------------------------------------------
CHFLS_Polr <- Polr(orgasm ~ AincomeSD + AheightSD + RAdurationSD + 
                   RageSD + edudiffSD + wealthdiffSD + Redu + 
                   Rhealth + Rhappy + Region, data = orgAcc)
logLik(CHFLS_Polr)
orgAcc$ID <- factor(1:nrow(orgAcc))
CHFLS_mtram <- mtram(CHFLS_Polr, ~ (1 | ID), 
                     data = orgAcc)
logLik(CHFLS_mtram)
coef(CHFLS_mtram)


## ----sim-setup, echo = FALSE, message = FALSE---------------------------------
set.seed(150594)

library("lme4")
library("glmmTMB")
library("tramME")
library("mvtnorm")
library("survival")
library("lattice")
library("latticeExtra")
library("gridExtra")

mytheme <- standard.theme("pdf")

mytheme$box.umbrella <- list(col = "black", lty = 1)
mytheme$box.rectangle <- list(col = "black", lwd = 1)
mytheme$box.dot <- list(col = "black")
mytheme$plot.symbol <- list(col = "black")


Nsim <- 100


## ----sim-BP_data, eval=FALSE--------------------------------------------------
## N <- 100 # number of clusters
## Ni <- 5  # size of clusters
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## x <- cbind(1, matrix(runif(N * Ni * 3), ncol = p))
## prb <- pnorm(x %*% beta + rnorm(N, sd = tau)[cls])
## y <- factor(rbinom(nrow(x), size = 1, prob = prb))
## d <- data.frame(y = y, x[,-1], cls = cls)


## ----sim-BP, echo = FALSE, eval = FALSE---------------------------------------
## N <- 100 # number of clusters
## Ni <- 5  # size of clusters
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## rga <- rgl <- rgm <- rmt <- matrix(NA, ncol = 5, nrow = Nsim)
## 
## for (i in 1:nrow(rgl)) {
## 
##   x <- cbind(1, matrix(runif(N * Ni * 3), ncol = p))
##   prb <- pnorm(x %*% beta + rnorm(N, sd = tau)[cls])
##   y <- factor(rbinom(nrow(x), size = 1, prob = prb))
##   d <- data.frame(y = y, x[,-1], cls = cls)
## 
##   ga <- glmer(y ~ X1 + X2 + X3 + (1 | cls), data = d,
##               family = binomial(link = "probit"), nAGQ = 20)
##   gl <- glmer(y ~ X1 + X2 + X3 + (1 | cls), data = d,
##               family = binomial(link = "probit"))
##   gm <- glmmTMB(y ~ X1 + X2 + X3 + (1 | cls), data = d,
##                 family = binomial(link = "probit"))
##   cm <- ctm(as.basis(~ y, data = d), shifting = ~ X1 + X2 + X3,
##             data = d, todistr = "Normal", negative = TRUE)
##   m <- mlt(cm, data = d, fixed = c("y1" = 0))
##   mt <- mtram(m, ~ (1 | cls), data = d)
## 
##   rga[i, ] <- c(fixef(ga), c(sqrt(VarCorr(ga)$cls)))
##   rgl[i, ] <- c(fixef(gl), c(sqrt(VarCorr(gl)$cls)))
##   rgm[i, ] <- c(fixef(gm)$cond, c(sqrt(VarCorr(gm)$cond$cls)))
##   cf <- coef(mt)
##   cf[1] <- -cf[1]
##   rmt[i, ] <- cf
## 
##   print(i)
## }
## coefs <- data.frame(coef = c(rga, rgl, rgm, rmt),
##                     algo = as.factor(rep(c("AGQ", "glmer", "glmmTMB", "mtram"),
##                                          each = Nsim)),
##                     which = as.factor(rep(rep(c("beta1", "beta2", "beta3",
##                                                 "beta4", "gamma"),
##                                               each = Nsim), times = 4)))


## ----sim-BP_plot, echo = FALSE, eval = TRUE-----------------------------------
tau <- sqrt(3)/pi
beta <- c(-.5, 0, 1, 2)

true_coefs <- c(beta, tau)

coefs <-
structure(list(coef = c(-0.19730739423201743, -0.52303591803377802, 
-0.71371671955558746, -0.67609139108537541, -0.98997176394992059, 
-0.79091706720727895, -0.73883938113813463, -0.57017980209763164, 
-0.93453263302268441, -0.73766230248977405, -0.46380096320957176, 
-0.63170826348028497, -0.5031033915683728, -0.40707686750224625, 
-0.61949988694428004, -0.68384336808774537, -0.4841530479935115, 
-0.69415613501905915, -0.29142166128371033, -0.88791218264830296, 
-0.46500047909031883, -0.45785695366741641, -0.69902346119204051, 
-0.82373977215947924, -0.88921574160183137, -0.42619469891532391, 
-0.65513707636467156, -0.99181590635049344, -0.67357558076305402, 
-0.76385222072440362, -0.84999218754187478, -0.37282595173078942, 
-1.0045345883534864, -0.8303705208624852, -0.50974319206774454, 
-0.62436343089164059, -0.59768088368772498, -0.90053807634908112, 
-0.46751457049545886, -0.5186998634234643, -0.3767995963623007, 
-0.19151416273994626, -0.3743678539390618, -0.13155664648464893, 
-0.29868684814527513, -0.34951177573229686, -0.47003088294068507, 
-0.75416065390197895, -1.3408031435966765, -0.47921289841885917, 
-0.5070322465526188, -0.15944231156054617, -0.50353391367576195, 
-0.60587382327078121, -0.36576313719693687, -0.60845968564115338, 
-0.34350497671634916, -0.67853279498795138, -0.31648427791689038, 
-0.24561669437816233, -0.43408234806555768, -0.4910540051239613, 
-0.45255681213768773, -0.35023712211457192, -0.44855686144678253, 
-0.41586516566587473, -0.47700792279901788, -0.88005607797742857, 
-0.84412301762248143, -0.54740047668635805, -0.48333675414783561, 
-0.55471539693371397, -0.5387405772544791, -0.67594090995847655, 
-0.37549804742870124, -0.71803539445117415, -0.82487036428166172, 
-0.29676556432938972, -0.1889518373155048, -0.78841161130138127, 
-0.2758741780644518, -0.54286251524835838, -0.69579432315590228, 
-0.65486996284161014, -0.29830644349579205, -0.3579889671639106, 
-0.325025456505035, -0.41828541074959091, -0.44100605928210956, 
-1.0122679575516083, -0.59388536062992381, 0.19139826554847827, 
-0.56183362985801899, -0.528672154528392, -0.58792414036563412, 
-0.29335152894347294, -0.31476046204060226, -0.61894117579342478, 
-1.1788092324498345, -0.56060589231669422, -0.020378607764924318, 
-0.10974666305153913, 0.42695457191711783, -0.11116496085838842, 
0.36294735205696915, 0.55787190029067246, 0.055341686265196824, 
0.19312565430316214, 0.31080656276838214, -0.086495612248946321, 
-0.0719771052836609, -0.0049270188312650585, -0.056282686806038595, 
-0.11481805685248095, -0.33368372462985635, 0.2630044656055241, 
0.033402634855439178, 0.28110428953099309, 0.1547610451345886, 
0.43011492275075958, 0.4352850636730794, -0.20000284684764014, 
-0.22651661345689222, 0.16629488039842874, 0.36705553540627051, 
0.15604793180045143, 0.38499986338646874, 0.26088368864133382, 
0.39410859293013345, -0.022619314577626736, 0.010714235259305581, 
-0.10670568590568261, 0.38826424657043523, 0.086550288145939874, 
0.18331193957842384, 0.41979824037876784, -0.11583254061787746, 
0.70344777129086311, -0.15488362516644311, -0.60961593058394892, 
-0.19060293999075928, -0.32399117952250478, -0.14651381905540112, 
-0.16946307053782106, -0.28229726540191824, 0.08306452363275732, 
0.19597259219342589, 0.016682826049403299, 0.22290352065194338, 
-0.34490905212071155, 0.34883446966804049, -0.35004783676257006, 
-0.20997816898259217, 0.31456712135066872, 0.034843355852674035, 
-0.49928945485755549, -0.24173038268062208, -0.093104022866009897, 
-0.17424316978228899, 0.0062734590257765817, -0.11943980492985243, 
0.49930149189838335, 0.21889163619350416, 0.18073259921683005, 
-0.1410788750854815, -0.089599542603614551, -0.16837278042650036, 
0.16234763870896579, 0.08719581820248859, 0.3727427857426342, 
-0.16129198488270974, -0.27724212733118891, 0.25298205988711631, 
0.10261020698214689, -0.22359897726110917, 0.10787213490506603, 
-0.025450480786345722, -0.15458039181431815, -0.29881789718607954, 
0.16022520111723121, 6.8438221494683019e-06, 0.060979731424336936, 
0.34646670653329426, 0.10315666733114801, 0.0081701570696072236, 
0.16703175190298453, 0.097786741499548024, -0.26480620005715949, 
-0.095568262527841147, 0.15441086960660866, 0.25617273925821427, 
-0.44780599954749278, 0.069719019880899485, -0.047907605399203745, 
0.44293510530935737, -0.032828191228527578, -0.1664820931937217, 
0.42078599603721822, 0.80347849712273567, 0.086728381371890054, 
0.35947203541997802, 0.92629557933069107, 0.97251837596341706, 
1.1521250028058023, 1.4361990424355622, 0.82494587925236473, 
1.248630673629167, 1.1067752449805015, 1.0597056519169241, 1.4838313101081329, 
0.56990262756626175, 1.2531556084282438, 1.2107106889568997, 
0.91888415134639179, 1.6490276746710975, 0.62655671869743634, 
0.90482264857924455, 1.1171855030033739, 0.62100953649725721, 
1.2244013347521192, 1.1478917943864229, 1.2063269764572027, 1.1777767458368342, 
1.4097734112670479, 1.4726289965124182, 0.93117954596140318, 
1.3518617170406737, 1.3494763544842894, 0.94161193294464729, 
1.138346342699837, 1.2924095084968987, 1.0562606821937057, 1.1564808904081265, 
1.3983744850383824, 0.85414497164405068, 0.77141244498404227, 
1.0793337819049069, 1.4738663403680787, 1.271760975470317, 1.3721270727371373, 
0.79121012585403339, 1.0072573549882251, 0.78909600424539361, 
0.39629727382821167, 0.98911072054876992, 0.66813721252871405, 
0.98637468190745603, 1.0347912649244, 1.6797928504618627, 0.96506672132619942, 
0.85318838459012714, 0.91930187330785573, 1.3203121883340958, 
0.66274011678464761, 0.89632316902450704, 1.398626939436566, 
0.78275265332926147, 0.88100344244616113, 0.8647061439274919, 
0.69954984613075144, 0.84747009341604795, 0.66826558043527451, 
0.4823303771932626, 0.58138583484489048, 0.79108858543385852, 
1.111270959984261, 1.1538632317911426, 0.90206895677282084, 1.3592530730033727, 
0.88268434840394483, 1.3509348684999338, 0.96680335755721236, 
1.1703873229216761, 1.0500947692956173, 0.81284125921472627, 
1.2840234270640591, 1.3713660487867758, 0.76446755994511784, 
0.92356395627178145, 1.3956107180007771, 0.80402622763685483, 
0.89535019109101055, 0.92030177449726014, 1.1022035368678091, 
0.81287601483148619, 0.67744636490800314, 0.49763331844343306, 
1.2074836531769122, 1.4023935568910018, 1.4859512486381858, 0.88254863346770218, 
0.80942850266432986, 1.1055947051197534, 0.80453096717621564, 
0.68506209131839635, 0.5807649311995895, 1.0219024046367799, 
1.2551254997018422, 1.5827942195057152, 1.2016558093344751, 1.7038825631404315, 
1.9048211190121389, 2.056821237781175, 2.1120857193321192, 2.5065196462264638, 
2.1509745541735512, 2.1031155837678659, 1.7528802690510796, 2.2678508980915586, 
1.938091555546555, 2.1304970343363259, 2.2499901085662506, 2.120541598028177, 
2.2287690523142007, 2.1753729416048517, 2.0735370777146427, 1.9706618396946312, 
1.8757204649202972, 1.8451204205303489, 2.1160698638035229, 1.4916584875754286, 
1.9070409875886747, 2.3258485210808164, 2.5748207822976634, 2.0769021908248035, 
2.1264754780376784, 1.4391468524898823, 2.3676659829423716, 1.8392362693811739, 
2.175465436306883, 2.3894347062649448, 1.8836646621530393, 2.215821649168916, 
1.9558119557337346, 1.8477269662290554, 1.6350476013041313, 2.3490373256395247, 
1.8087236319281543, 1.8905002183155775, 2.3322431179645315, 2.3749020510153578, 
1.8902878298449544, 1.9143649370689424, 2.0359879022201026, 1.6359873853855493, 
1.8929693612967429, 2.0275709438878411, 2.5191802739497433, 3.0369111570560525, 
2.1938215332313726, 2.1031029054026931, 1.6760361119690739, 2.3131244809015961, 
2.249713713671039, 1.9045189842619425, 2.3487520882462749, 2.1680311050751966, 
2.285145398674036, 1.7553300780604473, 2.0026153907918496, 2.2944517585240733, 
1.9304262543244701, 1.9669203526944627, 2.0634143876136672, 2.0538766441698768, 
1.8790432540241273, 1.7452522278822948, 2.8698745877206422, 2.415301644306096, 
1.9112609190608914, 2.0988269047646302, 2.3746266335588047, 1.9078547477995809, 
2.1497516366415139, 2.3033779817288509, 2.2882515585664622, 2.2135847656370702, 
1.8613855465771945, 1.8944536101589815, 1.9996323161347922, 1.8777535302185941, 
2.5334179888773569, 2.1525039236374086, 2.2072817549177595, 1.938701614154243, 
1.8760476789111489, 2.1938625813231578, 2.0281382886395298, 1.6853717103073396, 
2.4455242242313293, 2.0961770167351048, 1.5548366298707381, 1.7450711176389828, 
2.2916575748932653, 1.89941520025581, 1.7387966475836005, 1.6465432809027913, 
2.0979444673631615, 2.0711910929107922, 1.920562373116522, 0.51317373623328355, 
0.33092791935287486, 0.4535975716753643, 0.48157155468129109, 
0.75808346508968638, 0.45014008423947638, 0.82809759748183653, 
0.39918485794184766, 0.61382052023428746, 0.40875744629560989, 
0.49492611342618786, 0.38570280894789793, 0.61535838103540885, 
0.5988778475513552, 0.68812602424406633, 0.44031810500938706, 
0.39533367481594117, 0.56426462904943286, 0.4057109676892916, 
0.67894264609866195, 0.48671467927455625, 0.68560181502015671, 
0.73745266543500287, 0.58910380418558228, 0.70847323300127196, 
1.0022316492970504, 0.2088647263137165, 0.45034257343222373, 
0.55089389926993748, 0.46964019656066558, 0.33478597885933375, 
0, 0.49034833172425063, 0.58511419113089813, 0.58628757031814471, 
0.64970322423890736, 0.57042123822098545, 0.52043519708499852, 
0.44545230133705038, 0.73709130748112006, 0.40939544584264403, 
0.55944412893576168, 0.34794960869977931, 0.53657887381258695, 
0.4792597847909571, 0.63911733148485905, 0.58869261980875731, 
0.76031843051684367, 0.84545172210722419, 0.27775527632272512, 
0.43717379206407414, 0.48253453490454534, 0.72200158351052224, 
0.426519327900729, 0.63912315925127028, 0.82872006129627951, 
0.51402157025086004, 0.39558538940654597, 0.74123079024033789, 
0.47649487569612142, 0.55602050037203576, 0.63943877060055898, 
0.54393401349774395, 0.49273802280828205, 0.53610705656352142, 
0.44687701738983482, 0.46832209190418905, 0.65086945749610225, 
0.62166717864498955, 0.54771156613300165, 0.65211010255876845, 
0.66710883657008069, 0.56018920356433788, 0.57841466443763268, 
0.64735845995233721, 0.64183320811216227, 0.41071733282783573, 
0.63121866790053804, 0.40370620275094715, 0.565661363641355, 
0.51335171441098071, 0.54950829750952312, 0.66243085032271032, 
0.7214454065178546, 0.60898060416551081, 0.45722061273287601, 
0.67303744224313566, 0.34082707078018259, 0.55605716339572675, 
0.53712260616496532, 0.32276809911471294, 0.60070603228607666, 
0.45161274234356252, 0.6110249715923014, 0.60756310789624324, 
0.61757223468312916, 0.49107255908757064, 0.78086377586035605, 
0.6538161824994233, 0.63691942064496221, -0.18815200413793776, 
-0.52206824926423945, -0.71463075239582996, -0.68195901907032996, 
-0.99857618889980093, -0.78039384875606621, -0.74404874098687679, 
-0.57262009714189177, -0.94014822837046519, -0.74139339825879957, 
-0.46525802151279527, -0.63667287526446403, -0.49928926637194299, 
-0.41239137892251754, -0.63703488857229162, -0.68025583814355695, 
-0.48888839277831397, -0.700463788043865, -0.29086354039625606, 
-0.88415522975575933, -0.46004064582412907, -0.45924081745735074, 
-0.69230066600320106, -0.82624549253845347, -0.88940979478092097, 
-0.42164008703121453, -0.65507874996938176, -0.99299269356713327, 
-0.67515234127547652, -0.77132740692837087, -0.8514555656502687, 
-0.37282595173078942, -1.0114589148914241, -0.83314181833169521, 
-0.51105788840905886, -0.63034878386546955, -0.60056561627474003, 
-0.90641084720337206, -0.46321227694458134, -0.5307958859213544, 
-0.37654539814350463, -0.19492046678268599, -0.37513107016415026, 
-0.13765135065072928, -0.30036183290344498, -0.34467543813107832, 
-0.46031739189563325, -0.75791941007578401, -1.3608788364116906, 
-0.47994490429192893, -0.49916271555832914, -0.16130105149402324, 
-0.50651706894734372, -0.60300481648469861, -0.36526736205710464, 
-0.6122919856782536, -0.35407604164265283, -0.67366350052547752, 
-0.32075693411320499, -0.23657550430203705, -0.43404251372499802, 
-0.48670616330736072, -0.45937803316819892, -0.33841546488870877, 
-0.44109398523348758, -0.41480989405083568, -0.48142483705713918, 
-0.86517324198594181, -0.8509677756407148, -0.54014545735069042, 
-0.46941599867529332, -0.55156622059912497, -0.53924532791916024, 
-0.6743120484291949, -0.37907633896267645, -0.71855179667618152, 
-0.8273398828320051, -0.29534467703581235, -0.18485167928892321, 
-0.79523271395312078, -0.27671101485042993, -0.53630697360086232, 
-0.6886971916212925, -0.64459161232147455, -0.30050152331520696, 
-0.36353941835384501, -0.30852353603134874, -0.41724140572931079, 
-0.44346614666788498, -1.0083977969121265, -0.59179339422823585, 
0.19022385231945127, -0.56459921171517369, -0.52614957063321244, 
-0.58848896638271631, -0.30642600905936868, -0.31212888098983177, 
-0.62251021764873304, -1.172585035428511, -0.55791613446047683, 
-0.028874964989746947, -0.10511691207951199, 0.42876432797566055, 
-0.10718944127139091, 0.36543323180371812, 0.55122518821410926, 
0.066940199124764649, 0.20474330381209388, 0.30552141990943582, 
-0.086458512042108374, -0.068954757061266095, -0.00075549464241045047, 
-0.048500868950919471, -0.10681821649392834, -0.32249564445599005, 
0.26325437680733565, 0.035588439403521993, 0.29110176459193621, 
0.1561947182154392, 0.44170393956262111, 0.43758550209445007, 
-0.20174325620694666, -0.23952243004742299, 0.17275675366774268, 
0.3708340604418453, 0.15738006004713362, 0.3847466202579819, 
0.25997560877136311, 0.39396292542070555, -0.014064516436064489, 
0.013881260172227837, -0.10670568590568261, 0.3924056626854604, 
0.082053062269324739, 0.18633325933749031, 0.42122711402716517, 
-0.11297231350623126, 0.69929943036632136, -0.1579357505890629, 
-0.60253736915427314, -0.18548701428488665, -0.31379473985062845, 
-0.14579012593532664, -0.16282018292590617, -0.27843027595732489, 
0.089271411831551614, 0.20027900879034299, 0.0062292708682939709, 
0.22736324456028201, -0.34456583376956251, 0.35207627301780764, 
-0.34767600035589563, -0.20828395953580514, 0.31438885755167273, 
0.049616519696490932, -0.49773761983244169, -0.23386851749607185, 
-0.10097224566326163, -0.16227455404810218, -0.00064567467921536944, 
-0.12254912024308273, 0.49805550808064447, 0.22321515677346065, 
0.1683181216590692, -0.14274652772402588, -0.09944930517523215, 
-0.16165127268935725, 0.16184725375828596, 0.077184868086042149, 
0.36413996906811336, -0.16796431476806917, -0.27710278585166281, 
0.25567237048466962, 0.099978195338498971, -0.22718882931840445, 
0.11511362785521068, -0.02502562664970856, -0.1530802830157954, 
-0.30062023604592059, 0.16475030232716809, -0.00043254983964623704, 
0.057425530607224884, 0.33610037640666335, 0.10616320225750657, 
0.016857654600980512, 0.17343116067502912, 0.08341859616939637, 
-0.26394580146221736, -0.093870389155226858, 0.14555574922996173, 
0.2556833038896934, -0.45104161334830428, 0.079947051262054564, 
-0.044137165057427069, 0.45127351816121658, -0.021328367564022628, 
-0.1706926693405148, 0.42290038609844527, 0.79609885589743323, 
0.084972779210623739, 0.35730129018974832, 0.92127178176791702, 
0.98254394157739089, 1.1575769615554905, 1.4392081774746019, 
0.82420430789101951, 1.253889283744497, 1.1100778679995988, 1.0721990246693098, 
1.4877452140186633, 0.57370283824745616, 1.2582751566218611, 
1.2051978068759226, 0.92349818208927492, 1.6732326141039455, 
0.62133795956288451, 0.90866795969079406, 1.1232876330830173, 
0.6228917602902897, 1.228421822034381, 1.1481416173847796, 1.2182408563616578, 
1.1745666323180493, 1.4094494297761693, 1.4613889298648028, 0.94033006674060293, 
1.3520607264797722, 1.355534430303488, 0.94175744463470967, 1.1335590617228917, 
1.2921371332124596, 1.0562606821937057, 1.1595945620929999, 1.4030330574334786, 
0.85975053869463314, 0.77944395596200367, 1.0885482072954007, 
1.4801321617249728, 1.2715899462912963, 1.3898478895208473, 0.78541885872285833, 
1.0135941740175198, 0.78773390747213634, 0.40294118128144707, 
0.98753359736165147, 0.65936812039464709, 0.97726526265732261, 
1.0541528046740543, 1.7042816438608503, 0.96602539975716351, 
0.84751461460827748, 0.92417376654013306, 1.3212094497252131, 
0.66066411687881776, 0.89170079858178175, 1.4008371165813078, 
0.79889615231818412, 0.87897945770129138, 0.86850444917978209, 
0.69595253569147819, 0.84758889084107625, 0.66959472829879041, 
0.48663353296304568, 0.5768922948212909, 0.78582111218870843, 
1.1107505740610146, 1.1614592510705111, 0.89471707674937517, 
1.3693381650678633, 0.886444387034651, 1.3509142592288863, 0.9592594335594582, 
1.1753313045848013, 1.0480680577819479, 0.82472688057428356, 
1.2813997433836528, 1.3724016103554111, 0.76235579533287212, 
0.92168495190708177, 1.4035384512323108, 0.80652416127964621, 
0.88625812542470617, 0.92109708465452489, 1.0947426758505792, 
0.82074907194525171, 0.67833086481812077, 0.48863643440033366, 
1.2092392901660034, 1.4035926814935773, 1.4828439144655303, 0.8807280588087093, 
0.81352649130771759, 1.1095617962434754, 0.80700697964384238, 
0.68244805561327559, 0.59737958764451293, 1.0247409801978722, 
1.2642822221515706, 1.5895348530186459, 1.1963111442446412, 1.70081824773415, 
1.9041890566461643, 2.0478517387790962, 2.11707667045604, 2.5300272748286918, 
2.1399364978671827, 2.1130870991831836, 1.7441648735577466, 2.2809000615162622, 
1.9472990799943259, 2.1301161375797508, 2.2560284168592446, 2.1133134985364053, 
2.2333419469289995, 2.1912743064359339, 2.0742630109352449, 1.9764964227987447, 
1.8790545701116863, 1.8441404109195749, 2.0996375573336232, 1.4823387730601041, 
1.9070638355975031, 2.3397176151630008, 2.5813153058783422, 2.0885594350971042, 
2.1252468682141781, 1.4399605636404778, 2.3685492072377241, 1.852473051611593, 
2.1888973285301567, 2.3896531880531953, 1.8836646621530393, 2.2291802547421371, 
1.9689167703707713, 1.8482014381203709, 1.6466905096548734, 2.3501253029520566, 
1.824896500817389, 1.8899118473087511, 2.3399702897645458, 2.3750299486432924, 
1.8837899096298332, 1.917623355433917, 2.0433397247393383, 1.6436258020333601, 
1.8914392868727683, 2.0202349624871214, 2.5250775632073714, 3.0715185461011787, 
2.1955596472895134, 2.0919688193537951, 1.6779446654990899, 2.3279746639223062, 
2.2523817639524548, 1.8979116163914576, 2.3608946358623979, 2.1718984995303647, 
2.2885527335685794, 1.7647085454283968, 2.0003559138408464, 2.3029449231951657, 
1.9245085020652208, 1.9789700693225389, 2.0565958786300347, 2.0522585187441313, 
1.8925688572577708, 1.7453922275997713, 2.8490247121796424, 2.4348954064995336, 
1.9135382684850439, 2.0843569957713011, 2.3836935348551553, 1.9113317216211214, 
2.1597462918896473, 2.3147661718875905, 2.29277009186755, 2.2200475164807547, 
1.8690901230428971, 1.8930860084621257, 2.0060244257852951, 1.8802325670809483, 
2.5283436788237568, 2.1511319019798929, 2.1929334177354569, 1.9300731779065456, 
1.8874648267632592, 2.1880986311504862, 2.027922566226803, 1.6950664298962637, 
2.4502628660832961, 2.0964786914515572, 1.5584657136104618, 1.7414754458263624, 
2.2918768024377298, 1.9010684354819598, 1.745695197124288, 1.6507528542605827, 
2.1188678360532336, 2.0607353664872603, 1.9313897797398885, 0.47229763297595251, 
0.29994477628067451, 0.4134664685835393, 0.44240284130816621, 
0.70669381773926943, 0.41364319465708166, 0.77912829681023044, 
0.36407480016087185, 0.57093566572827903, 0.37866265267815524, 
0.45464110137418656, 0.35790391467578336, 0.56493704938319822, 
0.55415447182379751, 0.64436013605607367, 0.40398986778706925, 
0.36268506067583423, 0.52016526323792134, 0.37305316404265321, 
0.62739520888268441, 0.44820196467618179, 0.63606049482757865, 
0.6888536141550905, 0.54517410234220942, 0.65449827837536623, 
0.93769440367660484, 0.19120038986225268, 0.41518979420738378, 
0.51313127408778914, 0.43027176632201514, 0.30487570575579204, 
1.9999999999999999e-07, 0.45475727140270505, 0.54148236470928568, 
0.54281546415980864, 0.6059168913760522, 0.52790453406872873, 
0.48139460256208122, 0.41044482783322545, 0.6843706081283778, 
0.37389855267834893, 0.51529032703392441, 0.31662070841973933, 
0.495395007120984, 0.44236680634815312, 0.59044685387357987, 
0.54364754784713587, 0.70675125096640357, 0.80214824240713023, 
0.25615216229311122, 0.40125694239652737, 0.44517018461887065, 
0.67224413156738061, 0.39334226992811799, 0.58878984787632005, 
0.77177414365116714, 0.47516549459207613, 0.362505950022875, 
0.69415173764496563, 0.4390663135292962, 0.51182198477542251, 
0.58725291663670909, 0.50454384994970281, 0.44984993703499609, 
0.49591505455311358, 0.40938549269656377, 0.43195772940797916, 
0.59869345082725323, 0.57233147231368031, 0.50849078092738687, 
0.60181048776191159, 0.62050890729769692, 0.51947503548074658, 
0.53776279497221524, 0.60551428130155593, 0.59757803185989433, 
0.37734282090568561, 0.58808018644940496, 0.37226203817698111, 
0.52355385928452114, 0.46928837235932486, 0.49973708214473389, 
0.611081088389827, 0.66689499428814225, 0.5602447059889667, 0.4238560244101795, 
0.62031098933174911, 0.31415463379496805, 0.51445589969718919, 
0.49394303199141149, 0.29685926728881035, 0.5487303448842682, 
0.41423094851436315, 0.56892790461849385, 0.5606123952860923, 
0.57495116172085758, 0.45537144375539296, 0.73334280070114222, 
0.59786715278659908, 0.59188177180592427, -0.19601968472967435, 
-0.5227832586586959, -0.71364416054449109, -0.67659358639561384, 
-0.99012259702264926, -0.78962440673643153, -0.7390661903439435, 
-0.57035106733813534, -0.93424566717158586, -0.73788430396240845, 
-0.46374431390209936, -0.63221556507696308, -0.50294265203409427, 
-0.40797464127846761, -0.62238601152862127, -0.68282992286642563, 
-0.48461855857693314, -0.69502506490714111, -0.29167247043859057, 
-0.88662707115411832, -0.46475709428595963, -0.45745995717986576, 
-0.69703578524668619, -0.82338602463499866, -0.88863911092065517, 
-0.4239677604201233, -0.65508201533225952, -0.99192882410956595, 
-0.67323348751321355, -0.76439953680876827, -0.85012774928101731, 
-0.37282798315629612, -1.0046838179976612, -0.83030366926957699, 
-0.50992942868214075, -0.62482874364866525, -0.59799803885919933, 
-0.90114479129448599, -0.46728669579844201, -0.52063090520984268, 
-0.37714856057690072, -0.19259528789548885, -0.37448405467127238, 
-0.13309831967639407, -0.29897085436389981, -0.34880687883471723, 
-0.46935445793472164, -0.75381733278733687, -1.3417718010154951, 
-0.47927765866631034, -0.50692070754531937, -0.16002983567542364, 
-0.50372486734545496, -0.60540070700055859, -0.36622047244515965, 
-0.60844604212326359, -0.34505010063649411, -0.67761637067064318, 
-0.31733441891187497, -0.24515750964795868, -0.43451585849669772, 
-0.49086726327902613, -0.45368874843293505, -0.34900006328419214, 
-0.44728444860491945, -0.41577975835059733, -0.47746690748409343, 
-0.87730985560643893, -0.84457510189838225, -0.54673923077147646, 
-0.48181632359641446, -0.55420489655744365, -0.53897249972679695, 
-0.67579023678155881, -0.37612123925361168, -0.71810187502899769, 
-0.82494853109838884, -0.29660117547441039, -0.18889596042693838, 
-0.78923971181190489, -0.27654933755575423, -0.54238898955443493, 
-0.69452504811868199, -0.65215343699624373, -0.29966014406986152, 
-0.35889798569373499, -0.32308072515512859, -0.41834302296304704, 
-0.44135569558379217, -1.0115927086457366, -0.59389235787581296, 
0.19004810072094003, -0.56206071082360642, -0.52814429292327292, 
-0.5874859710368453, -0.29590057912374063, -0.31432818292531112, 
-0.61993873502716379, -1.1769168457587813, -0.5599599027848734, 
-0.021681166958517656, -0.10927035574581452, 0.42691360427825065, 
-0.11073807770751687, 0.36268399415876096, 0.55672414377060686, 
0.058659800206204732, 0.19443970613386172, 0.30962297119063625, 
-0.086514058128544402, -0.07130163416094748, -0.0047575402267616357, 
-0.05517771465477013, -0.1140175744849518, -0.33209284395515692, 
0.26274766236264901, 0.033625868233746692, 0.28230384615937831, 
0.15494720206741422, 0.43096509031427827, 0.43547542202779616, 
-0.20051526959368068, -0.22848338138036345, 0.1667221410005951, 
0.36697521662455407, 0.15638815760302047, 0.38495768781156947, 
0.26079548654394352, 0.39342274737872407, -0.021584976511902807, 
0.010817819795299104, -0.10670454665959395, 0.38838408013574172, 
0.086079222800653027, 0.18340818031634687, 0.41899410093591488, 
-0.11552606473306157, 0.70294120442744434, -0.15510666891130948, 
-0.60735665783889847, -0.1905408195991323, -0.32281746860734301, 
-0.1463546062950262, -0.16818485424260549, -0.28128608685917728, 
0.083647993534346113, 0.19638415356570019, 0.014921567079441773, 
0.22250770566718908, -0.34487177442309125, 0.34920939830868469, 
-0.34941945563486676, -0.20957658411685701, 0.31425189145466942, 
0.037367694415496581, -0.49768639270229909, -0.24047767569585493, 
-0.093907925575512918, -0.17067081475112728, 0.005448012761509761, 
-0.11985414452319322, 0.49872253335998507, 0.21927089895506674, 
0.17903427449127235, -0.14129585229960065, -0.090638077784211846, 
-0.16706422389385941, 0.16209519941018596, 0.085983890416622202, 
0.37130924660495912, -0.16164762902700233, -0.27671427387144532, 
0.25299850389626649, 0.10257339261407836, -0.22403727930853154, 
0.10863181886192487, -0.025415796047545548, -0.15383020563951139, 
-0.29902616836669271, 0.16065758711628075, 0.00011183309249001896, 
0.060609599662857339, 0.34400631264099069, 0.1033104868857902, 
0.0098301862393843746, 0.16754523671929999, 0.095542140362170777, 
-0.26472653161331355, -0.095447347153992376, 0.1534828514972236, 
0.25615767339891343, -0.44821218996192524, 0.071031555132554311, 
-0.047302167768922584, 0.44347861888202444, -0.030649509831392866, 
-0.16702380527187247, 0.4207896616472922, 0.80119410606875863, 
0.086045218803284682, 0.35849150999308438, 0.92529542802461695, 
0.97298448254192726, 1.1520809800027771, 1.4353569002172326, 
0.82444760932665118, 1.2465846820584985, 1.1064560739736797, 
1.0599605104004324, 1.4836665014874975, 0.56960213218185851, 
1.2538421080378983, 1.2097170672276827, 0.91912001291693113, 
1.6531273051119439, 0.62508373258169636, 0.9048573835813768, 
1.1171833067563424, 0.62093930008195064, 1.2235520905903208, 
1.1473863347577911, 1.2062914933161206, 1.1752072573628678, 1.4085708675277591, 
1.4692735683265103, 0.9305336949384041, 1.3517118276369673, 1.3497677449966703, 
0.94022275568859792, 1.1368556616721566, 1.2924800696918641, 
1.0562615321648983, 1.1558271470671746, 1.3974769892396002, 0.85387408610295479, 
0.77141882917439386, 1.0797027896629978, 1.4743908412268818, 
1.2713720112272151, 1.3738587167525562, 0.79161839144761215, 
1.0079860815063928, 0.78860061991635266, 0.39697322506571237, 
0.98754207557278828, 0.66545049958422564, 0.98481812002807423, 
1.0360901150222781, 1.6811075393008876, 0.96507692527923683, 
0.8528909700232048, 0.91914696811680563, 1.3186832798806356, 
0.6620906794275262, 0.89437460166086247, 1.3961860489237685, 
0.78422693346985606, 0.8800926038820025, 0.8623318311628605, 
0.69916194646355778, 0.84741918089927915, 0.66785181772598112, 
0.48257766402392588, 0.58038944441472229, 0.78930041474049728, 
1.1104926218469215, 1.1536507482213405, 0.89983185773238927, 
1.3591368006502396, 0.88258778702948237, 1.3502653675443885, 
0.96419483434247633, 1.1706121231310886, 1.0489161618207721, 
0.81397744838064923, 1.2832596947690182, 1.371039354382956, 0.76255966524095886, 
0.92348707257605522, 1.3960388404919934, 0.80407735195579133, 
0.89443777441213157, 0.91974550777201081, 1.0984389664754151, 
0.81378379317254523, 0.67731061561558459, 0.49547932423190477, 
1.2075604739520815, 1.4013094516585498, 1.4849759684746746, 0.88248890579509864, 
0.80969098947126827, 1.1052004151828296, 0.80380395413141414, 
0.6832621537936151, 0.58288940015358892, 1.0209050594094702, 
1.2568765027043158, 1.5823327568272836, 1.1989535719162627, 1.7008085308317504, 
1.9039315745307497, 2.0551568443901593, 2.1117581816360502, 2.5085629259890077, 
2.1490508805956612, 2.1000921899992995, 1.7508929680633716, 2.2669640684597607, 
1.9385050160482489, 2.1280953626053751, 2.2513825315061196, 2.1194463327371169, 
2.2291355084600051, 2.177953694441916, 2.071731254274856, 1.9707465200091281, 
1.8744621139618491, 1.8447080755549083, 2.1117749358701632, 1.4899747261122556, 
1.903206528514144, 2.3254495378465103, 2.5752338338895258, 2.0772543708361852, 
2.1177219786053403, 1.4390124774933031, 2.3678333212939666, 1.8390473147588986, 
2.1759040419722333, 2.3897778439517894, 1.8836675638870126, 2.2158751333117617, 
1.9556106657100714, 1.8463132776028977, 1.6335206211631292, 2.3487709112114215, 
1.8103596807580487, 1.8902610872580872, 2.3315597414458757, 2.3768651251758559, 
1.8894351163852687, 1.9140206566167945, 2.0354676593323981, 1.6352500317816063, 
1.8911412682526876, 2.0267665602096727, 2.5179166087413529, 3.043501794585195, 
2.1939253895030584, 2.1029983521395126, 1.6750918256925629, 2.3143391072504462, 
2.249074527811127, 1.9019334373215242, 2.3484112871324587, 2.1675771253422016, 
2.2841246750828765, 1.7517172661507341, 2.0029704096772907, 2.2965959184066591, 
1.9288752407409213, 1.9666231920154689, 2.0619560896128086, 2.0517293851489495, 
1.8800396989390444, 1.7434137792713011, 2.8646392819720163, 2.4163591502760595, 
1.9110937792323961, 2.0957321820282622, 2.3747697244316752, 1.9077538965377652, 
2.1502992550480027, 2.3036040822376007, 2.2883447019158409, 2.2137332177973086, 
1.8599077621142903, 1.8947624460434964, 2.0004472519859564, 1.8782296407079642, 
2.5342570207001258, 2.1510621943533454, 2.2007827965613478, 1.9373289365538073, 
1.8770714551198906, 2.1928400910681227, 2.0281706386491072, 1.6857599026780177, 
2.445395247063312, 2.0964262702072642, 1.5548720587533331, 1.7431049740976849, 
2.289584610319245, 1.8969096299327874, 1.7367009500791009, 1.6455984609105041, 
2.1028359177330058, 2.0678885425365952, 1.9206542683704042, 0.50426597647516525, 
0.32729967167738433, 0.44880448815647034, 0.47626123090112882, 
0.74775584793934369, 0.44565983734629072, 0.81198296573147033, 
0.39463732422672765, 0.60530953324109582, 0.40575503055159173, 
0.48781236609217876, 0.38521352905504169, 0.60751184169397077, 
0.59154531213365813, 0.68067000942662714, 0.43451684135388075, 
0.39201914652218334, 0.55555923236723592, 0.40181135994033274, 
0.66864108940778821, 0.48098174469891652, 0.67289912591217149, 
0.72573357227358049, 0.58386131158036392, 0.69677603990654369, 
0.97389868056221196, 0.20783113198403766, 0.44735992223493026, 
0.54315478720459665, 0.46422439019168632, 0.33377920210417317, 
0.00016671241644863145, 0.4851953658404759, 0.5766397455343153, 
0.57733676711316262, 0.63697972728550123, 0.56406379184427902, 
0.51607724876783312, 0.44151072198553387, 0.72571553679681589, 
0.40920196098747119, 0.5523236564954257, 0.34439592201849822, 
0.52860464492891079, 0.47217535316185938, 0.62809629124729405, 
0.58132113390856421, 0.74847921862762812, 0.8370849020148361, 
0.27693315887813869, 0.43429435126207933, 0.47622845182664314, 
0.71133269367136998, 0.42237068149734558, 0.62798661667698097, 
0.81401601818913727, 0.50811849638742668, 0.39144816255314241, 
0.72489951409143938, 0.47246517662267074, 0.55085240685818693, 
0.62892378761969048, 0.53565517055810552, 0.4866060285364367, 
0.52892316492028268, 0.4421779230484984, 0.46191476950504357, 
0.64201517215488568, 0.61293571498028643, 0.5411757133258509, 
0.64316681011662358, 0.65797648380366691, 0.55362960503393366, 
0.57181538047443226, 0.63905412961149288, 0.63427414736744836, 
0.407679636978723, 0.62110555618829333, 0.40137780603372697, 
0.55994650403092872, 0.5073529219506806, 0.54518237365070876, 
0.65201808614047785, 0.70821890533505727, 0.60001094827513779, 
0.45266043259248989, 0.66234781980451707, 0.33917591080472809, 
0.54865295733729924, 0.53196495264214194, 0.32163790635628448, 
0.59007512519069205, 0.44548481743752039, 0.60319512274230502, 
0.59695522108837529, 0.60648176134846044, 0.48430951288059493, 
0.76993446891776174, 0.64329068474102546, 0.62700728649307025, 
-0.19708861891015725, -0.52237736708181615, -0.71306938012235621, 
-0.67452456832296737, -0.99001419292519111, -0.79069315204798085, 
-0.73799654561936812, -0.56978811896556902, -0.9340298152795119, 
-0.73707176521119921, -0.46317992421620063, -0.63112624673412732, 
-0.50212866387620736, -0.40655537677633136, -0.61830482793800712, 
-0.6831400564378588, -0.48338367857823733, -0.69394435629389195, 
-0.29088987034082014, -0.88778789103246636, -0.46420550801475985, 
-0.45748197910368699, -0.69883248789420738, -0.82338510357115946, 
-0.88890510879395968, -0.42609373891778801, -0.65461926179351204, 
-0.99119499881571993, -0.67306218091251169, -0.76324582398016683, 
-0.84945929769482642, -0.37201604342778527, -1.00360912439253, 
-0.82975262612313083, -0.50939898544658679, -0.62383322396766983, 
-0.59702034121668868, -0.90021673775408739, -0.46694643317005968, 
-0.518481624194848, -0.37639251944215368, -0.19089924891035193, 
-0.37369244166596877, -0.13106690444693614, -0.29802884272110713, 
-0.34956624637988043, -0.46970241691156528, -0.75345336442255095, 
-1.3399876697245916, -0.47894190072293241, -0.50667190337959789, 
-0.15858224671338073, -0.50276978800794925, -0.6055146038716136, 
-0.36506043177269409, -0.60834538770608104, -0.34307290305388266, 
-0.67856485477570383, -0.31592317514187801, -0.24516132604810831, 
-0.4332232602145199, -0.49035611701292875, -0.45234860228805407, 
-0.34979471137782225, -0.44787682460855255, -0.4153285998554152, 
-0.47597199471302448, -0.87940827308935321, -0.84367901137117118, 
-0.54685477630349033, -0.48282961308093753, -0.55416507767495304, 
-0.53784493317303428, -0.67444660530452416, -0.37488127441556174, 
-0.71740369455738862, -0.82415494530957023, -0.29631125242688766, 
-0.18754903965143296, -0.78759243744338792, -0.2751346264574217, 
-0.54221097768452009, -0.69520412939841125, -0.65427394502354452, 
-0.29762211812472783, -0.3575764628284267, -0.32432314407319479, 
-0.41737997338252669, -0.44046556938653902, -1.0112912518810659, 
-0.59366253280574088, 0.19218607231457815, -0.56117538821922763, 
-0.52805164669861782, -0.58737416033139134, -0.2927154267501868, 
-0.31419557764367023, -0.61669340197061639, -1.178271133055329, 
-0.56013888799722966, -0.020357498840113113, -0.11033920591511355, 
0.42650614576108226, -0.11274079199431385, 0.36298376038120117, 
0.55766736903657721, 0.05497786690619675, 0.19265967574122594, 
0.31042977597528487, -0.086713359877989962, -0.072511873376870484, 
-0.0053268240426701972, -0.056788820899785826, -0.11512799664307415, 
-0.33477121405817284, 0.26250223319296839, 0.033009038911593548, 
0.28035254070087756, 0.15433375513838346, 0.43024627880544775, 
0.43564529797471541, -0.20031233081391145, -0.22670152319503786, 
0.16602000367879807, 0.36678625055114616, 0.15608887949077563, 
0.38475704753444384, 0.26052107044175105, 0.39365420718811689, 
-0.023015280070006017, 0.010241982050615465, -0.10717885293780299, 
0.38754747534844236, 0.086322207444499241, 0.18318054461837441, 
0.41947493001784469, -0.11626537204152722, 0.70315511003994713, 
-0.15523204779782845, -0.60940676043774678, -0.19104429799026504, 
-0.32427037413860754, -0.14694704612945447, -0.16958458542404808, 
-0.28262635794574842, 0.082681588974087714, 0.19563186003629404, 
0.016607960329194628, 0.22233492263164323, -0.34436457779766355, 
0.34855916410172372, -0.35098086528484385, -0.21037627517442947, 
0.31440490316561542, 0.034482660294367161, -0.49974584712458103, 
-0.24218356931951021, -0.092852542286362888, -0.17470428986925357, 
0.0060519795386575048, -0.11980131129887404, 0.49897294544292564, 
0.21882129129288905, 0.18063934834696085, -0.14148574861563998, 
-0.089942194822695581, -0.16791842284307587, 0.16219617575442527, 
0.086809893346650166, 0.37236911661623168, -0.16168499841024561, 
-0.27701397982549897, 0.25204884030892527, 0.10181009617533643, 
-0.22385515721986654, 0.10760021949359659, -0.026038409668310305, 
-0.15481337164982215, -0.29907881997139174, 0.15986630999588891, 
-0.00035342210714114577, 0.060596552674064663, 0.34614582385324982, 
0.10308195383312908, 0.007742475148232203, 0.16716155815856024, 
0.097476899766620148, -0.26504482347817199, -0.095878565537772478, 
0.15398628941454504, 0.25628369938795975, -0.44823967778364704, 
0.069154465525832001, -0.048336631304466283, 0.44261606044700536, 
-0.033112815303420792, -0.16713099620765401, 0.4174396339712298, 
0.80322294579628251, 0.086445160631474519, 0.3592285443294494, 
0.92596297490711943, 0.97214978867767832, 1.1517323072337986, 
1.4364174168241939, 0.82497573982076688, 1.2483209690094967, 
1.1068006015658571, 1.0594655330380818, 1.4833804836349502, 0.56969693100148056, 
1.252702900712632, 1.2095445077286904, 0.91880032082045482, 1.6480629801712254, 
0.62603395687245444, 0.90442923919021345, 1.1171135506749106, 
0.62086454335811703, 1.2245943477458618, 1.1465611021880402, 
1.2063284784629944, 1.1775315326976192, 1.4097484344125204, 1.4726441373150394, 
0.93146701528993403, 1.3516310367668045, 1.3491776929421493, 
0.94073130267506877, 1.1381950015009845, 1.2921761477380012, 
1.0557970351626982, 1.1562551142811199, 1.397942494665259, 0.85385809736015583, 
0.77136117884905919, 1.0790468590715527, 1.4737634952257426, 
1.2715596450035045, 1.372189612124046, 0.79102904868696355, 1.0069594972671356, 
0.78888368927937025, 0.39585990078443845, 0.98803637479273487, 
0.66831260153313321, 0.98649489317436834, 1.0342782259568466, 
1.6794294179724452, 0.96419778405797385, 0.85319937788629041, 
0.9185965606828711, 1.3199373342360921, 0.66244112119404241, 
0.8959504760530812, 1.398761453878032, 0.78258660388706003, 0.88097847148624941, 
0.86442263559399735, 0.69936998809965301, 0.84709688776778247, 
0.66801231814992235, 0.48218110100834921, 0.58119281723698002, 
0.79084310461983287, 1.1112116374963414, 1.1522028431474116, 
0.90152736137520173, 1.3590502880459121, 0.88183512240137096, 
1.3513308598542155, 0.96625051420550945, 1.1695834325138119, 
1.048687900244208, 0.81253051784164065, 1.2837794516315733, 1.3712035164767467, 
0.76404782444538766, 0.92185566477787184, 1.3952021636169509, 
0.80378145938647416, 0.89509903290966875, 0.91998766439178203, 
1.1018485971153369, 0.81262687961562152, 0.67671412674624065, 
0.49734476319587628, 1.2063198785863396, 1.4026741794480204, 
1.4849816532464546, 0.88245535224467442, 0.80908546257873137, 
1.1052030219487095, 0.80429379462774409, 0.6849291897453238, 
0.58048177774026577, 1.0218442867412383, 1.2553020990092105, 
1.5824616784499344, 1.2011610500941483, 1.7034822188977359, 1.9046201773602081, 
2.0570642860327042, 2.109208723339179, 2.5063579704215915, 2.1511220951340646, 
2.102802846971052, 1.7527377590861262, 2.2676790674944125, 1.9376745864952769, 
2.1304456869163535, 2.2498494241460056, 2.1200403213587644, 2.2288324378214819, 
2.1754496261828553, 2.0731566517280386, 1.9703899289759432, 1.8757370805635178, 
1.8448648773738987, 2.1161703041740312, 1.4903638738825655, 1.9074524368049977, 
2.325530199761825, 2.5748128867038975, 2.0769249517182313, 2.1265572903150414, 
1.4388956554817274, 2.3675978693558535, 1.8386693413049975, 2.1752064174025665, 
2.3892132221417879, 1.8831915635961298, 2.2148526663887997, 1.9555282555317797, 
1.8461714336961164, 1.6348103524808784, 2.3490194375853348, 1.8089075836586193, 
1.890473582662356, 2.330792836849632, 2.3755682767393478, 1.8902361561685379, 
1.9141504467740147, 2.0361313572686153, 1.6359200139242844, 1.8938206009021918, 
2.0280770650811091, 2.518818551769638, 3.0364722046661989, 2.19156987763039, 
2.1033646885439237, 1.6755948655502504, 2.3128275001735608, 2.2493795490679638, 
1.9045208132133213, 2.3489157921294472, 2.1679069291939133, 2.2854018622515824, 
1.7554180523129279, 2.0029055303563772, 2.2941744071072763, 1.9299949379404513, 
1.9671230068483796, 2.0634455807783976, 2.0537344892047642, 1.8786752934055102, 
1.7422753050416242, 2.8696418786375619, 2.4151935619417828, 1.9117002246625909, 
2.0981403007516124, 2.3738103148507048, 1.9083403203253031, 2.1480449322161017, 
2.3035932148479725, 2.2880070212030175, 2.2133359246316799, 1.8617869795865105, 
1.8926251519969799, 1.9991620617036074, 1.8774543306112599, 2.5333608703134667, 
2.152529660280043, 2.2068931883989871, 1.9386633413039285, 1.875674612099844, 
2.193466500043709, 2.0272610674638942, 1.6847969880551781, 2.4446181625156989, 
2.0961017168965737, 1.5544576741466016, 1.7450589548422291, 2.2918191065573637, 
1.8993023722748743, 1.7385463022792955, 1.6460305370959964, 2.0947615008143181, 
2.0711286748394762, 1.9205481143937027, 0.51386924169226644, 
0.33173588646457114, 0.45445918466469143, 0.48121985298427777, 
0.75871782818724465, 0.45104376017114667, 0.82723022635482379, 
0.40005873380706447, 0.61368182389603287, 0.40976551558197954, 
0.49540310398883441, 0.38680317945632497, 0.61671005869610929, 
0.59956095145503796, 0.68982386511674865, 0.44078845436047165, 
0.39598023587956516, 0.56494558730400291, 0.40653784501909396, 
0.67984615269290216, 0.48788708187942781, 0.68677004657835528, 
0.73759792290038706, 0.58928454962774524, 0.7087341557408432, 
1.0017364928240939, 0.21014227998092394, 0.45069441232607882, 
0.55153349111475758, 0.47023171683368881, 0.33551080844097614, 
0.0089536761603593914, 0.4910668128513544, 0.58502161167482269, 
0.58741753592746415, 0.6498070024228807, 0.57086575003728179, 
0.52097079160277371, 0.44633706837366077, 0.73759327158947685, 
0.41026646816795836, 0.55984141771747975, 0.34894701575598425, 
0.53736351697588614, 0.47960275400245994, 0.64005845085015389, 
0.58970511530344383, 0.75974759748871801, 0.84485826091126726, 
0.27492797882918135, 0.4381867321917044, 0.48307087811097499, 
0.72163289813323139, 0.42706976384542517, 0.63929400203600639, 
0.83032790827361591, 0.51483612310982829, 0.39650034750564045, 
0.74210187998924615, 0.47776937295844307, 0.55617736772259674, 
0.63872006606596321, 0.54453237968214396, 0.4933226819093483, 
0.53629187440235238, 0.44760990537961109, 0.46743122416382243, 
0.65072883681086868, 0.62163196536082632, 0.5495498938567559, 
0.65301216657550898, 0.66919725575045375, 0.56240822903277832, 
0.5807746827999265, 0.64747694731702921, 0.64138150496671531, 
0.41166280710114739, 0.63266951470503452, 0.40414843932214445, 
0.56542830671052391, 0.51384814197137951, 0.54984497236424124, 
0.66326852757861632, 0.72108413050958642, 0.60899411507280909, 
0.45800787040351515, 0.67279300721916369, 0.34110195855681807, 
0.5571460446715949, 0.53824516815334422, 0.3239383959535308, 
0.60046410022456076, 0.4524593627553084, 0.61122500254509193, 
0.60778076172688555, 0.61736997769812485, 0.49182711092600329, 
0.78371897044966787, 0.65413879863405255, 0.63717124492360355
), algo = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L), class = "factor", .Label = c("AGQ", 
"glmer", "glmmTMB", "mtram")), which = structure(c(1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L), .Label = c("beta1", 
"beta2", "beta3", "beta4", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-2000L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(5, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|",
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[0]), 
                                              expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
       )


## ----sim-probit_data, eval=FALSE----------------------------------------------
## N <- 100 # number of clusters
## Ni <- 5  # size of clusters
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(0, 1, 2)
## 
## x <- matrix(runif(N * Ni * 3), ncol = p)
## y <- qlogis(pnorm(x %*% beta + rnorm(N, sd = tau)[cls] + rnorm(nrow(x))))
## ##   ^^^^^^^^^^^^^ <- h^{-1}
## d <- data.frame(y = y, x, cls = cls)


## ----sim-tramME_data, eval=FALSE----------------------------------------------
## x <- matrix(runif(N * Ni * 3), ncol = p)
## y <- qt(plogis(x %*% beta + rnorm(N, sd = tau)[cls] + rlogis(nrow(x))), df = 3)
## ##   ^^^^^^^^^^ <- h^{-1}
## d <- data.frame(y = y, x, cls = cls)


## ----sim-mtram_data, eval=FALSE-----------------------------------------------
## Ui <- matrix(1, ncol = 1, nrow = Ni)
## S <- tau^2
## Sigma <- S * tcrossprod(Ui) + diag(Ni)
## D <- diag(sqrt(diag(Sigma)))
## Z <- rmvnorm(N, sigma = Sigma)
## x <- matrix(runif(N * Ni * 3), ncol = p)
## h1 <- function(x) qt(plogis(x), df = 3)
## # ^^ <- h^{-1}
## y <- h1(c(D %*% qlogis(pnorm(solve(D) %*% t(Z)))) + x %*% beta)
## d <- data.frame(y = y, x, cls = cls)


## ----sim-probit_c, echo=FALSE, eval=FALSE-------------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## 
## beta <- beta[-1]
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## 
## for (i in 1:nrow(rmt)) {
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   y <- qlogis(pnorm(x %*% beta + rnorm(N, sd = tau)[cls] + rnorm(nrow(x))))
##   ##   ^^^^^^^^^^^^^ <- h^{-1}
##   d <- data.frame(y = y, x, cls = cls)
##   m <- BoxCox(y ~ X1 + X2 + X3, data = d)
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
##   rmt[i, ] <- c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                 diag(solve(mt$Hessian))[-(1:7)])
## 
##   mm <- BoxCoxME(y ~ X1 + X2 + X3 + (1 | cls), data = d)
##   rME[i, ] <- c(coef(mm), sqrt(VarCorr(mm)$cls$var),
##                 diag(vcov(mm))[-(1:7)])
## 
##   lmt[i] <- logLik(mt)
##   lME[i] <- logLik(mm)
## 
##   print(i)
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"), each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 4)))


## ----sim-probit_c_plot, echo=FALSE--------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(beta, tau)

coefs <-
structure(list(coef = c(-0.038747654959092936, -0.061196119174379456, 
0.10623990035188051, -0.28751659490629017, -0.034052726643013974, 
-0.18225874022790187, -0.28961945492274593, 0.22890754074172681, 
-0.2537668276353155, 0.086399790632402301, -0.03274900633936164, 
-0.021498321169341264, -0.073484486811730532, 0.14976846657527201, 
0.42576503169257718, -0.14451232942925438, 0.21685809486997895, 
-0.098195432731629895, 0.0092203914088660518, 0.34837621402467478, 
0.052534419064231619, -0.27048316369233794, -0.23000391329204281, 
-0.067831086714499764, -0.20242251336118963, 0.011265395820317811, 
0.20318161298416235, -0.041558754860045934, 0.21572449143402353, 
0.081241459446766334, 0.035120997677976537, -0.0075832675133531077, 
-0.10011792510366675, -0.01360387170827624, -0.031568398209838103, 
0.01650189368462144, -0.058080624882797117, -0.070810285650816013, 
-0.018493123857008881, 0.30031728428009469, -0.13461106744781204, 
-0.0733469677163745, 0.076752992565815178, 0.020612731694190541, 
0.20551454033866093, 0.015435692078666951, 0.13876480552312132, 
0.17325411114474915, -0.23437549492030127, 0.07372536471245078, 
-0.092632373583238573, -0.15993610617153597, 0.39055603143890039, 
-0.0031410077619358802, 0.2015045298472074, -0.19748122793904935, 
0.010538083195133824, -0.12893063094787632, -0.1395452478715988, 
-0.048291806491672418, 0.18997588945795893, 0.24703118565630297, 
-0.18077701365890797, 0.11683135633798832, 0.11370742032223227, 
0.020223153412543986, -0.21644241462844654, 0.011286424477427554, 
0.22021721070167741, 0.23940596837835715, -0.10209029724644986, 
-0.048292827215391243, -0.033110765842736742, -0.28434398125578586, 
0.37381663702005963, -0.096417771375808994, 0.22867342855179981, 
-0.06025779846066362, -0.0063155126119044643, 0.057632354778365702, 
-0.16210542041997869, -0.20623978205664287, 0.029374739300249468, 
0.15616972049600711, 0.2930669121294015, 0.21224489451870238, 
-0.36368795258220188, 0.030342670753904587, -0.53779199438512071, 
0.090609672973728461, -0.099096145081575049, 0.16160173043629228, 
0.075189424338959032, -0.064843734880053808, 0.18556680854635013, 
-0.19190732986118128, -0.086144191218592639, 0.17015051031971198, 
-0.16819118302172298, -0.37195953739262205, 0.83906158745999482, 
0.99843524851609311, 1.1041893120848381, 1.1502217742282768, 
1.0353571100391199, 1.1598247688773218, 1.0271826167946314, 0.90012513291966123, 
1.1442444335073356, 1.3356319205910416, 0.88824496651752483, 
1.0388109407029404, 1.0274112953756784, 1.2546779028184272, 0.8482004425618731, 
0.98173792608007737, 0.94055160454138598, 1.0832394353675112, 
0.82707791579948331, 1.1126517383689811, 0.96970860576428042, 
0.7528051249251877, 1.0554984679442236, 1.0874815555466311, 1.1374366259375808, 
0.84338566645021007, 1.0596289354862793, 0.91924700996372433, 
1.2591201387838993, 0.86979352539198806, 0.84817251886910228, 
0.88097422173879547, 0.76247328571401762, 1.1134995762139859, 
0.8557424185554684, 1.0014556148957598, 1.1873637076829249, 0.80403173192210431, 
0.80052267434477853, 1.1009339171842765, 1.0265357401223951, 
1.1567864298541952, 1.0617656982327432, 0.80156487919168473, 
0.70483498013431234, 0.98079277741633808, 1.1327012555261657, 
0.79940146057581862, 0.79845782303417878, 0.97951070149841846, 
0.87033350127637221, 1.0213097706470091, 0.83838492529738406, 
1.0586539016268848, 0.86468144004639236, 0.89707296032844563, 
1.2697159773642523, 0.84042569163407832, 1.1150992301570566, 
1.2450405671261064, 1.2144274928891785, 1.2878949816010641, 1.1586669805785959, 
1.1079493546286594, 0.95243486440980429, 0.42826262467482074, 
1.0319214715276328, 1.2246619328571966, 1.1337388468993972, 0.95981337198758876, 
0.96288835558835895, 1.0059555207806876, 0.89543056696077117, 
0.99433818186226897, 0.79940565361069948, 1.0793458125066215, 
0.77347794578639562, 0.62055727220493317, 1.2772825134654131, 
0.91478495992359954, 0.76216971608788564, 1.0792287466840833, 
0.95627874962997028, 1.1889219580280193, 0.91855264813429738, 
1.3982460602379871, 1.1628037286646071, 0.84951740859762137, 
0.83949049771357076, 1.0623778417448644, 0.89665517074073819, 
1.3664174308238775, 0.8172396874713409, 0.99381705413709764, 
0.9950886070394489, 0.79305669908655907, 0.8580202018899421, 
1.0687435970661368, 0.96682679408050842, 1.0618870485810015, 
1.8421820468083727, 2.2138355993729233, 2.0504680777259381, 2.1124368124113309, 
2.0896734461440252, 1.8676115530551558, 2.2306809783576793, 1.9132970529625724, 
2.1479023888741398, 2.2179248582207296, 2.1663578738546994, 1.9591183526571094, 
1.8329604082120123, 2.1063087376071938, 1.8193062947236935, 1.7585683606033577, 
2.2373725880753712, 1.9028049948708738, 1.9987110497006961, 1.8233384190239561, 
1.9719895070406888, 1.8547870964356004, 1.8491654312752914, 2.0682691485086946, 
2.2673154579643211, 2.1200906403626001, 2.3491947025122575, 2.0030373289029373, 
2.0719241341107293, 2.3076158655572385, 1.966082819916964, 2.0033853153712289, 
2.1919125972534763, 2.2415180924367917, 2.1369555451067881, 2.0989154095348752, 
2.1110741752120661, 1.8206041171099832, 2.2560594581147262, 2.1738466254498792, 
2.0422674504555203, 2.3101124107276316, 2.1456315462145739, 1.8364954818818731, 
2.033457326080355, 2.3417002755072356, 2.0579114118460988, 1.8242203674397834, 
2.1236211588091751, 2.1668406101002788, 1.7310617885610016, 1.829619783162751, 
1.9481892936657441, 1.9778167489277891, 2.0238265856043953, 2.2414988680247117, 
2.0104382359688349, 1.8095426660721095, 2.127559610456236, 2.0952727198786616, 
1.8935835966345955, 1.9812996971857051, 2.2293313543679782, 2.0589561583479821, 
1.9663468077222177, 2.09179558496181, 1.7883401118959759, 2.1564140879275771, 
2.0366612086570601, 2.1256206145870427, 1.9885469343247997, 2.1738364813142521, 
2.0200491655674879, 1.9148290858488255, 1.8866323152751689, 2.001347842143522, 
2.1219521348846566, 1.824305157615618, 2.1201824257046828, 2.0836488562794346, 
1.7949551802827588, 2.0419004482138972, 2.2005916936772061, 1.7918873099418979, 
1.8480533221849686, 1.9944386081037035, 2.0556576595011617, 2.0472328845140964, 
2.2161819008981256, 1.7143784486722857, 1.882028163249819, 1.8565913031500585, 
2.0481675078749486, 1.9413226248722224, 2.0343588713461385, 2.0564396548939592, 
2.0434711933747036, 2.0509136234191256, 1.921284332428477, 1.9582077743605686, 
0.46744581160620696, 0.45975169420642398, 0.65689582135331215, 
0.58209516787299498, 0.50582411612429701, 0.47674560967353791, 
0.53406556491940194, 0.49734311828250222, 0.54228432519549208, 
0.57114173502880761, 0.4116824981373105, 0.51331838372986616, 
0.46209030788791522, 0.62312755001819087, 0.39232756538287272, 
0.45887474108416915, 0.51727535454515039, 0.64868785694988418, 
0.60268370153711437, 0.62028716568982434, 0.39841641710174597, 
0.41496736509637649, 0.55909348022946292, 0.48101434347700878, 
0.53271395634638641, 0.66464008931076812, 0.58926457407446631, 
0.65708823717269671, 0.61437964611429041, 0.48023933439259958, 
0.4365296628960661, 0.57333053647741639, 0.49327217077120561, 
0.43288115561545154, 0.54405560848485113, 0.46654729028424319, 
0.6165323387309799, 0.5953739721249236, 0.50980201547052295, 
0.53510322886705264, 0.57132681249842499, 0.57294197918985978, 
0.4989971656350482, 0.55744233788271202, 0.51517549574589749, 
0.57802161915068995, 0.47873598337988355, 0.31060580787336894, 
0.49712020937612728, 0.56345248290551175, 0.48031500713189906, 
0.55553746441976781, 0.54895917599131905, 0.49819471266639909, 
0.51234751633625086, 0.41956403928669916, 0.59229830962737862, 
0.38535594835514492, 0.61345400098967828, 0.54488084859104857, 
0.5801005477606026, 0.57282113655017275, 0.58364957683678154, 
0.5870963246340607, 0.44785416193591343, 0.45389903804076848, 
0.47671273953940452, 0.58181886402401539, 0.50541494466113268, 
0.31927250837315035, 0.52637845429594599, 0.65170601900612335, 
0.65312833348165589, 0.40564491382215251, 0.58298512422379167, 
0.64770380225538693, 0.6328212463357884, 0.36156119780771456, 
0.67978069569953725, 0.50859768264408955, 0.43302099865469301, 
0.43516284297970875, 0.45559607910372618, 0.60398051573325595, 
0.51762907711920347, 0.5765683714982115, 0.67210755652386389, 
0.44586238823721386, 0.68403085876790315, 0.59103234383771586, 
0.55100020926957383, 0.52568946633449565, 0.62236933380478721, 
0.57552116306619538, 0.5547991641847263, 0.63299205998493513, 
0.55313047385475422, 0.57627452244612687, 0.53024848779240319, 
0.46449629390481562, -0.038750579120820236, -0.061180216760025469, 
0.10627680522437449, -0.28752067775602258, -0.034019125167339723, 
-0.1823604644226986, -0.28967491897007397, 0.22890217822232992, 
-0.25374187620760491, 0.086335973121391518, -0.032783516483453221, 
-0.021496924761858653, -0.073427181958931026, 0.14977774167656052, 
0.42577631451489467, -0.14451096822988502, 0.21685792555521879, 
-0.098133673935751814, 0.0092260466046971377, 0.34838515204568987, 
0.052543898280385366, -0.27048308579368141, -0.23001454309378908, 
-0.067827120157909176, -0.20246967787706485, 0.011235740358204807, 
0.20318697759478763, -0.041554946957508332, 0.21573378991550451, 
0.081237218323307991, 0.035112775636978982, -0.0075863029056874674, 
-0.10008400424054871, -0.013689489827070972, -0.031564062185071511, 
0.016553861114674107, -0.05809905381253401, -0.070810408365043126, 
-0.018482140081006532, 0.30032847394720613, -0.13457709278812591, 
-0.073346932854730623, 0.076680043536006243, 0.020808978949099749, 
0.20551425771465417, 0.01541848577130144, 0.13877480462438624, 
0.1732516424104453, -0.23437623547373987, 0.073694694428876606, 
-0.092636339351183816, -0.15987455368775672, 0.3905562161241613, 
-0.0031414707239969131, 0.20146980321791544, -0.19745301241631638, 
0.010506881199463329, -0.12852211476089886, -0.13954707239138522, 
-0.048212139256606161, 0.18995990223187326, 0.24704654899179787, 
-0.18078122857857074, 0.11677594036608295, 0.11370134954939957, 
0.020242672656067339, -0.21636792869360166, 0.011251275114624497, 
0.22021129395094538, 0.23955987642092175, -0.10198695592324399, 
-0.048280601355003235, -0.033223158606177805, -0.28435091823810532, 
0.37381316351945887, -0.096419023486067304, 0.22866238652123427, 
-0.060192627726327108, -0.0063121192279352607, 0.057640775816760005, 
-0.16208853742510629, -0.20624699669529231, 0.029385928875913195, 
0.15616904693252873, 0.29306140022847332, 0.21224929718209437, 
-0.36367125918008647, 0.030332933605098859, -0.53776892019530809, 
0.090567654151445007, -0.099094821057807508, 0.16167946429079444, 
0.075193512324121525, -0.064844993138475041, 0.18560480370830215, 
-0.19194591934071001, -0.086077577105268296, 0.1701312998309015, 
-0.16821342123321809, -0.3719548302429585, 0.83905437971648444, 
0.99845582586011705, 1.1041633789172003, 1.150218339909884, 1.0353727831730342, 
1.1596926531363194, 1.0271764727452921, 0.9001030185356681, 1.1442644492457472, 
1.335546059626098, 0.88824506736262276, 1.0388133029682525, 1.0274791617796233, 
1.2546828736213347, 0.84818204330783975, 0.98173111528589063, 
0.94055204299358264, 1.0832731823592485, 0.8270776183435542, 
1.1126419617503593, 0.96971134257371461, 0.75280254142916059, 
1.0554958068586338, 1.0874808596845518, 1.1373482543272451, 0.84330181463520493, 
1.0596385892338882, 0.91925121214361927, 1.2591145837520434, 
0.86980822466553531, 0.84816423426907284, 0.8809871637233434, 
0.76251129058105982, 1.113437487381413, 0.85576662069849352, 
1.0015020073629137, 1.1873379577178558, 0.80404309786096051, 
0.80053192613037649, 1.1009521748977666, 1.0266124863029622, 
1.1567726681232218, 1.0617596083831493, 0.80149253244397378, 
0.70484047552969253, 0.98079484065858413, 1.1327124691644808, 
0.79939589573253911, 0.7984261207477088, 0.97950580416430288, 
0.87035452746094999, 1.0212569032044734, 0.8383950299020605, 
1.0586831485446515, 0.86460862225881108, 0.89711433475669, 1.2697948248750823, 
0.84058884095733122, 1.115100739895216, 1.2451443340465, 1.2141916767345668, 
1.2878808361536482, 1.1586636328957212, 1.1078772679497069, 0.95243498653529801, 
0.42827536933851174, 1.0319478953489409, 1.2248657055281829, 
1.1337559979454384, 0.9597736923891379, 0.96283399446245077, 
1.0059593917741145, 0.8954166697494379, 0.994325365976377, 0.79940941272449639, 
1.0793598203850585, 0.77346355688891821, 0.62052497717431432, 
1.2772914358891743, 0.91479860485508413, 0.76221258668542746, 
1.0792212911918992, 0.95628351629983155, 1.1889214446192504, 
0.91852220829647258, 1.398247623680787, 1.1628084702071455, 0.84949267580049714, 
0.83947943107074685, 1.0625156238531359, 0.89665051592482714, 
1.3665648455335486, 0.81724445317514471, 0.99381829830960966, 
0.99516267609382603, 0.79302489154338007, 0.85801778290826403, 
1.0688936979079169, 0.96682300937747845, 1.0618758919523892, 
1.8421561817042142, 2.2138605252373007, 2.0504497528011529, 2.112433947577649, 
2.0896791444683585, 1.8675466843986361, 2.2305533894267833, 1.9132510513561802, 
2.1479087756961857, 2.2178226862635726, 2.1662391211374947, 1.9591196958268144, 
1.8330361553651922, 2.1063001251460096, 1.819348769018363, 1.7585656464213331, 
2.2373756115026815, 1.9028190682118582, 1.9987144815465174, 1.8233496929984314, 
1.9719840345697939, 1.8547958668910809, 1.8491619214137009, 2.0682669351083756, 
2.2672174984016085, 2.1200530107545181, 2.3492000376835067, 2.0030420294544702, 
2.0718986925496785, 2.3076027626767384, 1.9660728120180559, 2.003384256041215, 
2.1919589225185119, 2.24160476777437, 2.1369893522240542, 2.0988949811946882, 
2.1110379956760861, 1.8206289220661922, 2.2560240630446455, 2.1738590413551369, 
2.042370824766722, 2.3100867594761629, 2.1456547536824009, 1.8363817617715414, 
2.0334626798013429, 2.3417146462216683, 2.0579243443167061, 1.8242143681860179, 
2.1235484246716556, 2.1668580642378905, 1.7310926544648493, 1.8295227069591962, 
1.948189131587682, 1.9778319921024718, 2.0237786573004266, 2.241462088660775, 
2.0104207787776898, 1.8097451738253267, 2.1275632184902054, 2.0954025689910978, 
1.8934261574853681, 1.9812600945370371, 2.2293313572136224, 2.0588701219789405, 
1.9663450234857962, 2.0917653215481469, 1.7882187353388268, 2.1563269001478118, 
2.0366834255446364, 2.1255737053977879, 1.9883201224214393, 2.1738322393089486, 
2.019972206089431, 1.914820246899279, 1.8866413865881952, 2.0013345773843905, 
2.1219257788326051, 1.8242482119419989, 2.1201742937031245, 2.0836466495313921, 
1.7949854397360587, 2.0418921016708871, 2.2006114452913139, 1.7918910422761956, 
1.8479911986355662, 1.9944391364358265, 2.055672498402473, 2.047281919579679, 
2.2161898656004562, 1.7144673858631259, 1.8820235388309319, 1.8567091096882891, 
2.0481732885963497, 1.9413242566617834, 2.034424311070961, 2.0563978708843442, 
2.043448922453746, 2.0507876292133562, 1.9213092929292108, 1.9581941221121753, 
0.46745241893074224, 0.45975435198658887, 0.65686177172505911, 
0.5820956821286204, 0.50581762929709539, 0.47674491259531188, 
0.53406152492315984, 0.49735463400556729, 0.54227960904304839, 
0.57112525843338102, 0.4116501195399544, 0.51331775819419612, 
0.46206556668622134, 0.62301663483336212, 0.39233805627065915, 
0.45887150310017644, 0.51727717761933301, 0.64871133902883849, 
0.60268502757904896, 0.62028195858591584, 0.39841282655288812, 
0.41492438245489843, 0.55909468835087328, 0.48101528147613459, 
0.5326630175201309, 0.66465379616078912, 0.58926499985856584, 
0.6570894229311951, 0.61438866953852256, 0.48023550165300299, 
0.43652855840147647, 0.57331487687044869, 0.49327904862190092, 
0.43293162596693391, 0.54405826741613417, 0.46657202025142075, 
0.616530854929775, 0.59536982663459315, 0.50980833212973065, 
0.53510581111318267, 0.57131252879388938, 0.57293541541518089, 
0.49899123024041431, 0.55749407084793612, 0.51517535864876474, 
0.57801362787861454, 0.47874178903253545, 0.31060372683181309, 
0.49712726248902894, 0.56351088740342248, 0.4802962525435831, 
0.55548140833814885, 0.54896206675714876, 0.49819255836115484, 
0.51232918346533529, 0.41955589265544752, 0.59231308798781013, 
0.38527003558158113, 0.61345491825341458, 0.54490340834906414, 
0.58004958121618388, 0.57305366602837804, 0.58364731458568686, 
0.58708553105100736, 0.44785205607273393, 0.4538848654997672, 
0.47671006134398136, 0.58184985748018903, 0.50534675916027949, 
0.31927331717328211, 0.52643242586198713, 0.65168611058886183, 
0.653148911039373, 0.40564475781159953, 0.58298930633715529, 
0.64771809785952184, 0.63281771469540316, 0.36151351238380247, 
0.67976539557965254, 0.50859808715740429, 0.43301742898838169, 
0.43516267691980981, 0.45558759614356481, 0.60398272774970774, 
0.51762355727813547, 0.57656887200635176, 0.67209586620311457, 
0.44585839215971867, 0.68401569548563734, 0.59109570039115766, 
0.55100315696911117, 0.52572378124065811, 0.62237111582806615, 
0.57552127544585274, 0.55480625744374534, 0.63294190260357985, 
0.55311984881505682, 0.57628215086308732, 0.53022305059177155, 
0.46449481009148352, -0.038747654959092936, -0.061196119174379456, 
0.10623990035188051, -0.28751659490629017, -0.034052726643013974, 
-0.18225874022790187, -0.28961945492274593, 0.22890754074172681, 
-0.2537668276353155, 0.086399790632402301, -0.03274900633936164, 
-0.021498321169341264, -0.073484486811730532, 0.14976846657527201, 
0.42576503169257718, -0.14451232942925438, 0.21685809486997895, 
-0.098195432731629895, 0.0092203914088660518, 0.34837621402467478, 
0.052534419064231619, -0.27048316369233794, -0.23000391329204281, 
-0.067831086714499764, -0.20242251336118963, 0.011265395820317811, 
0.20318161298416235, -0.041558754860045934, 0.21572449143402353, 
0.081241459446766334, 0.035120997677976537, -0.0075832675133531077, 
-0.10011792510366675, -0.01360387170827624, -0.031568398209838103, 
0.01650189368462144, -0.058080624882797117, -0.070810285650816013, 
-0.018493123857008881, 0.30031728428009469, -0.13461106744781204, 
-0.0733469677163745, 0.076752992565815178, 0.020612731694190541, 
0.20551454033866093, 0.015435692078666951, 0.13876480552312132, 
0.17325411114474915, -0.23437549492030127, 0.07372536471245078, 
-0.092632373583238573, -0.15993610617153597, 0.39055603143890039, 
-0.0031410077619358802, 0.2015045298472074, -0.19748122793904935, 
0.010538083195133824, -0.12893063094787632, -0.1395452478715988, 
-0.048291806491672418, 0.18997588945795893, 0.24703118565630297, 
-0.18077701365890797, 0.11683135633798832, 0.11370742032223227, 
0.020223153412543986, -0.21644241462844654, 0.011286424477427554, 
0.22021721070167741, 0.23940596837835715, -0.10209029724644986, 
-0.048292827215391243, -0.033110765842736742, -0.28434398125578586, 
0.37381663702005963, -0.096417771375808994, 0.22867342855179981, 
-0.06025779846066362, -0.0063155126119044643, 0.057632354778365702, 
-0.16210542041997869, -0.20623978205664287, 0.029374739300249468, 
0.15616972049600711, 0.2930669121294015, 0.21224489451870238, 
-0.36368795258220188, 0.030342670753904587, -0.53779199438512071, 
0.090609672973728461, -0.099096145081575049, 0.16160173043629228, 
0.075189424338959032, -0.064843734880053808, 0.18556680854635013, 
-0.19190732986118128, -0.086144191218592639, 0.17015051031971198, 
-0.16819118302172298, -0.37195953739262205, 0.83906158745999482, 
0.99843524851609311, 1.1041893120848381, 1.1502217742282768, 
1.0353571100391199, 1.1598247688773218, 1.0271826167946314, 0.90012513291966123, 
1.1442444335073356, 1.3356319205910416, 0.88824496651752483, 
1.0388109407029404, 1.0274112953756784, 1.2546779028184272, 0.8482004425618731, 
0.98173792608007737, 0.94055160454138598, 1.0832394353675112, 
0.82707791579948331, 1.1126517383689811, 0.96970860576428042, 
0.7528051249251877, 1.0554984679442236, 1.0874815555466311, 1.1374366259375808, 
0.84338566645021007, 1.0596289354862793, 0.91924700996372433, 
1.2591201387838993, 0.86979352539198806, 0.84817251886910228, 
0.88097422173879547, 0.76247328571401762, 1.1134995762139859, 
0.8557424185554684, 1.0014556148957598, 1.1873637076829249, 0.80403173192210431, 
0.80052267434477853, 1.1009339171842765, 1.0265357401223951, 
1.1567864298541952, 1.0617656982327432, 0.80156487919168473, 
0.70483498013431234, 0.98079277741633808, 1.1327012555261657, 
0.79940146057581862, 0.79845782303417878, 0.97951070149841846, 
0.87033350127637221, 1.0213097706470091, 0.83838492529738406, 
1.0586539016268848, 0.86468144004639236, 0.89707296032844563, 
1.2697159773642523, 0.84042569163407832, 1.1150992301570566, 
1.2450405671261064, 1.2144274928891785, 1.2878949816010641, 1.1586669805785959, 
1.1079493546286594, 0.95243486440980429, 0.42826262467482074, 
1.0319214715276328, 1.2246619328571966, 1.1337388468993972, 0.95981337198758876, 
0.96288835558835895, 1.0059555207806876, 0.89543056696077117, 
0.99433818186226897, 0.79940565361069948, 1.0793458125066215, 
0.77347794578639562, 0.62055727220493317, 1.2772825134654131, 
0.91478495992359954, 0.76216971608788564, 1.0792287466840833, 
0.95627874962997028, 1.1889219580280193, 0.91855264813429738, 
1.3982460602379871, 1.1628037286646071, 0.84951740859762137, 
0.83949049771357076, 1.0623778417448644, 0.89665517074073819, 
1.3664174308238775, 0.8172396874713409, 0.99381705413709764, 
0.9950886070394489, 0.79305669908655907, 0.8580202018899421, 
1.0687435970661368, 0.96682679408050842, 1.0618870485810015, 
1.8421820468083727, 2.2138355993729233, 2.0504680777259381, 2.1124368124113309, 
2.0896734461440252, 1.8676115530551558, 2.2306809783576793, 1.9132970529625724, 
2.1479023888741398, 2.2179248582207296, 2.1663578738546994, 1.9591183526571094, 
1.8329604082120123, 2.1063087376071938, 1.8193062947236935, 1.7585683606033577, 
2.2373725880753712, 1.9028049948708738, 1.9987110497006961, 1.8233384190239561, 
1.9719895070406888, 1.8547870964356004, 1.8491654312752914, 2.0682691485086946, 
2.2673154579643211, 2.1200906403626001, 2.3491947025122575, 2.0030373289029373, 
2.0719241341107293, 2.3076158655572385, 1.966082819916964, 2.0033853153712289, 
2.1919125972534763, 2.2415180924367917, 2.1369555451067881, 2.0989154095348752, 
2.1110741752120661, 1.8206041171099832, 2.2560594581147262, 2.1738466254498792, 
2.0422674504555203, 2.3101124107276316, 2.1456315462145739, 1.8364954818818731, 
2.033457326080355, 2.3417002755072356, 2.0579114118460988, 1.8242203674397834, 
2.1236211588091751, 2.1668406101002788, 1.7310617885610016, 1.829619783162751, 
1.9481892936657441, 1.9778167489277891, 2.0238265856043953, 2.2414988680247117, 
2.0104382359688349, 1.8095426660721095, 2.127559610456236, 2.0952727198786616, 
1.8935835966345955, 1.9812996971857051, 2.2293313543679782, 2.0589561583479821, 
1.9663468077222177, 2.09179558496181, 1.7883401118959759, 2.1564140879275771, 
2.0366612086570601, 2.1256206145870427, 1.9885469343247997, 2.1738364813142521, 
2.0200491655674879, 1.9148290858488255, 1.8866323152751689, 2.001347842143522, 
2.1219521348846566, 1.824305157615618, 2.1201824257046828, 2.0836488562794346, 
1.7949551802827588, 2.0419004482138972, 2.2005916936772061, 1.7918873099418979, 
1.8480533221849686, 1.9944386081037035, 2.0556576595011617, 2.0472328845140964, 
2.2161819008981256, 1.7143784486722857, 1.882028163249819, 1.8565913031500585, 
2.0481675078749486, 1.9413226248722224, 2.0343588713461385, 2.0564396548939592, 
2.0434711933747036, 2.0509136234191256, 1.921284332428477, 1.9582077743605686, 
0.46744581160620696, 0.45975169420642398, 0.65689582135331215, 
0.58209516787299498, 0.50582411612429701, 0.47674560967353791, 
0.53406556491940194, 0.49734311828250222, 0.54228432519549208, 
0.57114173502880761, 0.4116824981373105, 0.51331838372986616, 
0.46209030788791522, 0.62312755001819087, 0.39232756538287272, 
0.45887474108416915, 0.51727535454515039, 0.64868785694988418, 
0.60268370153711437, 0.62028716568982434, 0.39841641710174597, 
0.41496736509637649, 0.55909348022946292, 0.48101434347700878, 
0.53271395634638641, 0.66464008931076812, 0.58926457407446631, 
0.65708823717269671, 0.61437964611429041, 0.48023933439259958, 
0.4365296628960661, 0.57333053647741639, 0.49327217077120561, 
0.43288115561545154, 0.54405560848485113, 0.46654729028424319, 
0.6165323387309799, 0.5953739721249236, 0.50980201547052295, 
0.53510322886705264, 0.57132681249842499, 0.57294197918985978, 
0.4989971656350482, 0.55744233788271202, 0.51517549574589749, 
0.57802161915068995, 0.47873598337988355, 0.31060580787336894, 
0.49712020937612728, 0.56345248290551175, 0.48031500713189906, 
0.55553746441976781, 0.54895917599131905, 0.49819471266639909, 
0.51234751633625086, 0.41956403928669916, 0.59229830962737862, 
0.38535594835514492, 0.61345400098967828, 0.54488084859104857, 
0.5801005477606026, 0.57282113655017275, 0.58364957683678154, 
0.5870963246340607, 0.44785416193591343, 0.45389903804076848, 
0.47671273953940452, 0.58181886402401539, 0.50541494466113268, 
0.31927250837315035, 0.52637845429594599, 0.65170601900612335, 
0.65312833348165589, 0.40564491382215251, 0.58298512422379167, 
0.64770380225538693, 0.6328212463357884, 0.36156119780771456, 
0.67978069569953725, 0.50859768264408955, 0.43302099865469301, 
0.43516284297970875, 0.45559607910372618, 0.60398051573325595, 
0.51762907711920347, 0.5765683714982115, 0.67210755652386389, 
0.44586238823721386, 0.68403085876790315, 0.59103234383771586, 
0.55100020926957383, 0.52568946633449565, 0.62236933380478721, 
0.57552116306619538, 0.5547991641847263, 0.63299205998493513, 
0.55313047385475422, 0.57627452244612687, 0.53024848779240319, 
0.46449629390481562, -0.038750579120820236, -0.061180216760025469, 
0.10627680522437449, -0.28752067775602258, -0.034019125167339723, 
-0.1823604644226986, -0.28967491897007397, 0.22890217822232992, 
-0.25374187620760491, 0.086335973121391518, -0.032783516483453221, 
-0.021496924761858653, -0.073427181958931026, 0.14977774167656052, 
0.42577631451489467, -0.14451096822988502, 0.21685792555521879, 
-0.098133673935751814, 0.0092260466046971377, 0.34838515204568987, 
0.052543898280385366, -0.27048308579368141, -0.23001454309378908, 
-0.067827120157909176, -0.20246967787706485, 0.011235740358204807, 
0.20318697759478763, -0.041554946957508332, 0.21573378991550451, 
0.081237218323307991, 0.035112775636978982, -0.0075863029056874674, 
-0.10008400424054871, -0.013689489827070972, -0.031564062185071511, 
0.016553861114674107, -0.05809905381253401, -0.070810408365043126, 
-0.018482140081006532, 0.30032847394720613, -0.13457709278812591, 
-0.073346932854730623, 0.076680043536006243, 0.020808978949099749, 
0.20551425771465417, 0.01541848577130144, 0.13877480462438624, 
0.1732516424104453, -0.23437623547373987, 0.073694694428876606, 
-0.092636339351183816, -0.15987455368775672, 0.3905562161241613, 
-0.0031414707239969131, 0.20146980321791544, -0.19745301241631638, 
0.010506881199463329, -0.12852211476089886, -0.13954707239138522, 
-0.048212139256606161, 0.18995990223187326, 0.24704654899179787, 
-0.18078122857857074, 0.11677594036608295, 0.11370134954939957, 
0.020242672656067339, -0.21636792869360166, 0.011251275114624497, 
0.22021129395094538, 0.23955987642092175, -0.10198695592324399, 
-0.048280601355003235, -0.033223158606177805, -0.28435091823810532, 
0.37381316351945887, -0.096419023486067304, 0.22866238652123427, 
-0.060192627726327108, -0.0063121192279352607, 0.057640775816760005, 
-0.16208853742510629, -0.20624699669529231, 0.029385928875913195, 
0.15616904693252873, 0.29306140022847332, 0.21224929718209437, 
-0.36367125918008647, 0.030332933605098859, -0.53776892019530809, 
0.090567654151445007, -0.099094821057807508, 0.16167946429079444, 
0.075193512324121525, -0.064844993138475041, 0.18560480370830215, 
-0.19194591934071001, -0.086077577105268296, 0.1701312998309015, 
-0.16821342123321809, -0.3719548302429585, 0.83905437971648444, 
0.99845582586011705, 1.1041633789172003, 1.150218339909884, 1.0353727831730342, 
1.1596926531363194, 1.0271764727452921, 0.9001030185356681, 1.1442644492457472, 
1.335546059626098, 0.88824506736262276, 1.0388133029682525, 1.0274791617796233, 
1.2546828736213347, 0.84818204330783975, 0.98173111528589063, 
0.94055204299358264, 1.0832731823592485, 0.8270776183435542, 
1.1126419617503593, 0.96971134257371461, 0.75280254142916059, 
1.0554958068586338, 1.0874808596845518, 1.1373482543272451, 0.84330181463520493, 
1.0596385892338882, 0.91925121214361927, 1.2591145837520434, 
0.86980822466553531, 0.84816423426907284, 0.8809871637233434, 
0.76251129058105982, 1.113437487381413, 0.85576662069849352, 
1.0015020073629137, 1.1873379577178558, 0.80404309786096051, 
0.80053192613037649, 1.1009521748977666, 1.0266124863029622, 
1.1567726681232218, 1.0617596083831493, 0.80149253244397378, 
0.70484047552969253, 0.98079484065858413, 1.1327124691644808, 
0.79939589573253911, 0.7984261207477088, 0.97950580416430288, 
0.87035452746094999, 1.0212569032044734, 0.8383950299020605, 
1.0586831485446515, 0.86460862225881108, 0.89711433475669, 1.2697948248750823, 
0.84058884095733122, 1.115100739895216, 1.2451443340465, 1.2141916767345668, 
1.2878808361536482, 1.1586636328957212, 1.1078772679497069, 0.95243498653529801, 
0.42827536933851174, 1.0319478953489409, 1.2248657055281829, 
1.1337559979454384, 0.9597736923891379, 0.96283399446245077, 
1.0059593917741145, 0.8954166697494379, 0.994325365976377, 0.79940941272449639, 
1.0793598203850585, 0.77346355688891821, 0.62052497717431432, 
1.2772914358891743, 0.91479860485508413, 0.76221258668542746, 
1.0792212911918992, 0.95628351629983155, 1.1889214446192504, 
0.91852220829647258, 1.398247623680787, 1.1628084702071455, 0.84949267580049714, 
0.83947943107074685, 1.0625156238531359, 0.89665051592482714, 
1.3665648455335486, 0.81724445317514471, 0.99381829830960966, 
0.99516267609382603, 0.79302489154338007, 0.85801778290826403, 
1.0688936979079169, 0.96682300937747845, 1.0618758919523892, 
1.8421561817042142, 2.2138605252373007, 2.0504497528011529, 2.112433947577649, 
2.0896791444683585, 1.8675466843986361, 2.2305533894267833, 1.9132510513561802, 
2.1479087756961857, 2.2178226862635726, 2.1662391211374947, 1.9591196958268144, 
1.8330361553651922, 2.1063001251460096, 1.819348769018363, 1.7585656464213331, 
2.2373756115026815, 1.9028190682118582, 1.9987144815465174, 1.8233496929984314, 
1.9719840345697939, 1.8547958668910809, 1.8491619214137009, 2.0682669351083756, 
2.2672174984016085, 2.1200530107545181, 2.3492000376835067, 2.0030420294544702, 
2.0718986925496785, 2.3076027626767384, 1.9660728120180559, 2.003384256041215, 
2.1919589225185119, 2.24160476777437, 2.1369893522240542, 2.0988949811946882, 
2.1110379956760861, 1.8206289220661922, 2.2560240630446455, 2.1738590413551369, 
2.042370824766722, 2.3100867594761629, 2.1456547536824009, 1.8363817617715414, 
2.0334626798013429, 2.3417146462216683, 2.0579243443167061, 1.8242143681860179, 
2.1235484246716556, 2.1668580642378905, 1.7310926544648493, 1.8295227069591962, 
1.948189131587682, 1.9778319921024718, 2.0237786573004266, 2.241462088660775, 
2.0104207787776898, 1.8097451738253267, 2.1275632184902054, 2.0954025689910978, 
1.8934261574853681, 1.9812600945370371, 2.2293313572136224, 2.0588701219789405, 
1.9663450234857962, 2.0917653215481469, 1.7882187353388268, 2.1563269001478118, 
2.0366834255446364, 2.1255737053977879, 1.9883201224214393, 2.1738322393089486, 
2.019972206089431, 1.914820246899279, 1.8866413865881952, 2.0013345773843905, 
2.1219257788326051, 1.8242482119419989, 2.1201742937031245, 2.0836466495313921, 
1.7949854397360587, 2.0418921016708871, 2.2006114452913139, 1.7918910422761956, 
1.8479911986355662, 1.9944391364358265, 2.055672498402473, 2.047281919579679, 
2.2161898656004562, 1.7144673858631259, 1.8820235388309319, 1.8567091096882891, 
2.0481732885963497, 1.9413242566617834, 2.034424311070961, 2.0563978708843442, 
2.043448922453746, 2.0507876292133562, 1.9213092929292108, 1.9581941221121753, 
0.46745241893074224, 0.45975435198658887, 0.65686177172505911, 
0.5820956821286204, 0.50581762929709539, 0.47674491259531188, 
0.53406152492315984, 0.49735463400556729, 0.54227960904304839, 
0.57112525843338102, 0.4116501195399544, 0.51331775819419612, 
0.46206556668622134, 0.62301663483336212, 0.39233805627065915, 
0.45887150310017644, 0.51727717761933301, 0.64871133902883849, 
0.60268502757904896, 0.62028195858591584, 0.39841282655288812, 
0.41492438245489843, 0.55909468835087328, 0.48101528147613459, 
0.5326630175201309, 0.66465379616078912, 0.58926499985856584, 
0.6570894229311951, 0.61438866953852256, 0.48023550165300299, 
0.43652855840147647, 0.57331487687044869, 0.49327904862190092, 
0.43293162596693391, 0.54405826741613417, 0.46657202025142075, 
0.616530854929775, 0.59536982663459315, 0.50980833212973065, 
0.53510581111318267, 0.57131252879388938, 0.57293541541518089, 
0.49899123024041431, 0.55749407084793612, 0.51517535864876474, 
0.57801362787861454, 0.47874178903253545, 0.31060372683181309, 
0.49712726248902894, 0.56351088740342248, 0.4802962525435831, 
0.55548140833814885, 0.54896206675714876, 0.49819255836115484, 
0.51232918346533529, 0.41955589265544752, 0.59231308798781013, 
0.38527003558158113, 0.61345491825341458, 0.54490340834906414, 
0.58004958121618388, 0.57305366602837804, 0.58364731458568686, 
0.58708553105100736, 0.44785205607273393, 0.4538848654997672, 
0.47671006134398136, 0.58184985748018903, 0.50534675916027949, 
0.31927331717328211, 0.52643242586198713, 0.65168611058886183, 
0.653148911039373, 0.40564475781159953, 0.58298930633715529, 
0.64771809785952184, 0.63281771469540316, 0.36151351238380247, 
0.67976539557965254, 0.50859808715740429, 0.43301742898838169, 
0.43516267691980981, 0.45558759614356481, 0.60398272774970774, 
0.51762355727813547, 0.57656887200635176, 0.67209586620311457, 
0.44585839215971867, 0.68401569548563734, 0.59109570039115766, 
0.55100315696911117, 0.52572378124065811, 0.62237111582806615, 
0.57552127544585274, 0.55480625744374534, 0.63294190260357985, 
0.55311984881505682, 0.57628215086308732, 0.53022305059177155, 
0.46449481009148352), algo = structure(c(2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), class = "factor", .Label = c("mtram", 
"tramME")), which = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", "beta2", 
"beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-1600L))



bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
)


## ----sim-tramME_c, echo=FALSE, eval=FALSE-------------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## 
## beta <- beta[-1]
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## 
## for (i in 1:nrow(rmt)) {
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   y <- qt(plogis(x %*% beta + rnorm(N, sd = tau)[cls] + rlogis(nrow(x))), df = 3)
##   ##   ^^^^^^^^^^ <- h^{-1}
##   d <- data.frame(y = y, x, cls = cls)
##   m <- Colr(y ~ X1 + X2 + X3, data = d)
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
##   rmt[i, ] <- c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                 diag(solve(mt$Hessian))[-(1:7)])
## 
##   mm <- ColrME(y ~ X1 + X2 + X3 + (1 | cls), data = d)
##   rME[i, ] <- c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)])
## 
##   lmt[i] <- logLik(mt)
##   lME[i] <- logLik(mm)
## 
##   print(i)
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"),
##                                          each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 2)))


## ----sim-tramME_c_plot, echo=FALSE--------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(-beta, tau)

coefs <-
structure(list(coef = c(0.17669557238143982, -0.050831839049816009, 
-0.17804912999799935, 0.092502775887395514, -0.42002701495815753, 
0.64910469840326135, -0.39629329076705661, 0.19421540025205003, 
-0.10466994910377778, 0.40189643018211296, -0.27955884050850555, 
0.11344451377568118, -0.0687356417647568, -0.039799702277745271, 
-0.054862047782831894, 0.036179718138488422, 0.45807633600525716, 
0.34761640311989822, -0.19078216171844192, -0.26766816356199391, 
-0.23045974019892815, -0.414590656740221, 0.26222907830231346, 
0.31358930660210121, 0.042929834817526741, -0.43781200414908988, 
0.069857512821384332, -0.068447297972315704, -0.58442716272945949, 
0.34439120006665108, 0.020075023740892858, 0.024044877752570862, 
-0.71602883575449727, -0.13572471727253649, -0.28794065164720695, 
-0.14825007741373103, -0.080602782826523597, 0.17568591961115729, 
0.033334404913050786, -0.19096509895213462, -0.56398518332440006, 
0.02425659833687506, -0.4026922329731224, 0.52155536682327341, 
0.25524688295736792, 0.12797915112903033, -0.38742606326514156, 
0.77788348284403996, 0.42919764946134403, -0.25907871542772637, 
-0.28043589329862323, -0.6727123467488042, -0.46242361323157699, 
-0.25100022449757842, 0.13944159365595707, 0.80689107842717267, 
-0.17646041779605282, -0.38527471167655386, -0.17471205758392239, 
0.37808277478601726, 0.40323300373321352, 0.10957567197572567, 
0.50079590916317762, -0.034910532370477552, 0.34441480426472759, 
0.10016547553120229, -0.23372451511980866, 0.2210399071819579, 
0.16624026445525195, 0.64786407491169906, 0.33114277601868936, 
0.031773387019957383, -0.60521160619258296, 0.13746230632707551, 
0.2033694355325246, -0.17333341410589534, 0.23416428483575211, 
-0.51574830680218042, -0.56554396188817502, -0.34469598754298175, 
0.25999896337359629, 0.36092145274221293, -0.0056887861274186637, 
-0.39817811294419009, -0.052278189777653646, 0.082934330907391704, 
0.55321161770276617, 0.33028227057892351, 0.81276866401455228, 
0.40732967184507957, -0.10964826403629623, -0.14953937384623245, 
-0.062878057167642068, -0.10590544328715915, 0.17263428401208908, 
0.25148753754310382, 0.060777241841265484, -0.093620326100404189, 
-0.049728123219487573, -0.43452008253882013, -0.92779284795993189, 
-0.98840837549742289, -0.44298152506976024, -0.40481210642188692, 
-0.76970776156612175, -0.91452061889436231, -0.83820877325968524, 
-0.97856667663613728, -0.81181139469458108, -0.51421151824845945, 
-1.2413713148944621, -0.57466950895409852, -1.111174194250286, 
-0.82041772487160458, -0.67329179597476019, -0.68634970104266557, 
-1.0992059639930989, -1.4410959160232373, -0.64585366963139379, 
-0.57368307679596986, -1.3714306800627749, -1.001598058463586, 
-0.79927669365155984, -1.0745143903554168, -0.56296269352935979, 
-0.91937906594603591, -0.98390434132849058, -0.84383122949704292, 
-0.36872585281923465, -1.1955473786304025, -1.4219875294459419, 
-0.98062500019388499, -0.71077794679982553, -0.92654244814318631, 
-1.1413079452418142, -1.0521886039258821, -1.4214777258625988, 
-0.68688880516064321, -0.89649785874495569, -0.51525602024336525, 
-1.3456267701519951, -0.96784457536646817, -0.72321351565281611, 
-0.40074752598294161, -0.60021523776923225, -1.131998989749208, 
-1.4078209037859886, -0.99986851626183837, -1.1335876926969866, 
-0.78371769758217236, -0.85415646007814716, -1.2372482261867161, 
-0.97823698047503727, -1.0457421257887389, -0.86519026770023522, 
-0.69444746717898442, -0.6494917121530559, -0.83310745230635574, 
-1.1402542111602338, -1.0303584637813543, -1.1787824396033457, 
-0.88407234736566576, -1.0091223794104265, -0.84204057253452436, 
-1.4337317454247882, -0.63931802395432802, -1.3457057160195169, 
-0.076026219078555737, -0.14118040594407419, -1.1070939019389165, 
-1.369062321473854, -0.96703609738561946, -0.92431020892074178, 
-1.4052317553817546, -0.77362651554633943, -1.1521917410073501, 
-1.2304014572836364, -0.93661582749909988, -0.76325404447827727, 
-0.91246539005871907, -1.3170027491194443, -1.0233238826406152, 
-0.68252728706890553, -1.2139031362954136, -1.2139785259772122, 
-1.4798325541810202, -0.9186107645455498, -0.91045454770954515, 
-1.195710930705365, -0.45664654345654465, -0.93285489091865914, 
-0.74911659594852409, -0.73132210078614113, -0.51528018852747315, 
-0.63788587563757448, -0.95277164537011627, -0.99081454792424306, 
-0.7853221098205927, -0.7007777946399858, -1.1232822682122598, 
-2.3300300428292617, -1.5839402732315169, -1.8204339185521679, 
-2.1179566234086722, -1.8924058107888913, -2.3085554931308225, 
-2.1166157682545426, -1.9456206044921522, -1.8932543330090872, 
-2.1912331856513472, -1.9668105974789529, -2.0665851822169703, 
-2.0619975034318907, -1.9644457732537746, -2.0227367179808815, 
-1.7310834495740737, -2.015818852679466, -2.2855307896959345, 
-1.9673176483407602, -2.3866093232871637, -2.9341537620689184, 
-2.0999651830503732, -1.8967484801558092, -1.6718724215318668, 
-2.3238430532011725, -1.4782782473834364, -2.0923964225807539, 
-2.101111055349612, -1.8260900665662871, -1.8748007273882314, 
-1.9012870791419356, -1.9671634822315629, -1.533416791030191, 
-2.0456689527842147, -1.8536903934864031, -1.4711690632685959, 
-1.7846327343013448, -1.8696602842883685, -1.8137000305551345, 
-2.284275844794057, -1.980398252296216, -1.8827679564199098, 
-1.6704526448626489, -1.9450900636516695, -2.1236323544074693, 
-2.5098421168175675, -1.8224319093501995, -2.4096101941721906, 
-1.6958558000766273, -2.14774045274515, -2.0412516081835506, 
-1.8822462104275015, -2.4759159634102552, -2.1859401811518655, 
-1.7492932771221159, -2.603005073636171, -1.2550098118762449, 
-2.245264972589974, -2.2339685446689668, -2.401541714196429, 
-2.0164081519708716, -1.7004108584864297, -1.5151035283825383, 
-2.335114512216983, -1.4904407886409117, -1.7772763504533393, 
-2.0539174747493072, -1.7111616382783108, -1.1386813118924006, 
-2.8688752540062938, -2.440484725654716, -1.9348213684138991, 
-1.3560974007260644, -2.149359268192585, -2.5918634228661781, 
-1.9655697092273521, -1.5108732712617754, -1.652803555046098, 
-1.5947418839605416, -2.2246047736890247, -1.854199348320668, 
-1.9645722783146737, -1.4313465231865536, -2.3778114063014462, 
-1.5061679358352109, -2.3078640784033495, -2.0836586638560983, 
-2.465311307322354, -1.7511333458343883, -1.9402131980989605, 
-1.9327777363839678, -2.1400433215906771, -2.200686604230027, 
-2.1973995121871659, -2.119206440893632, -1.6960145321108746, 
-2.0602721834819526, -1.5782032882891339, -2.577366102695855, 
-1.4382711053380033, 0.34744533190086513, 0.50232847311929163, 
0.46941853912385395, 0.49532851385459453, 0.51118475240031114, 
0.58361726696360372, 0.56577749678551437, 0.50063921323654681, 
0.55525806085191542, 0.47345225642277528, 0.34318701739704793, 
0.58809004990768399, 0.51412687500555321, 0.32578269974433827, 
0.304425879552782, 0.49074449454953256, 0.24821681664112033, 
0.79812891374796047, 0.31081631344532201, 0.32865909817384836, 
0.69015028832528225, 0.48482040340331711, 0.23988089537401069, 
0.42860016631042291, 0.21493191387108165, 0.51127785634533884, 
0.40315475661729466, 0.40535470456706618, 0.45213259077240009, 
0.45379006047186199, 0.42734791168092989, 0.50435544531956433, 
0.73943336963568618, 0.60094413132238467, 0.41429036525665475, 
0.497941384534481, 0.4278450824919306, 0.50830716743955973, 0.27481760033664587, 
0.20887147760832769, 0.24139413964003958, 0.4475476606370038, 
0.32183486965760233, 0.70058824993953472, 0.33316545895715283, 
0.61567395249231016, 0.55060342720386735, 0.70990233910460332, 
7.6413982965490025e-05, 0.3519123554944546, 2.5290307467279874e-05, 
0.4711478177247192, 0.49799897601453363, 0.58166251071017305, 
0.47295424712735129, 0.51092175008897156, 0.45902755331118106, 
0.50851061994215696, 0.57560245127631027, 0.61400793931002684, 
0.35938588490387302, 0.56251547594662443, 0.44716840135516789, 
0.65163054773340479, 0.68505925669562817, 0.36215407257521914, 
0.48107529765008566, 0.44300847833479268, 0.33204306738010586, 
0.76226404371440082, 0.4908180448861561, 0.58026656918335118, 
0.59880880179016849, 0.71576815036894925, 0.44546300128392169, 
0.65225433291924029, 0.59763602446353303, 0.41137393280580414, 
0.62786109466364315, 0.67032419517602204, 0.50263179653792212, 
0.66782718235325023, 0.50611455838982444, 0.57044059418773996, 
0.3278766728685481, 0.38077188842009063, 0.76422850396783282, 
0.30023392699861379, 0.64272278481066503, 0.55347268211035883, 
0.51715780415333501, 0.070389910158084767, 0.52067688622860042, 
0.48213368469461515, 0.67217229723833971, 0.38466342662598607, 
0.49943119541469222, 0.39098783134318099, 0.47684503747293727, 
0.17977006159009293, 0.17307946036703384, -0.044403990021328103, 
-0.18149817939689405, 0.11129047673324793, -0.42343317508452716, 
0.66748748056366936, -0.4072548444726935, 0.18681275037397529, 
-0.13534000997645254, 0.40210551650829374, -0.26489400930145141, 
0.11606660602713373, -0.055686793260439162, -0.051304277931811137, 
-0.046883581781802561, 0.031394191177009305, 0.45612661153391881, 
0.37853118216714643, -0.18162628840263459, -0.27372981980503586, 
-0.26821005116410623, -0.38710138067005251, 0.26242970478540351, 
0.31866845203257732, 0.046448496619168, -0.43847322481619766, 
0.040546100924403861, -0.050295542506133065, -0.59473446660899132, 
0.35602762290982193, 0.020591784815428744, 0.021826666832727406, 
-0.70486252432991403, -0.19924261316963351, -0.28814937287330905, 
-0.14162020261661057, -0.064130823864496006, 0.20630526393829829, 
0.033340095533287178, -0.19054020723645487, -0.57078480127562503, 
0.04927740220945076, -0.4080695266296876, 0.48723760370713443, 
0.19768116652354326, 0.090275486786805637, -0.38873670387774978, 
0.79778242107290931, 0.42082377929403614, -0.26128762493357183, 
-0.28046321911990468, -0.682561422328687, -0.44442435143541797, 
-0.25291553618196788, 0.13975400063411866, 0.79594875509798091, 
-0.1789443688608082, -0.39226481158552251, -0.14512130902911372, 
0.3947621500246179, 0.39869306806091709, 0.12961315638964066, 
0.50525219740045135, -0.011289777976984374, 0.29641332386742875, 
0.098245808658231598, -0.23625306311510266, 0.22715536068519246, 
0.14900053725910464, 0.6564856742016425, 0.3360801584935163, 
0.019263702015271209, -0.59296966010187979, 0.12198721924116512, 
0.1954136562534885, -0.15026703037574782, 0.23983425450085044, 
-0.53220952672639266, -0.54463729430481955, -0.33231601223240315, 
0.23314603383750718, 0.39263047560930958, -0.010395251812304692, 
-0.41227438605176103, -0.044546582334041185, 0.080993119445615769, 
0.52753805860472891, 0.32746833169541312, 0.80974196257276543, 
0.42146069904608663, -0.115008660151502, -0.15069048640277313, 
-0.041459095987796256, -0.11120839964455509, 0.23282355886193595, 
0.24889782809624703, 0.050003842738253354, -0.09338491587485348, 
-0.089029815146022012, -0.43513090454245212, -0.9179555047393243, 
-1.0061307116670255, -0.45277138405419115, -0.38426758319771104, 
-0.77114765455867584, -0.89470147934250532, -0.86181821164215588, 
-0.97473307958801014, -0.83250428563533596, -0.50979341244832665, 
-1.2466994579647495, -0.56473787734109804, -1.0873122294409003, 
-0.82307020441670042, -0.66430567205209201, -0.64816614934133276, 
-1.0880128470591783, -1.4468517863281263, -0.65392219492004511, 
-0.5850939934561824, -1.3621098503246889, -0.9813114828958075, 
-0.79551042096994118, -1.0643242106059503, -0.56381953073721203, 
-0.92257270183403683, -1.0070696926486116, -0.84391762752087796, 
-0.36543148705383188, -1.1896044681644404, -1.4504262073109011, 
-0.99776788054787635, -0.72364633797222144, -0.90447037180052758, 
-1.1298293438039122, -1.0239375533934134, -1.3823095348255441, 
-0.68271847682090259, -0.91449380434864491, -0.51374716800970455, 
-1.3512930924448945, -0.96152243400040005, -0.7283095806498272, 
-0.42902943405615268, -0.6359959136617277, -1.0978493251713168, 
-1.411489530127896, -1.0066977022023791, -1.1372610090634609, 
-0.79704168759201688, -0.85425837990020859, -1.2319535437775726, 
-0.96828087711791477, -1.0438283410896849, -0.83798537686450392, 
-0.69404730141697391, -0.63299226848045664, -0.81983021037385451, 
-1.1407647687063336, -1.0437013430231594, -1.1730921412761832, 
-0.88521062059517241, -1.020441055718494, -0.84938097667452328, 
-1.3887050722509429, -0.62785159376262423, -1.3719834708771588, 
-0.068628350356833664, -0.15596689717319229, -1.1239743680975658, 
-1.3891515142445676, -0.97148452752008707, -0.86811698312682484, 
-1.4483161508800955, -0.78002916418138124, -1.1575593449809209, 
-1.2294457238081795, -0.94006996807679433, -0.73784523738372865, 
-0.90620967068727343, -1.3062954855262552, -0.99095298210706428, 
-0.66462213485111987, -1.2269664662998323, -1.2106175357794773, 
-1.4871140171054433, -0.92554731800706458, -0.91350087845476113, 
-1.1996593994901297, -0.45286407134765183, -0.90625466474147476, 
-0.74921404164850336, -0.72913114328209916, -0.54210173494361213, 
-0.61871560530693359, -0.94063199511407847, -0.97821052832275934, 
-0.76333535826645171, -0.70615525114126576, -1.1298787465237248, 
-2.3235860236984944, -1.5836997771307131, -1.7991658759518507, 
-2.1084596815398688, -1.8851804409896014, -2.3090675100038953, 
-2.1114851324625081, -1.9465966732280768, -1.8769657687537389, 
-2.1716127637133309, -1.9541586843036476, -2.0282686329166584, 
-2.048178490507496, -1.9581425046184686, -2.042916498670285, 
-1.7084726158024464, -2.0084633468743975, -2.229668119836191, 
-1.9915018410235958, -2.3942841037068257, -2.9761415391462975, 
-2.0805581367316632, -1.8855850882283489, -1.6898457527644, -2.3314203872254313, 
-1.4933620048960017, -2.1222118096380642, -2.0927832015816326, 
-1.8278379730388057, -1.9036797581539735, -1.8993054121845041, 
-1.9542863435986353, -1.5138722180870388, -2.0336987815681922, 
-1.8449264147300735, -1.4629785665124018, -1.7834020584536949, 
-1.8791070808142019, -1.820847402279937, -2.2893801293893858, 
-1.9924640526675044, -1.8763863403318537, -1.671436662180166, 
-1.9248072818358228, -2.0977300650376209, -2.4928698611881503, 
-1.8192914562304066, -2.3784862101839903, -1.7100540405754092, 
-2.176522772472953, -2.0413125382663386, -1.8866139216295261, 
-2.466552528532131, -2.1793716366325326, -1.7496178806986202, 
-2.5781612193883334, -1.2165111229497974, -2.249376067200934, 
-2.2078442368211197, -2.3945649060452019, -2.0132328094915373, 
-1.6869121705477321, -1.5101242330135576, -2.2966460496833672, 
-1.460307322974864, -1.7840744096072982, -2.0582914055840971, 
-1.714431509104992, -1.1324907724889792, -2.8335124656386457, 
-2.4556172159832013, -1.906939823338363, -1.3306961780304596, 
-2.1370246523696133, -2.5921128320535236, -1.9349535163196716, 
-1.4828580464722894, -1.6799410398355326, -1.5554066913925317, 
-2.2105825257184564, -1.8711941295904604, -1.9491167154629121, 
-1.4374849872529289, -2.3868628899265243, -1.5025060823892051, 
-2.2965099581093757, -2.0185769173025547, -2.4612529105437484, 
-1.7519166688559755, -1.9161702005216861, -1.9281002596926866, 
-2.1429347416782778, -2.2068966537291819, -2.2181763524932294, 
-2.1376745396510093, -1.6908238000096856, -2.0384250100643602, 
-1.586127064487342, -2.5336621319668033, -1.4393205305316139, 
0.21193225480837807, 0.31438559840035479, 0.27569700695756849, 
0.30722391426419998, 0.30002967468042652, 0.33867069645366316, 
0.34130560301586332, 0.28032369073482105, 0.29754481492999646, 
0.29699637419471192, 0.17756153306217201, 0.32243169865294452, 
0.29134381690034805, 0.21304023857705384, 0.21085082909546893, 
0.26563873819886685, 0.11642435252857224, 0.45883312832404516, 
0.21180567398340228, 0.22459316549207076, 0.42487819399321269, 
0.2822002319758683, 0.11362086133416426, 0.28215681202509379, 
0.16686259761153663, 0.32728428141918914, 0.28235248406197389, 
0.18727490281584358, 0.2645546448327743, 0.29185794056272707, 
0.27178763004568635, 0.31916567645412097, 0.41261548779084312, 
0.37255901918496515, 0.26566404340784916, 0.26598735782563693, 
0.21502007778267204, 0.31986908864880936, 0.17304477869081122, 
0.13137397506276513, 0.17666283141653527, 0.22559196667987622, 
0.20229080291402846, 0.40906089279004948, 0.024415060002519039, 
0.36914826919960647, 0.31743233725673126, 0.42439905647335768, 
0.10972671840390014, 0.26447454724273772, 3.5184426647747039e-05, 
0.31249801019031392, 0.2803864130968724, 0.33452772695263427, 
0.29354131685856005, 0.31458287548312597, 0.21785875812734457, 
0.3015744857676228, 0.35601055964111744, 0.37956131800647935, 
0.18527793355254804, 0.33981780487114965, 0.26838110326204012, 
0.39444194470058719, 0.41058489413212357, 0.21636034933064019, 
0.2748763923871691, 0.26454641336304086, 0.22048769013658762, 
0.43984258043754476, 0.32637455749949867, 0.35049142476066902, 
0.30579203623954732, 0.41111758321483377, 0.28022398709066976, 
0.38303371428273747, 0.31182740527457686, 0.26677020348147218, 
0.3538273242038566, 0.38022940165313562, 0.30137874271014919, 
0.3850722808558622, 0.30184322138335301, 0.34518486333559134, 
0.19181407068713799, 0.22784690611485914, 0.41046571184486952, 
0.18207488111545961, 0.35883585516751126, 0.30451409816679403, 
0.26676252339302997, 0.065188698711018511, 0.30460908733505265, 
0.30055107179283358, 0.4078552750568924, 0.24975509382071742, 
0.26613397393374122, 0.25548214336368741, 0.25856543123148362, 
0.12673467494529422), algo = structure(c(2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), .Label = c("mtram", 
"tramME"), class = "factor"), which = structure(c(1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", 
"beta2", "beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-800L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
)



## ----sim-mtram_c, echo=FALSE, eval=FALSE--------------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## 
## beta <- beta[-1]
## Ui <- matrix(1, ncol = 1, nrow = Ni)
## S <- tau^2
## Sigma <- S * tcrossprod(Ui) + diag(Ni)
## D <- diag(sqrt(diag(Sigma)))
## 
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## 
## for (i in 1:nrow(rmt)) {
##   Z <- rmvnorm(N, sigma = Sigma)
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   h1 <- function(x) qt(plogis(x), df = 3)
##   # ^^ <- h^{-1}
##   y <- h1(c(D %*% qlogis(pnorm(solve(D) %*% t(Z)))) + x %*% beta)
##   d <- data.frame(y = y, x, cls = cls)
##   m <- Colr(y ~ X1 + X2 + X3, data = d)
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
##   rmt[i, ] <- c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                 diag(solve(mt$Hessian))[-(1:7)])
## 
##   mm <- ColrME(y ~ X1 + X2 + X3 + (1 | cls), data = d)
##   rME[i, ] <- c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)])
## 
##   lmt[i] <- logLik(mt)
##   lME[i] <- logLik(mm)
## 
##   print(i)
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"),
##                                          each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 2)))


## ----sim-mtram_c_plot, echo=FALSE---------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(-beta, tau)

coefs <-
structure(list(coef = c(-0.075342911213530761, 0.18173089830214081, 
0.42116017894588148, 0.12322505624121982, 0.32963563824453457, 
0.47265255528749445, 0.20106072273144893, -0.71070992372788677, 
-0.071930178560310531, -0.21960382913238874, -0.1217721730721758, 
-0.35079334333513551, -0.52883304327157565, 0.11930753842170615, 
-0.28536920754046452, 0.071896288104446746, -0.59051549037584117, 
-0.093342999122686607, -0.005970335972299237, -0.19950342387572714, 
0.065434511789064162, 0.22638502866556387, -0.21803881035083419, 
-0.097790278751520451, -0.26747141261560453, -0.26353564760779813, 
-0.08431802347762303, -0.11246464827072403, 0.069814189427347267, 
0.1920229623021743, -0.66659168128117885, 0.31840352437900032, 
0.069009414234189423, -0.35648413453446204, 0.25948480511054606, 
-0.38998665112163683, -0.056744184356678114, 0.15834868750203318, 
0.080966571067068296, -0.16265659378770944, -0.13172755921689744, 
-0.077937928170704093, -0.31498487498111311, 0.039002140249084119, 
-0.16943037456571824, 0.22303449506710896, -0.067335378248570388, 
-0.51244030697921628, 0.16898269599081753, -0.43912748741796387, 
0.08984472569005969, -0.095684509221854189, -0.11012702752008602, 
0.067131134916065205, 0.0391225132339236, -0.12959263894474832, 
0.057706716380406649, -0.055668621923765357, 0.24942077586901076, 
0.13304428186207531, 0.33448479644671125, 0.19665016179957043, 
-0.1917119671129438, 0.10418156886303893, -0.27533615738004547, 
-0.098381635135750253, 0.2870007139998127, 0.38692506781210662, 
-0.061136358888231684, -0.13169900633705353, 0.34410585423466544, 
0.45904248357340666, -0.10758412085429851, 0.58824925665928829, 
0.070872313582178256, 0.38754499077216925, -0.28826544762406853, 
-0.292697906415197, -0.09313014358980129, 0.35699661545104061, 
0.086301328863217674, 0.06562855622721897, -0.12039674055832397, 
-0.25269378541000059, 0.01103787898823726, -0.32064444144745979, 
-0.23653801366657706, -0.2968017717922482, -0.28010781580784527, 
0.17559863873910925, 0.13649853934089237, -0.28174018696487479, 
-0.45934836250389144, -0.25528808731077901, 0.063611480765709466, 
-0.25020309800628987, 0.15714293395618559, 0.48516556635817554, 
0.31449200408409544, 0.16530829573809253, -1.2801982960079512, 
-0.99981525629320134, -1.0376667677951958, -1.3299284506655311, 
-1.5080781539620991, -1.1646930158234559, -1.0516984041393433, 
-1.1081341185029174, -0.91599817736305134, -1.3456474348441578, 
-1.2380612226034666, -0.97209206371872192, -1.2027665903253957, 
-0.94376615798717456, -1.0887930123603493, -1.1141648058225886, 
-1.6164618055874163, -1.3537562588430245, -1.3399043434293283, 
-1.0699581519304349, -0.91042999545926229, -0.78768995009971188, 
-0.67533690817798531, -1.1345113168340257, -0.76537270566114546, 
-1.1771570142990806, -0.55319652208095482, -1.2023735309840222, 
-0.68157365560318817, -0.64096878918032063, -1.0453837915451094, 
-1.0607950019586572, -1.2219206277063734, -0.99773700229509543, 
-1.2981855193652687, -1.3638073536928477, -0.64041429526665417, 
-1.0245835555974694, -1.068527945232131, -0.99060357718238834, 
-1.1591165138443076, -1.0600599125449204, -0.97804468528322874, 
-0.51946314591124854, -0.70783550229233494, -0.98556371247950469, 
-1.2948547023995598, -0.88959983330890713, -0.87511637524724395, 
-0.7500553903623941, -0.91172937890369088, -0.84080326942272554, 
-0.9918054874718083, -1.1718500260149693, -1.0570407127048629, 
-0.70547944814461494, -0.74470661151095674, -0.98786915900572758, 
-0.82171669675698333, -1.425909577480307, -0.77953876672074052, 
-1.1109034392286226, -1.3036854477783026, -1.0188370023547479, 
-0.92817121053660467, -0.86765748323584801, -0.93576606924099681, 
-0.40011734063899551, -0.85088865299592786, -1.4337637837367661, 
-1.0002133413957524, -1.0830419960125683, -0.62310860167613424, 
-1.1010749488142071, -0.88690320879024542, -1.5127400821438839, 
-0.84482319324549349, -0.89961004769686692, -0.97306857414109516, 
-0.93380199496079153, -1.2104245157523201, -1.225995893661217, 
-0.82144584066108106, -0.95209240683417706, -0.71670821609383761, 
-1.0744883580356162, -1.1629777244697934, -0.84227662676782966, 
-0.84107450318980481, -0.63646273763143113, -1.420255012291554, 
-0.90450275104504008, -1.3325373774545541, -0.90692106367967817, 
-0.93773703307347145, -1.3313296722112686, -1.3189028012181359, 
-0.91692914161189853, -0.97376499584642784, -0.79261208825131402, 
-2.0991817072338668, -2.2244359729219987, -2.1370505322397437, 
-1.8670621477980305, -1.7075324416315911, -1.6569146732681708, 
-2.298008909739929, -2.3882397657987546, -2.265995551785069, 
-1.3326123988225349, -1.6153699472306167, -2.0776900516608805, 
-1.8598097350427192, -1.9748881305239676, -2.0746140994700797, 
-2.014919090191611, -2.4000009120967745, -1.7331477083879125, 
-2.0853200599217079, -2.5476027733941105, -1.7466841529387411, 
-1.7278676753424551, -2.1976280317736014, -1.6673452373155926, 
-2.1933385318601402, -1.9942799247538845, -1.1137583662176498, 
-2.084966757199056, -1.8866196140127196, -2.1642145344470434, 
-2.0478079552211343, -1.7943073060848265, -2.3633760115170457, 
-1.9021584138252277, -1.9267501205978568, -2.1276898225965533, 
-1.7347204246059722, -1.8235560922343004, -2.1256032051731446, 
-1.5618884090581722, -2.6921824077122722, -2.123931801751302, 
-2.2920142427755303, -2.0021355222038708, -1.8653084465306082, 
-1.9932130965003294, -2.043068044249758, -2.2016569649960616, 
-1.703847422108375, -2.0366147346540866, -2.0185169099379903, 
-2.0266950912763111, -2.2196119459806729, -2.5322558188347877, 
-2.1728653665356523, -2.0051771089364592, -1.7694539648495407, 
-2.0949981195587593, -1.6176442553320864, -2.0782138409721544, 
-1.8041582029302718, -1.7545027113637059, -2.3601439244474385, 
-2.386840427304215, -1.3997551256369589, -2.2615736286880703, 
-2.1387602968505557, -1.6320694456503699, -2.0207590915925424, 
-1.9271526557532628, -1.92462287649048, -1.9009307312734225, 
-1.8950687456685011, -2.0614709041825061, -2.477182921259677, 
-2.030761701754618, -2.6782906494848047, -1.9645100024570028, 
-2.6967157398376034, -2.3289103729178868, -2.4637232740196366, 
-2.128431960142064, -1.8166582220565182, -1.8782151829053355, 
-1.6689962128748435, -2.2558506607979005, -2.5961538780682041, 
-1.6988939917887014, -2.2213660102497341, -2.2255345031581317, 
-1.7874772251956239, -2.041976450417105, -2.3668682108547276, 
-1.6659469859216833, -1.9880862463417728, -1.9428612838582422, 
-1.5676639652863762, -1.8207422448293766, -1.9864079252313291, 
-2.2884760280664431, 0.83292503683771368, 0.76096668019839797, 
0.86078019900118319, 1.105621202509621, 0.92701764219362104, 
0.89097185646477317, 0.97974535990203404, 1.2634281642496243, 
0.93247482357678413, 0.88864038839442883, 1.2013283564888877, 
0.90171678653287179, 0.86246373030273482, 0.83452393596233532, 
0.86584074443778147, 0.82639745696500455, 0.83198517854508347, 
0.87906742473470711, 0.94973964564022539, 1.0294494678495476, 
0.80595122976652922, 0.96159572519184922, 1.0005042618568221, 
0.96068259280420643, 0.83364365324837397, 0.93915121237332178, 
1.142036981733799, 1.0370198865911091, 0.80336671611372845, 0.87050178432998138, 
0.96889323354128509, 0.76218016606223793, 0.86127190160897116, 
0.8691282555412061, 0.81412351564147745, 0.82570187305751974, 
0.79104383010268309, 0.88378356894737697, 0.59499764680704581, 
0.67105601322064856, 0.96744165551979588, 1.0664677473872313, 
0.68554222999465364, 0.93001072894557368, 0.90751899161955518, 
0.95334353287045148, 0.82061866640781511, 1.0119292157841109, 
0.77705515986816998, 0.94481627012187908, 0.67282787317647619, 
0.78054792504200632, 0.92010443219558857, 0.62565898480414983, 
0.75349886896183926, 0.9701472042235777, 0.75111676739714828, 
1.0853104088624335, 0.91942404782581211, 1.0408678050283453, 
0.8571756409727328, 0.98885878288988804, 0.5748097510978577, 
0.82610886289844709, 0.8958888704671818, 0.96113919168446194, 
0.78807580976894198, 0.77178695130252062, 0.89348190576769726, 
1.0109871269716584, 0.85571206721832216, 0.95065271574016685, 
0.84159728574396719, 0.91904065392649459, 0.89932262422826992, 
1.030008258730057, 0.88941027126073757, 0.8861217593900419, 1.0007689039034398, 
0.77841551824254707, 0.90642568993721173, 0.89284788218261268, 
0.93465436612859576, 0.40624587943745888, 1.1183167328810764, 
0.9232139261173874, 0.9833753829677383, 0.71782124049017215, 
1.0191750378536255, 0.94550969199694734, 0.96243917520787359, 
0.70749816886386863, 0.99510955633725828, 0.73053264264383155, 
0.877894869253803, 0.78279958226946345, 0.68231959827215893, 
0.95231707102927832, 0.73146844717205839, 0.7882962561595861, 
-0.045018308700379178, 0.22546723475291502, 0.41699200495275435, 
0.091631942957386636, 0.32053320930640616, 0.42831772427506959, 
0.21369346588021865, -0.6752446069553909, -0.11452754443327684, 
-0.25560222989063702, -0.049902354544591361, -0.34734443145446586, 
-0.48223048359538478, 0.1259578890525565, -0.27957521885714698, 
0.05588924902120037, -0.60396952090531975, -0.11845569849785484, 
-0.0086236687427442699, -0.17368281023686069, 0.10412142691185279, 
0.26462500418392076, -0.20800084374537961, -0.078145030192433351, 
-0.26174437142720813, -0.21879571343403167, -0.1208634451862752, 
-0.10753222760331919, 0.026742699379275696, 0.20341695725891046, 
-0.63067807721118929, 0.29093392633292037, 0.026327301383287191, 
-0.39457696122984454, 0.28560496278669661, -0.44266842934874956, 
-0.0018091175398459471, 0.11779758302881607, 0.07082823044383324, 
-0.14063069471640183, -0.096323190574896037, -0.089483291353649574, 
-0.31729752922365823, 0.089784797831000129, -0.13739283234327684, 
0.32037315100438452, -0.01882439655260992, -0.52288010768466253, 
0.14210130733647819, -0.42422197397324518, 0.11545729507895219, 
-0.10490456370916014, -0.017757659784469325, 0.091354630783472557, 
0.022633545909142429, -0.078504360439192511, 0.043534997996305551, 
-0.13090655923809499, 0.26656203197307954, 0.10867095529768113, 
0.32139730218049706, 0.20176521276438558, -0.16824182330186069, 
0.13026627785633682, -0.28258606448666529, -0.13282281947196661, 
0.30810148969376017, 0.37808371755922532, -0.065744560882790612, 
-0.19655764825999672, 0.24438377092557251, 0.44230855185549295, 
-0.10229632371598887, 0.61183710223975274, 0.074149248147134417, 
0.36791339725584088, -0.31857191122988771, -0.33818636866421775, 
-0.095305473650238268, 0.34854679213165479, 0.11922478217605685, 
0.087231185844086845, -0.14487708319006717, -0.24724052405527366, 
0.082107697766630766, -0.32533231426596199, -0.27358817521911294, 
-0.31705599433765785, -0.25518852829721894, 0.17736558164410124, 
0.11398261820162281, -0.28628185143176044, -0.47345556117788479, 
-0.22642243605003176, 0.11609522312818059, -0.24773453024183584, 
0.15079917765883233, 0.40650964385828403, 0.31357520021171459, 
0.18960788930803907, -1.3324818011585335, -1.0386176560171307, 
-1.0399409118331755, -1.3161987269927364, -1.4710506748259125, 
-1.0987719935799201, -1.0718222939866733, -1.1044064569386793, 
-0.99923876128126077, -1.2772013235918991, -1.2664349878919081, 
-0.88209474661697285, -1.227892767155536, -0.89506490494273061, 
-1.0498029248575202, -1.1411049201577539, -1.6351361912343065, 
-1.3180600919272438, -1.330888103161002, -1.0743961455727296, 
-0.88660031195375755, -0.79515338110334566, -0.6641429359068598, 
-1.0846159583240618, -0.8377959681448498, -1.1762319884506423, 
-0.53085262253933696, -1.1916198061976584, -0.67635636337031169, 
-0.69507740379281968, -1.0901600285461088, -1.0453919081728842, 
-1.2397121620271494, -0.98955536687325807, -1.3479336510692443, 
-1.3846787479466318, -0.63638326642769893, -1.0007717648321992, 
-1.0713077280970327, -1.0267970678045333, -1.1612415391462407, 
-1.0071574691346408, -0.96135701607783763, -0.51847835072023518, 
-0.7014780518367254, -0.97932300654471527, -1.2885543205535159, 
-0.93384852559780063, -0.90186891261070923, -0.77082080695801714, 
-0.89191628005655998, -0.83161254178217403, -1.0579220041762123, 
-1.1711434357339678, -1.0104558173374176, -0.77320979214953445, 
-0.71556339294122284, -0.98963889342241507, -0.77273606102323866, 
-1.4567107480988917, -0.76738246364302554, -1.1288547187625066, 
-1.292132693359785, -0.95039417739918142, -0.89745174575960696, 
-0.87834657820136286, -0.97318101292043391, -0.41965251320604624, 
-0.87613536368812894, -1.3239854960740003, -1.0371672817316933, 
-1.1122474628915275, -0.63891327070044057, -1.0596318264853877, 
-0.9809620227727528, -1.4715154928531069, -0.8562654652206767, 
-0.91458786951144933, -0.92937138205976422, -0.9686252329762477, 
-1.2558581126728789, -1.2380348626553139, -0.83666781096499143, 
-0.94037371072000631, -0.73318967333401563, -1.0011197758079537, 
-1.1380679443958923, -0.84401786777259613, -0.8840605219667288, 
-0.67134250842400689, -1.4188579253344078, -0.87151927532816487, 
-1.337737375873189, -0.91749032184519597, -0.89433365503293005, 
-1.3913729416780665, -1.3177563501187468, -0.88063385110741588, 
-0.99410145994921928, -0.77143871638545902, -2.0678762985073593, 
-2.2751022687414695, -2.1323690947840492, -1.8135663531103536, 
-1.7866527217539316, -1.6397014387782856, -2.344601071419782, 
-2.3491196896216797, -2.2789126778559496, -1.2976923238288238, 
-1.5820086423172051, -2.0057966397696645, -1.8801376454684429, 
-2.0492197270007861, -2.0616166409362071, -1.9939836679022007, 
-2.4441355740649433, -1.6996352947848106, -2.0909395519135834, 
-2.533322306217813, -1.7572400825157992, -1.6892719392407243, 
-2.0799019059031338, -1.705517010127608, -2.2198223148265455, 
-1.9701710887915791, -1.0680435593339461, -2.110408350596209, 
-1.8599818839432494, -2.1913403787302941, -2.0468995894117445, 
-1.8263639483971765, -2.370788107845049, -1.8806351737558133, 
-1.9519305756601855, -2.1652687487557265, -1.7810264256089923, 
-1.8115698973761238, -2.1286958130078073, -1.5611666298480342, 
-2.6916595053614172, -2.1148263899471234, -2.3120509321425664, 
-1.9863731943809839, -1.8396359005571583, -2.0570370605690291, 
-2.0535040700506526, -2.2317090715021375, -1.7650383195681145, 
-1.9997073804003835, -2.0269591459954581, -2.0638794322577541, 
-2.0190652871685182, -2.5080765971474412, -2.1466976451197555, 
-1.990668297056237, -1.766752501702912, -2.1908497984322848, 
-1.6017616174398777, -2.0642180752113806, -1.8114822285771, -1.7659035328985984, 
-2.3658623686280462, -2.3390956664117311, -1.4030055769836216, 
-2.2635260226447103, -2.0949424644678918, -1.6487344346200425, 
-2.015876253294191, -1.8929787489439218, -1.9332550975777341, 
-1.9049526667704317, -1.895220920846483, -2.0126028373753977, 
-2.4357868467753687, -2.0734467600721689, -2.6696596208472512, 
-2.0358273466669421, -2.6983235083067463, -2.3439732140836274, 
-2.4966551745273886, -2.1863695042067803, -1.7752559377778545, 
-1.8999726559719945, -1.7201188294787482, -2.19020679833329, 
-2.5954745430843436, -1.7194252353146438, -2.2062300560967367, 
-2.2712052777711089, -1.7677309720516787, -2.0530843567242765, 
-2.3420870540556087, -1.6657865193819148, -1.9438216419413115, 
-1.9837958070909307, -1.5566078249531681, -1.8164957941846107, 
-2.0228993045400712, -2.2681845997372738, 0.50341323239965496, 
0.4775484133087341, 0.51015350213239452, 0.63293458163423866, 
0.5493763697664954, 0.48971666875568542, 0.5832200244343112, 
0.71131804227680784, 0.57770720593609037, 0.50411663195969536, 
0.71612806870186474, 0.51089903173153994, 0.49784908914888271, 
0.48485292095493487, 0.51670433121046089, 0.4923820056242536, 
0.50920244381752566, 0.51184874924855173, 0.56316494583547705, 
0.61248251621092864, 0.51021225393216152, 0.57621759005058837, 
0.63407245002337798, 0.56547336087893452, 0.50602518813991393, 
0.56935436440897169, 0.66749546362744605, 0.61527870510450045, 
0.48457580056104105, 0.52642233359337309, 0.57968185547981499, 
0.47587437919423387, 0.52852134841917953, 0.49955800630521602, 
0.50126333914314325, 0.48251753344570203, 0.4911316659695848, 
0.50988869735988518, 0.35303241910523814, 0.41744470720196325, 
0.57749416713000401, 0.61997966042109887, 0.41741041203602508, 
0.53635124635203268, 0.52153753594162233, 0.56705369510260639, 
0.49789306074191891, 0.61399701835409082, 0.45887586054892632, 
0.55036842656356622, 0.39099186052573964, 0.49229110984915436, 
0.4424719880200283, 0.38243569073727562, 0.4371081620673774, 
0.58227130625558665, 0.46545669761333097, 0.66592414795087362, 
0.55040240722686873, 0.59143173520780157, 0.49973409084180775, 
0.58865539921326471, 0.34041356008346108, 0.49040235249640135, 
0.5268007437309129, 0.58823751958260817, 0.47447110880346893, 
0.45953084424993657, 0.5166671285806792, 0.59006767769625634, 
0.5080043622346857, 0.55178524403847884, 0.52050296696169385, 
0.53524886124928084, 0.54355703288938051, 0.597672500947656, 
0.51400239688283378, 0.52309854604115846, 0.5911128511601188, 
0.47765622273406816, 0.55127583436304273, 0.52498920882413602, 
0.54006981914614949, 0.2487840956127885, 0.68425058470150368, 
0.54905648620129799, 0.58016722588925773, 0.43240430237479061, 
0.59923622200412818, 0.53693550695003178, 0.57150704612445258, 
0.41786984215618472, 0.58922900757321561, 0.44940779499781852, 
0.49944839610957198, 0.47408852638817062, 0.41322564504492137, 
0.56637572326012375, 0.46070847421523919, 0.46959270415518212
), algo = structure(c(2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L), .Label = c("mtram", "tramME"), class = "factor"), 
    which = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", 
    "beta2", "beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-800L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
)


## ----sim-convert_IC1, eval=FALSE----------------------------------------------
## d$yS <- Surv(floor(y), ceiling(y), type = "interval2")


## ----sim-convert_IC10, eval=FALSE---------------------------------------------
## d$yS <- Surv(floor(10*y)/10, ceiling(10*y)/10, type = "interval2")


## ----sim-probit_IC1, echo=FALSE, eval=FALSE-----------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## beta <- beta[-1]
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## 
## for (i in 1:nrow(rmt)) {
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   y <- qlogis(pnorm(x %*% beta + rnorm(N, sd = tau)[cls] + rnorm(nrow(x))))
##   ##   ^^^^^^^^^^^^^ <- h^{-1}
##   d <- data.frame(y = y, x, cls = cls)
##   d$yS <- Surv(floor(y), ceiling(y), type = "interval2")
##   ### d$yS <- Surv(floor(y), ceiling(y), rep(3, length(y)), type = "interval")
##   m <- BoxCox(yS ~ X1 + X2 + X3,
##               data = d, bounds = c(-Inf, Inf),# add = c(-100, 100),
##               support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
## 
##   ## mtram model
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
##   rmt[i, ] <- c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                 diag(solve(mt$Hessian))[-(1:7)])
## 
##   mm <- BoxCoxME(yS ~ X1 + X2 + X3 + (1 | cls), data = d,
##                  bounds = c(-Inf, Inf),# add = c(-100, 100),
##                  support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   rME[i, ] <- c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)])
## 
##   lmt[i] <- logLik(mt)
##   lME[i] <- logLik(mm)
## 
##   print(i)
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"), each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 4)))
## 


## ----sim-probit_IC1_plot, echo=FALSE------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(beta, tau)

coefs <-
structure(list(coef = c(-0.014452478878961502, -0.040375928280550213, 
0.11977992987096962, -0.29169826188766645, -0.015149440939211653, 
-0.18543856012116877, -0.26733877333872014, 0.20481288483768278, 
-0.22273646136267733, 0.088450544148442223, -0.017437758945688227, 
-0.028203090884514708, 0.22274078595853522, 0.10368575265803986, 
0.09138659429364307, -0.3122788156266752, -0.030237728509437928, 
0.08470236947379714, -0.35845495579159908, 0.065679989634010158, 
0.12524413824233163, 0.033337508373251577, -0.064755752209179673, 
0.15936991276617743, -0.031245862274061582, 0.10673421806597543, 
-0.032423420012913262, 0.044909147473462377, 0.11482497894249105, 
-0.048217590968590983, 0.056301234892995052, 0.17337457411171364, 
0.093449265362625628, 0.37686978781697578, -0.04023231000043849, 
-0.11889243468791984, -0.072638141603353482, 0.080782629366965425, 
-0.19484396682876709, -0.037102350570178862, 0.10378056729472555, 
-0.18267812036442976, -0.050477211244758437, -0.13061218167189259, 
-0.20417824250921512, 0.36661013517303032, 0.23246074554195834, 
0.0010891376462137676, -0.019365786628430345, 0.037129542046862855, 
-0.10690506241407177, -0.038036069813126225, -0.072810613632401197, 
0.02202070566833263, 0.18136139576281762, -0.20404457894596095, 
0.13603708025569916, 0.064263706888130046, -0.1821978298084215, 
-0.19013222300781166, -0.15032335142142916, -0.30282210944190113, 
0.3004705266623996, -0.1348850903458732, 0.13919073733586293, 
-0.14026357944554085, 0.074002842143459549, -0.13789879327534357, 
0.2477666000269059, -0.12248419900805615, 0.18345520268847296, 
0.17975576018684933, 0.11033897122521646, 0.21276534062577626, 
-0.099395440440137414, 0.20586207167625262, 0.15691282035576118, 
-0.15951159208555832, -0.087123030020926598, -0.17320787916996433, 
-0.32311707180923344, -0.25472738269175349, 0.026794343469290733, 
0.034699543209631256, -0.15633910404325724, 0.050832532922236086, 
-0.011827493573507602, 0.092409225046298296, -0.24191840206451443, 
0.014588222850804113, 0.075218365354643962, -0.18608338230971663, 
-0.11488180431846753, -0.15720828094007513, -0.24047422322472981, 
0.29168412972541374, 0.10263250244815003, 0.038162978387134637, 
0.36284198410765173, 0.11541942915053729, 0.84719382725170567, 
0.99751897467800299, 1.1560716188858042, 1.1554531140526809, 
1.0455854043086708, 1.1356024599626295, 1.0255708425697783, 0.8825037561664798, 
1.162908673559585, 1.3066064738816632, 0.8973006015381253, 0.99337691281418483, 
0.94698395229008359, 0.98291068472462906, 0.90528046514612548, 
1.1451536137168776, 0.9146086233212104, 0.61966041086383272, 
0.92096916647440463, 1.2558055893504012, 1.0510835764045887, 
1.2635411565587864, 1.0664981009243908, 1.12376212164273, 1.2529600694447975, 
1.3861512909162674, 0.79005773537444346, 0.92824323164444245, 
1.1612827413929909, 0.94447911606608903, 1.1585240473966718, 
0.78429419067694306, 1.0686113249563554, 1.4124614483583755, 
1.2753737867395087, 0.91341749060752453, 1.2306181388269908, 
1.1122959690746872, 1.26809025080861, 1.0265395879759938, 1.0276903675846025, 
0.92561762263294378, 0.99361061037498211, 1.0117131533027752, 
1.0672698815478221, 1.1975273220560856, 0.97386213406819788, 
0.94688989488449937, 0.90796821791617477, 0.75248801696571932, 
0.98555412462121739, 0.80958438392526311, 0.99379829964862465, 
0.91925254441005144, 0.97108792027018842, 1.0569510094769217, 
0.92614042421041332, 0.91976413083332598, 1.1709362465698694, 
0.83254658013081395, 1.1696550534838355, 0.92435502876040665, 
0.86667460052347145, 1.0433524508797181, 1.0159717689384491, 
0.78840433216845884, 0.85595837049757106, 0.75291837831860298, 
0.8042118539594808, 1.052140340512391, 1.0763600829331719, 1.134562576596138, 
1.3672024679589376, 0.7669287509771876, 0.83598290239948714, 
0.94127649251844103, 0.94666946512100336, 0.65771491268358251, 
1.1750726510322769, 0.84211854465964209, 0.97061388734697518, 
0.91182868981878473, 0.84134342744830992, 1.0409572841700265, 
0.78405926953547911, 0.91959137315822859, 1.0394955512837016, 
0.93493180370200235, 1.0989928033068954, 1.1912681123496522, 
0.89227631824470754, 0.99842103860267695, 1.1369314136826605, 
0.96873037625926728, 0.97655412850062817, 0.81614034858209561, 
0.75474321069632089, 0.86582004940290458, 1.1499783780986161, 
1.2840420133318677, 1.8386772158245523, 2.1581641773341187, 2.0560446185156649, 
2.1353455218940551, 2.1040836610050109, 1.8542671532192891, 2.2278943084401197, 
1.8826416914304427, 2.0829164969005123, 2.2153631997147865, 2.203524130702847, 
1.9273575463669614, 1.8790802224673233, 2.0534328179322876, 1.8101922510359494, 
1.8707735985934351, 2.0587892003595205, 2.2504369627673748, 1.8033390264870381, 
1.9672865085914975, 1.7682343521054167, 1.8281237613877028, 2.1895456354588738, 
2.3973174328100662, 2.1493726209638737, 2.1755081741741957, 1.6590009091949933, 
1.7515193664816906, 2.0125314966879673, 2.0022884339703277, 2.1436370964322555, 
2.0557160864155222, 2.0465567832720608, 1.8042895064172777, 1.9473341558166555, 
2.0474016251530469, 2.2651038311919844, 2.0867166259350327, 1.9306680067314548, 
1.9230969822027073, 2.0926975806616621, 1.7473002169901897, 2.0732343543019436, 
1.8430830788535437, 1.8773784962792661, 2.0182317070985318, 2.0831038223843383, 
2.0521102284393198, 1.9184121895918285, 2.1728130604290108, 1.8298728285674519, 
1.9570013390929371, 2.1476277865012858, 2.1111714084568081, 2.2189584971285181, 
1.8690140311781764, 2.0366860878314754, 1.8122115877364049, 2.0499150763546514, 
2.1704374940983766, 1.8924182347479985, 2.0597486880993179, 2.4007207856299924, 
2.1135499943386313, 1.9570917349270283, 1.8471570573493588, 2.0792908557341989, 
2.0908795590371043, 1.5470429884058314, 2.2066786667510647, 2.0594728281750152, 
2.0575725527716364, 1.9822814517888829, 2.2022492007679593, 2.1838339360918884, 
2.1965452572795607, 1.970141620903618, 1.7497642338536239, 1.921939064770041, 
2.2281645764400886, 1.8285424723809565, 1.9220752132299883, 1.8873763115001414, 
1.9766810905631593, 1.7009929111539606, 2.1757925851161382, 1.9003247878073284, 
1.8808711771883924, 2.1422519629910282, 1.741825872776174, 2.2129268236766944, 
1.9866294913159381, 2.7068810279733082, 2.1180575770554522, 1.849035805920616, 
2.1136632340399224, 2.0029361071153571, 1.6886361416609139, 2.1514171258214763, 
2.0768701871662358, 0.45421145693958387, 0.44473361852262094, 
0.66843622299182481, 0.59083235002109413, 0.50193081792990757, 
0.47131379139581353, 0.53851494620389084, 0.47561291322365284, 
0.53465272689098808, 0.55786079985446979, 0.43831859892610436, 
0.53387584212891959, 0.41414581118953947, 0.60330481435926175, 
0.32968407788552251, 0.54618804428449541, 0.46188390750217362, 
0.55949643493666357, 0.53697914997787743, 0.59242649796710622, 
0.40556981635862066, 0.43302552883317957, 0.5350914899171747, 
0.44482942114552576, 0.49673752860694492, 0.58830169117831732, 
0.56241530162107412, 0.68009368395825631, 0.53579544544963842, 
0.55798219803769222, 0.59033089560106333, 0.60702971808188011, 
0.47462345197908884, 0.53988110449792404, 0.55031693391735625, 
0.45326338496250246, 0.54000314206611644, 0.54278170155126304, 
0.52511299851989535, 0.48636992034526455, 0.53060478595590876, 
0.51165814193163106, 0.47036590725926497, 0.56487881793468908, 
0.52678023707512545, 0.50517956343444093, 0.5401399106770558, 
0.42653797360238593, 0.52753570439055719, 0.44878361781187986, 
0.53142738685137347, 0.54787731289116004, 0.53768323439244259, 
0.53487071766436955, 0.53665633925288581, 0.45316619641510247, 
0.6170757916912103, 0.4733906046191122, 0.56917779444850658, 
0.59681065528565092, 0.58128409672191084, 0.57656938931139878, 
0.64740995434773141, 0.72629493170177639, 0.48613641746145098, 
0.29358026109090479, 0.46928223643117156, 0.58148470121774565, 
0.59821409102416445, 0.44335505969705213, 0.52903550243749464, 
0.57378336330598478, 0.68609950648763696, 0.49076662379666386, 
0.49041077840406255, 0.62598082368484209, 0.58161414742923889, 
0.44868962161555775, 0.71650403394096007, 0.59681044867342958, 
0.49799624751933197, 0.48486315467404906, 0.47272433497527711, 
0.60854767387308129, 0.49022659565491766, 0.57206865204016344, 
0.66734940823403288, 0.50298538795708225, 0.59767955561516739, 
0.5796700790498639, 0.50260253990906401, 0.53460016759988993, 
0.53959727327725904, 0.65780018528040118, 0.38211275763068792, 
0.69154905312877102, 0.43892332257277628, 0.45843553055939296, 
0.55509349281615761, 0.53980506191786248, -0.014372256545807866, 
-0.040571413499099374, 0.11808227167830386, -0.29150452927679832, 
-0.015606348759729965, -0.18550730879883559, -0.26679357599829212, 
0.20094524034547909, -0.22297629602320099, 0.08849191778334714, 
-0.017737557893573782, -0.028030802748205866, 0.22238309212220328, 
0.1038421875739212, 0.090920456476812614, -0.31234418182851409, 
-0.029848047747277059, 0.084550836035644852, -0.35892218426570072, 
0.066132317138602056, 0.12526645049300109, 0.032143228202696758, 
-0.064850025927945795, 0.15891084534941408, -0.031732693654762237, 
0.10561134510882486, -0.033139004323175771, 0.043547312248015099, 
0.11427240787554957, -0.048277265182920699, 0.056937996541818928, 
0.17316750112718307, 0.093004771208675424, 0.37649486112169944, 
-0.040941020262949927, -0.11880099835167318, -0.075777581311729944, 
0.080269727446070263, -0.19539553047907784, -0.037096018932975948, 
0.1039946780932791, -0.18241869098573088, -0.050313944143333726, 
-0.13028135232567631, -0.2058513251492376, 0.36610000797783132, 
0.23233526651092837, 0.00086501796049521704, -0.020215755754760617, 
0.036913818314042737, -0.10715668634301501, -0.039281503821714726, 
-0.070942927115274779, 0.021955989730334698, 0.18010492404115813, 
-0.20268484938801426, 0.13870835201249396, 0.064039665676222685, 
-0.18246295799891099, -0.19077574601056035, -0.14996254746585397, 
-0.30355905030674835, 0.30077506601849402, -0.13630344809923964, 
0.13901295305395869, -0.14084617781270473, 0.074048538913509779, 
-0.14408169590493478, 0.24709091238212208, -0.12285992974452865, 
0.18438777766527664, 0.18021921735002683, 0.11017379281290661, 
0.21274938530824336, -0.099270082582969543, 0.20596779823925365, 
0.15680393285585914, -0.15960054236546067, -0.087869662253262121, 
-0.17205537485731959, -0.32317030416776038, -0.25455009480572205, 
0.026492500192362677, 0.034673651645630348, -0.15446639516591254, 
0.050414652495786055, -0.012113565887522925, 0.092273184619560009, 
-0.2423308077062569, 0.014822218814287868, 0.07467001599287236, 
-0.18579015697586704, -0.11547458328998365, -0.1577056800032394, 
-0.24116588495737348, 0.29159623920467131, 0.10244900508259865, 
0.037308853233420085, 0.36410534983968862, 0.1151296468547269, 
0.8472114497691392, 0.99734366548779474, 1.1481372809581185, 
1.1523165040307122, 1.0455847893207726, 1.1351426356270324, 1.0258851710736228, 
0.88165091791297434, 1.1609127649713549, 1.3065534786692019, 
0.89673793569236582, 0.99307892208024673, 0.94684669273889699, 
0.98290357270082673, 0.90525626197193365, 1.1462475239454455, 
0.91443589661265667, 0.61953382681502678, 0.92062233346877231, 
1.255574954200622, 1.050961699462168, 1.2630709980845622, 1.0662127964784966, 
1.1237107106147406, 1.2525583651854646, 1.3858680967847075, 0.79007661600022905, 
0.92811136351344314, 1.1610970198598878, 0.94445282885661963, 
1.1574400397068663, 0.7841289324891777, 1.0681288442389896, 1.41240388175439, 
1.2754995107862843, 0.91273563752893327, 1.2199413193875064, 
1.1120567121941787, 1.2680091472773105, 1.0265356123454441, 1.0286782864014306, 
0.92496680573813617, 0.99378096149059658, 1.0124605172947929, 
1.063028023026271, 1.1974729275037828, 0.97361331114523131, 0.94681903483258179, 
0.90551744315434035, 0.75239532309176416, 0.98610222856406793, 
0.81338651802122008, 0.99297035549026014, 0.91999868048297795, 
0.97111694231680545, 1.0559042065768611, 0.93332621910908775, 
0.9199239823124199, 1.1715219288895382, 0.83165557019081437, 
1.1686661198374955, 0.92369013704553971, 0.86637711077971546, 
1.0439950225589063, 1.0163816571278919, 0.788285332852807, 0.85566856791149315, 
0.75666678390394326, 0.80380530099248482, 1.052204137892393, 
1.0764626474838659, 1.1338362284488983, 1.367857503575737, 0.76753831913935244, 
0.83572135973116946, 0.94130568822046012, 0.9469868855791238, 
0.65806152972387899, 1.1737340743810487, 0.84224585766714444, 
0.97067825424666476, 0.91123657351702159, 0.84136423277985795, 
1.0409249192249785, 0.78298945605380887, 0.91993777517020936, 
1.0392374602787375, 0.9345508693551674, 1.0983218276195212, 1.1907945457116254, 
0.89245126825392085, 0.99838454317822289, 1.1364085702588682, 
0.96879671559492841, 0.97600569180239771, 0.81623572122573518, 
0.75470175977168763, 0.866781156661664, 1.1499271883657955, 1.2839686271183401, 
1.838890620097835, 2.158427231673453, 2.0487558557120882, 2.1324720764435146, 
2.1034709994051122, 1.8545703679970871, 2.2283082537398338, 1.8786143526137522, 
2.0809930224373452, 2.2155953181914456, 2.2034290233407621, 1.9273153652539121, 
1.8794943733440563, 2.0535187010795828, 1.8099541904589616, 1.8717606997608569, 
2.0588558224359299, 2.2504459174154285, 1.800807159013238, 1.96785560696937, 
1.7678204988605188, 1.8269455206609597, 2.189890078411969, 2.3976714639293566, 
2.1499373780204585, 2.172932067826721, 1.6592398825668857, 1.7518872193461947, 
2.0127658221451767, 2.0024258131622679, 2.1450315254611825, 2.0556712102031218, 
2.0468480411603545, 1.8042147958759096, 1.9473297394172253, 2.0471958241126464, 
2.2598570821029957, 2.08659317981737, 1.931135204356613, 1.9237083521243945, 
2.0938757744546925, 1.74800466068192, 2.0735264827542665, 1.8428111411761161, 
1.8745913679872461, 2.018588559087712, 2.0830267186586293, 2.0522248410241368, 
1.9180506931357095, 2.1731569702571001, 1.8298322847468451, 1.9577670717613209, 
2.1482239656511655, 2.1125072845103854, 2.2188229237714934, 1.866556441579138, 
2.03560158610034, 1.8122511849673026, 2.050215933939326, 2.1698815532913631, 
1.8934485062538318, 2.060128985099698, 2.4004116414874614, 2.1091721136573076, 
1.9571144585044018, 1.8470400927142259, 2.0790296902847158, 2.0886701790538202, 
1.5466833419371171, 2.2067883343472423, 2.0610460206221823, 2.0578948209426229, 
1.9831340261508403, 2.2029654963855081, 2.1837722773442989, 2.196806048172296, 
1.9701857411089498, 1.7497935888079403, 1.9231499218403705, 2.2278253591334485, 
1.8283920494460173, 1.9221008482062587, 1.8869105309638223, 1.9762798327518198, 
1.7001713573139257, 2.1767577891997494, 1.8997096092315211, 1.8820748926883719, 
2.1413613080049378, 1.7420956285653557, 2.2131966165612846, 1.9864831564312366, 
2.7067562334406468, 2.117905357306721, 1.8488739018887523, 2.1134560692937199, 
2.0033092267755741, 1.6878310182824012, 2.152147486254866, 2.0776715269368506, 
0.45569072377511582, 0.44530060114487957, 0.67619157372953798, 
0.59018116509338925, 0.50428223193329735, 0.47267645941924369, 
0.5391831885944286, 0.47688601695683125, 0.53981849650447611, 
0.55790315876569163, 0.43946027304119634, 0.53417732646003224, 
0.41509647108074377, 0.60401288472973891, 0.33022398706578326, 
0.54780650452022051, 0.46269558516469672, 0.55898743329526035, 
0.53771777667337617, 0.59567540675751074, 0.40596346323359056, 
0.43415329324113566, 0.5358318340238083, 0.44595551882808354, 
0.49775737847419349, 0.58737478299945134, 0.56345169160876274, 
0.68264392876407265, 0.53478464562680306, 0.55798755955207469, 
0.59325782492237389, 0.60579232479424061, 0.47712197504969439, 
0.54033680182951904, 0.55260057642980687, 0.45422844627686548, 
0.54204192031589438, 0.54221317550631454, 0.526454598722364, 
0.48645907251968601, 0.53128227421373175, 0.51445700821353779, 
0.47042154660369673, 0.5667167718188747, 0.52993010306577626, 
0.50628861271890258, 0.54012629255070754, 0.42696050568362492, 
0.52927439534156862, 0.44886388394343657, 0.53221682162163575, 
0.54954450304660729, 0.5404026589391947, 0.5366078684230019, 
0.53895355522920285, 0.4559080954778823, 0.62333124237024018, 
0.47395747313181591, 0.56958581719217383, 0.59578111206034134, 
0.5830508348165897, 0.57733618912490425, 0.65010640793860275, 
0.73047606025178669, 0.48681629113873653, 0.29416957910946684, 
0.46975919020370316, 0.58579679104853521, 0.59808607554430071, 
0.44365454555271366, 0.53223636244353711, 0.57631316128693932, 
0.68683219322847822, 0.49207292375133543, 0.49093275803577613, 
0.62747476343833253, 0.58142970560240348, 0.44953933094399529, 
0.72514960453191679, 0.60014993602723188, 0.49728520250531827, 
0.48534113884400143, 0.47206690786280348, 0.60808292253566032, 
0.4918557837853918, 0.57507745769771379, 0.66560555140774913, 
0.50411105057116468, 0.59787920133829675, 0.58147880900799553, 
0.50378789701505433, 0.53640613598232267, 0.54051596893712794, 
0.65727625164653303, 0.38224593843452809, 0.68968253716372885, 
0.44000606435399153, 0.46044030405352304, 0.55651360523950233, 
0.54086237860781317, -0.014452478878961502, -0.040375928280550213, 
0.11977992987096962, -0.29169826188766645, -0.015149440939211653, 
-0.18543856012116877, -0.26733877333872014, 0.20481288483768278, 
-0.22273646136267733, 0.088450544148442223, -0.017437758945688227, 
-0.028203090884514708, 0.22274078595853522, 0.10368575265803986, 
0.09138659429364307, -0.3122788156266752, -0.030237728509437928, 
0.08470236947379714, -0.35845495579159908, 0.065679989634010158, 
0.12524413824233163, 0.033337508373251577, -0.064755752209179673, 
0.15936991276617743, -0.031245862274061582, 0.10673421806597543, 
-0.032423420012913262, 0.044909147473462377, 0.11482497894249105, 
-0.048217590968590983, 0.056301234892995052, 0.17337457411171364, 
0.093449265362625628, 0.37686978781697578, -0.04023231000043849, 
-0.11889243468791984, -0.072638141603353482, 0.080782629366965425, 
-0.19484396682876709, -0.037102350570178862, 0.10378056729472555, 
-0.18267812036442976, -0.050477211244758437, -0.13061218167189259, 
-0.20417824250921512, 0.36661013517303032, 0.23246074554195834, 
0.0010891376462137676, -0.019365786628430345, 0.037129542046862855, 
-0.10690506241407177, -0.038036069813126225, -0.072810613632401197, 
0.02202070566833263, 0.18136139576281762, -0.20404457894596095, 
0.13603708025569916, 0.064263706888130046, -0.1821978298084215, 
-0.19013222300781166, -0.15032335142142916, -0.30282210944190113, 
0.3004705266623996, -0.1348850903458732, 0.13919073733586293, 
-0.14026357944554085, 0.074002842143459549, -0.13789879327534357, 
0.2477666000269059, -0.12248419900805615, 0.18345520268847296, 
0.17975576018684933, 0.11033897122521646, 0.21276534062577626, 
-0.099395440440137414, 0.20586207167625262, 0.15691282035576118, 
-0.15951159208555832, -0.087123030020926598, -0.17320787916996433, 
-0.32311707180923344, -0.25472738269175349, 0.026794343469290733, 
0.034699543209631256, -0.15633910404325724, 0.050832532922236086, 
-0.011827493573507602, 0.092409225046298296, -0.24191840206451443, 
0.014588222850804113, 0.075218365354643962, -0.18608338230971663, 
-0.11488180431846753, -0.15720828094007513, -0.24047422322472981, 
0.29168412972541374, 0.10263250244815003, 0.038162978387134637, 
0.36284198410765173, 0.11541942915053729, 0.84719382725170567, 
0.99751897467800299, 1.1560716188858042, 1.1554531140526809, 
1.0455854043086708, 1.1356024599626295, 1.0255708425697783, 0.8825037561664798, 
1.162908673559585, 1.3066064738816632, 0.8973006015381253, 0.99337691281418483, 
0.94698395229008359, 0.98291068472462906, 0.90528046514612548, 
1.1451536137168776, 0.9146086233212104, 0.61966041086383272, 
0.92096916647440463, 1.2558055893504012, 1.0510835764045887, 
1.2635411565587864, 1.0664981009243908, 1.12376212164273, 1.2529600694447975, 
1.3861512909162674, 0.79005773537444346, 0.92824323164444245, 
1.1612827413929909, 0.94447911606608903, 1.1585240473966718, 
0.78429419067694306, 1.0686113249563554, 1.4124614483583755, 
1.2753737867395087, 0.91341749060752453, 1.2306181388269908, 
1.1122959690746872, 1.26809025080861, 1.0265395879759938, 1.0276903675846025, 
0.92561762263294378, 0.99361061037498211, 1.0117131533027752, 
1.0672698815478221, 1.1975273220560856, 0.97386213406819788, 
0.94688989488449937, 0.90796821791617477, 0.75248801696571932, 
0.98555412462121739, 0.80958438392526311, 0.99379829964862465, 
0.91925254441005144, 0.97108792027018842, 1.0569510094769217, 
0.92614042421041332, 0.91976413083332598, 1.1709362465698694, 
0.83254658013081395, 1.1696550534838355, 0.92435502876040665, 
0.86667460052347145, 1.0433524508797181, 1.0159717689384491, 
0.78840433216845884, 0.85595837049757106, 0.75291837831860298, 
0.8042118539594808, 1.052140340512391, 1.0763600829331719, 1.134562576596138, 
1.3672024679589376, 0.7669287509771876, 0.83598290239948714, 
0.94127649251844103, 0.94666946512100336, 0.65771491268358251, 
1.1750726510322769, 0.84211854465964209, 0.97061388734697518, 
0.91182868981878473, 0.84134342744830992, 1.0409572841700265, 
0.78405926953547911, 0.91959137315822859, 1.0394955512837016, 
0.93493180370200235, 1.0989928033068954, 1.1912681123496522, 
0.89227631824470754, 0.99842103860267695, 1.1369314136826605, 
0.96873037625926728, 0.97655412850062817, 0.81614034858209561, 
0.75474321069632089, 0.86582004940290458, 1.1499783780986161, 
1.2840420133318677, 1.8386772158245523, 2.1581641773341187, 2.0560446185156649, 
2.1353455218940551, 2.1040836610050109, 1.8542671532192891, 2.2278943084401197, 
1.8826416914304427, 2.0829164969005123, 2.2153631997147865, 2.203524130702847, 
1.9273575463669614, 1.8790802224673233, 2.0534328179322876, 1.8101922510359494, 
1.8707735985934351, 2.0587892003595205, 2.2504369627673748, 1.8033390264870381, 
1.9672865085914975, 1.7682343521054167, 1.8281237613877028, 2.1895456354588738, 
2.3973174328100662, 2.1493726209638737, 2.1755081741741957, 1.6590009091949933, 
1.7515193664816906, 2.0125314966879673, 2.0022884339703277, 2.1436370964322555, 
2.0557160864155222, 2.0465567832720608, 1.8042895064172777, 1.9473341558166555, 
2.0474016251530469, 2.2651038311919844, 2.0867166259350327, 1.9306680067314548, 
1.9230969822027073, 2.0926975806616621, 1.7473002169901897, 2.0732343543019436, 
1.8430830788535437, 1.8773784962792661, 2.0182317070985318, 2.0831038223843383, 
2.0521102284393198, 1.9184121895918285, 2.1728130604290108, 1.8298728285674519, 
1.9570013390929371, 2.1476277865012858, 2.1111714084568081, 2.2189584971285181, 
1.8690140311781764, 2.0366860878314754, 1.8122115877364049, 2.0499150763546514, 
2.1704374940983766, 1.8924182347479985, 2.0597486880993179, 2.4007207856299924, 
2.1135499943386313, 1.9570917349270283, 1.8471570573493588, 2.0792908557341989, 
2.0908795590371043, 1.5470429884058314, 2.2066786667510647, 2.0594728281750152, 
2.0575725527716364, 1.9822814517888829, 2.2022492007679593, 2.1838339360918884, 
2.1965452572795607, 1.970141620903618, 1.7497642338536239, 1.921939064770041, 
2.2281645764400886, 1.8285424723809565, 1.9220752132299883, 1.8873763115001414, 
1.9766810905631593, 1.7009929111539606, 2.1757925851161382, 1.9003247878073284, 
1.8808711771883924, 2.1422519629910282, 1.741825872776174, 2.2129268236766944, 
1.9866294913159381, 2.7068810279733082, 2.1180575770554522, 1.849035805920616, 
2.1136632340399224, 2.0029361071153571, 1.6886361416609139, 2.1514171258214763, 
2.0768701871662358, 0.45421145693958387, 0.44473361852262094, 
0.66843622299182481, 0.59083235002109413, 0.50193081792990757, 
0.47131379139581353, 0.53851494620389084, 0.47561291322365284, 
0.53465272689098808, 0.55786079985446979, 0.43831859892610436, 
0.53387584212891959, 0.41414581118953947, 0.60330481435926175, 
0.32968407788552251, 0.54618804428449541, 0.46188390750217362, 
0.55949643493666357, 0.53697914997787743, 0.59242649796710622, 
0.40556981635862066, 0.43302552883317957, 0.5350914899171747, 
0.44482942114552576, 0.49673752860694492, 0.58830169117831732, 
0.56241530162107412, 0.68009368395825631, 0.53579544544963842, 
0.55798219803769222, 0.59033089560106333, 0.60702971808188011, 
0.47462345197908884, 0.53988110449792404, 0.55031693391735625, 
0.45326338496250246, 0.54000314206611644, 0.54278170155126304, 
0.52511299851989535, 0.48636992034526455, 0.53060478595590876, 
0.51165814193163106, 0.47036590725926497, 0.56487881793468908, 
0.52678023707512545, 0.50517956343444093, 0.5401399106770558, 
0.42653797360238593, 0.52753570439055719, 0.44878361781187986, 
0.53142738685137347, 0.54787731289116004, 0.53768323439244259, 
0.53487071766436955, 0.53665633925288581, 0.45316619641510247, 
0.6170757916912103, 0.4733906046191122, 0.56917779444850658, 
0.59681065528565092, 0.58128409672191084, 0.57656938931139878, 
0.64740995434773141, 0.72629493170177639, 0.48613641746145098, 
0.29358026109090479, 0.46928223643117156, 0.58148470121774565, 
0.59821409102416445, 0.44335505969705213, 0.52903550243749464, 
0.57378336330598478, 0.68609950648763696, 0.49076662379666386, 
0.49041077840406255, 0.62598082368484209, 0.58161414742923889, 
0.44868962161555775, 0.71650403394096007, 0.59681044867342958, 
0.49799624751933197, 0.48486315467404906, 0.47272433497527711, 
0.60854767387308129, 0.49022659565491766, 0.57206865204016344, 
0.66734940823403288, 0.50298538795708225, 0.59767955561516739, 
0.5796700790498639, 0.50260253990906401, 0.53460016759988993, 
0.53959727327725904, 0.65780018528040118, 0.38211275763068792, 
0.69154905312877102, 0.43892332257277628, 0.45843553055939296, 
0.55509349281615761, 0.53980506191786248, -0.014372256545807866, 
-0.040571413499099374, 0.11808227167830386, -0.29150452927679832, 
-0.015606348759729965, -0.18550730879883559, -0.26679357599829212, 
0.20094524034547909, -0.22297629602320099, 0.08849191778334714, 
-0.017737557893573782, -0.028030802748205866, 0.22238309212220328, 
0.1038421875739212, 0.090920456476812614, -0.31234418182851409, 
-0.029848047747277059, 0.084550836035644852, -0.35892218426570072, 
0.066132317138602056, 0.12526645049300109, 0.032143228202696758, 
-0.064850025927945795, 0.15891084534941408, -0.031732693654762237, 
0.10561134510882486, -0.033139004323175771, 0.043547312248015099, 
0.11427240787554957, -0.048277265182920699, 0.056937996541818928, 
0.17316750112718307, 0.093004771208675424, 0.37649486112169944, 
-0.040941020262949927, -0.11880099835167318, -0.075777581311729944, 
0.080269727446070263, -0.19539553047907784, -0.037096018932975948, 
0.1039946780932791, -0.18241869098573088, -0.050313944143333726, 
-0.13028135232567631, -0.2058513251492376, 0.36610000797783132, 
0.23233526651092837, 0.00086501796049521704, -0.020215755754760617, 
0.036913818314042737, -0.10715668634301501, -0.039281503821714726, 
-0.070942927115274779, 0.021955989730334698, 0.18010492404115813, 
-0.20268484938801426, 0.13870835201249396, 0.064039665676222685, 
-0.18246295799891099, -0.19077574601056035, -0.14996254746585397, 
-0.30355905030674835, 0.30077506601849402, -0.13630344809923964, 
0.13901295305395869, -0.14084617781270473, 0.074048538913509779, 
-0.14408169590493478, 0.24709091238212208, -0.12285992974452865, 
0.18438777766527664, 0.18021921735002683, 0.11017379281290661, 
0.21274938530824336, -0.099270082582969543, 0.20596779823925365, 
0.15680393285585914, -0.15960054236546067, -0.087869662253262121, 
-0.17205537485731959, -0.32317030416776038, -0.25455009480572205, 
0.026492500192362677, 0.034673651645630348, -0.15446639516591254, 
0.050414652495786055, -0.012113565887522925, 0.092273184619560009, 
-0.2423308077062569, 0.014822218814287868, 0.07467001599287236, 
-0.18579015697586704, -0.11547458328998365, -0.1577056800032394, 
-0.24116588495737348, 0.29159623920467131, 0.10244900508259865, 
0.037308853233420085, 0.36410534983968862, 0.1151296468547269, 
0.8472114497691392, 0.99734366548779474, 1.1481372809581185, 
1.1523165040307122, 1.0455847893207726, 1.1351426356270324, 1.0258851710736228, 
0.88165091791297434, 1.1609127649713549, 1.3065534786692019, 
0.89673793569236582, 0.99307892208024673, 0.94684669273889699, 
0.98290357270082673, 0.90525626197193365, 1.1462475239454455, 
0.91443589661265667, 0.61953382681502678, 0.92062233346877231, 
1.255574954200622, 1.050961699462168, 1.2630709980845622, 1.0662127964784966, 
1.1237107106147406, 1.2525583651854646, 1.3858680967847075, 0.79007661600022905, 
0.92811136351344314, 1.1610970198598878, 0.94445282885661963, 
1.1574400397068663, 0.7841289324891777, 1.0681288442389896, 1.41240388175439, 
1.2754995107862843, 0.91273563752893327, 1.2199413193875064, 
1.1120567121941787, 1.2680091472773105, 1.0265356123454441, 1.0286782864014306, 
0.92496680573813617, 0.99378096149059658, 1.0124605172947929, 
1.063028023026271, 1.1974729275037828, 0.97361331114523131, 0.94681903483258179, 
0.90551744315434035, 0.75239532309176416, 0.98610222856406793, 
0.81338651802122008, 0.99297035549026014, 0.91999868048297795, 
0.97111694231680545, 1.0559042065768611, 0.93332621910908775, 
0.9199239823124199, 1.1715219288895382, 0.83165557019081437, 
1.1686661198374955, 0.92369013704553971, 0.86637711077971546, 
1.0439950225589063, 1.0163816571278919, 0.788285332852807, 0.85566856791149315, 
0.75666678390394326, 0.80380530099248482, 1.052204137892393, 
1.0764626474838659, 1.1338362284488983, 1.367857503575737, 0.76753831913935244, 
0.83572135973116946, 0.94130568822046012, 0.9469868855791238, 
0.65806152972387899, 1.1737340743810487, 0.84224585766714444, 
0.97067825424666476, 0.91123657351702159, 0.84136423277985795, 
1.0409249192249785, 0.78298945605380887, 0.91993777517020936, 
1.0392374602787375, 0.9345508693551674, 1.0983218276195212, 1.1907945457116254, 
0.89245126825392085, 0.99838454317822289, 1.1364085702588682, 
0.96879671559492841, 0.97600569180239771, 0.81623572122573518, 
0.75470175977168763, 0.866781156661664, 1.1499271883657955, 1.2839686271183401, 
1.838890620097835, 2.158427231673453, 2.0487558557120882, 2.1324720764435146, 
2.1034709994051122, 1.8545703679970871, 2.2283082537398338, 1.8786143526137522, 
2.0809930224373452, 2.2155953181914456, 2.2034290233407621, 1.9273153652539121, 
1.8794943733440563, 2.0535187010795828, 1.8099541904589616, 1.8717606997608569, 
2.0588558224359299, 2.2504459174154285, 1.800807159013238, 1.96785560696937, 
1.7678204988605188, 1.8269455206609597, 2.189890078411969, 2.3976714639293566, 
2.1499373780204585, 2.172932067826721, 1.6592398825668857, 1.7518872193461947, 
2.0127658221451767, 2.0024258131622679, 2.1450315254611825, 2.0556712102031218, 
2.0468480411603545, 1.8042147958759096, 1.9473297394172253, 2.0471958241126464, 
2.2598570821029957, 2.08659317981737, 1.931135204356613, 1.9237083521243945, 
2.0938757744546925, 1.74800466068192, 2.0735264827542665, 1.8428111411761161, 
1.8745913679872461, 2.018588559087712, 2.0830267186586293, 2.0522248410241368, 
1.9180506931357095, 2.1731569702571001, 1.8298322847468451, 1.9577670717613209, 
2.1482239656511655, 2.1125072845103854, 2.2188229237714934, 1.866556441579138, 
2.03560158610034, 1.8122511849673026, 2.050215933939326, 2.1698815532913631, 
1.8934485062538318, 2.060128985099698, 2.4004116414874614, 2.1091721136573076, 
1.9571144585044018, 1.8470400927142259, 2.0790296902847158, 2.0886701790538202, 
1.5466833419371171, 2.2067883343472423, 2.0610460206221823, 2.0578948209426229, 
1.9831340261508403, 2.2029654963855081, 2.1837722773442989, 2.196806048172296, 
1.9701857411089498, 1.7497935888079403, 1.9231499218403705, 2.2278253591334485, 
1.8283920494460173, 1.9221008482062587, 1.8869105309638223, 1.9762798327518198, 
1.7001713573139257, 2.1767577891997494, 1.8997096092315211, 1.8820748926883719, 
2.1413613080049378, 1.7420956285653557, 2.2131966165612846, 1.9864831564312366, 
2.7067562334406468, 2.117905357306721, 1.8488739018887523, 2.1134560692937199, 
2.0033092267755741, 1.6878310182824012, 2.152147486254866, 2.0776715269368506, 
0.45569072377511582, 0.44530060114487957, 0.67619157372953798, 
0.59018116509338925, 0.50428223193329735, 0.47267645941924369, 
0.5391831885944286, 0.47688601695683125, 0.53981849650447611, 
0.55790315876569163, 0.43946027304119634, 0.53417732646003224, 
0.41509647108074377, 0.60401288472973891, 0.33022398706578326, 
0.54780650452022051, 0.46269558516469672, 0.55898743329526035, 
0.53771777667337617, 0.59567540675751074, 0.40596346323359056, 
0.43415329324113566, 0.5358318340238083, 0.44595551882808354, 
0.49775737847419349, 0.58737478299945134, 0.56345169160876274, 
0.68264392876407265, 0.53478464562680306, 0.55798755955207469, 
0.59325782492237389, 0.60579232479424061, 0.47712197504969439, 
0.54033680182951904, 0.55260057642980687, 0.45422844627686548, 
0.54204192031589438, 0.54221317550631454, 0.526454598722364, 
0.48645907251968601, 0.53128227421373175, 0.51445700821353779, 
0.47042154660369673, 0.5667167718188747, 0.52993010306577626, 
0.50628861271890258, 0.54012629255070754, 0.42696050568362492, 
0.52927439534156862, 0.44886388394343657, 0.53221682162163575, 
0.54954450304660729, 0.5404026589391947, 0.5366078684230019, 
0.53895355522920285, 0.4559080954778823, 0.62333124237024018, 
0.47395747313181591, 0.56958581719217383, 0.59578111206034134, 
0.5830508348165897, 0.57733618912490425, 0.65010640793860275, 
0.73047606025178669, 0.48681629113873653, 0.29416957910946684, 
0.46975919020370316, 0.58579679104853521, 0.59808607554430071, 
0.44365454555271366, 0.53223636244353711, 0.57631316128693932, 
0.68683219322847822, 0.49207292375133543, 0.49093275803577613, 
0.62747476343833253, 0.58142970560240348, 0.44953933094399529, 
0.72514960453191679, 0.60014993602723188, 0.49728520250531827, 
0.48534113884400143, 0.47206690786280348, 0.60808292253566032, 
0.4918557837853918, 0.57507745769771379, 0.66560555140774913, 
0.50411105057116468, 0.59787920133829675, 0.58147880900799553, 
0.50378789701505433, 0.53640613598232267, 0.54051596893712794, 
0.65727625164653303, 0.38224593843452809, 0.68968253716372885, 
0.44000606435399153, 0.46044030405352304, 0.55651360523950233, 
0.54086237860781317), algo = structure(c(2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), class = "factor", .Label = c("mtram", 
"tramME")), which = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", "beta2", 
"beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-1600L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]),
                                              expression(beta[2]),
                                              expression(beta[3]),
                                              expression(gamma)))
)


## ----sim-tramME_IC1, echo=FALSE, eval=FALSE-----------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## beta <- beta[-1]
## rmt <- rmtT <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lmtT <- lME <- numeric(nrow(rmt))
## res <- vector("list", Nsim)
## 
## aa <- function(i) {
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   y <- qt(plogis(x %*% beta + rnorm(N, sd = tau)[cls] + rlogis(nrow(x))), df = 3)
##   ##   ^^^^^^^^^^ <- h^{-1}
##   d <- data.frame(y = y, x, cls = cls)
##   d$yS <- Surv(floor(y), ceiling(y), type = "interval2")
##   m <- Colr(yS ~ X1 + X2 + X3,
##             data = d, bounds = c(-Inf, Inf),# add = c(-100, 100),
##             support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
## 
##   mm <- ColrME(yS ~ X1 + X2 + X3 + (1 | cls), data = d,
##                bounds = c(-Inf, Inf),# add = c(-100, 100),
##                support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
## 
##   return(list(rmt = c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                       diag(solve(mt$Hessian))[-(1:7)]),
##               rME = c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)]),
##               lmt = logLik(mt),
##               lME = logLik(mm)))
## }
## 
## for(i in 1:Nsim) {
##   res[[i]] <- try(aa(i), TRUE)
##   while(inherits(res[[i]], 'try-error')){
##     res[[i]] <- try(aa(i), TRUE)
##   }
##   print(i)
## 
##   rmt[i, ] <- res[[i]]$rmt
##   rME[i, ] <- res[[i]]$rME
##   lmt[i] <- res[[i]]$lmt
##   lME[i] <- res[[i]]$lME
## }
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"),
##                                          each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 2)))


## ----sim-tramME_IC1_plot, echo=FALSE------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(-beta, tau)

coefs <-
structure(list(coef = c(0.18604976187075381, -0.13114849660073852, 
-0.17355815592233406, 0.086367712289995194, -0.35714526098556776, 
0.62976600458176801, -0.27960199429185506, 0.24084587791662868, 
-0.1450432403490558, 0.31536875198782593, -0.25238169953639206, 
0.10992704501597397, -0.076168970197780517, -0.088902444581293316, 
-0.13619993856570606, 0.049883266633865032, 0.45705471031833383, 
0.4242359391793733, -0.190747874826078, -0.4324602293723957, 
-0.10554334206053621, -0.44145131060156423, 0.15457090453651659, 
0.27528479910242909, 0.07442871557705287, -0.32245243249897615, 
-0.1396178372780123, -0.27728823702462019, -0.43881718048374529, 
0.41879809661616935, -0.17785656694606097, 0.35384000687354189, 
-0.51390132220051588, 0.015023744018507481, -0.12357218421755874, 
-0.12497514198861824, 0.011108584309130405, 0.035874564582653666, 
-0.23290673156778777, -0.044373251360622736, -0.30501426703296008, 
-0.09187540642655169, -0.28226855687075525, 0.37728887825926077, 
0.30148450518982034, 0.10849593020326916, -0.21571915777188055, 
0.68900907377037968, 0.31078665107932202, -0.48314025085207402, 
-0.22092296971626815, -0.62887061006556211, -0.37656278610067206, 
-0.54830947304180933, -0.027615986837721609, 0.73810300526579686, 
-0.076247582164826244, -0.35728291029824938, -0.018446940781160534, 
0.34895883544548922, 0.4466758046299939, 0.098295310029393179, 
0.50361592814866252, -0.32312429915946045, 0.39142779328996741, 
0.069852712071928183, -0.38783723655094321, 0.174304694391864, 
0.098224471998874088, 0.81719915163745271, 0.39598083544630969, 
-0.089833435524212815, -0.77686505296950015, 0.2016058142088927, 
-0.067770457161935277, -0.17709038424455251, 0.22705140031351173, 
-0.29947645750142238, -0.53871313306855284, -0.3627149620229132, 
0.25263421648935147, 0.55585117460448441, -0.10037601278716077, 
-0.17263774189797443, 0.057646213703389342, 0.032831050414615301, 
0.53365777035000128, 0.3558963906447758, 0.77550421192219299, 
0.70872682381697072, -0.0086144794490925752, -0.085308014630552459, 
0.15374536609002404, -0.23788812775005908, 0.39212719691753878, 
0.359231215593111, 0.078313247010685247, -0.26300609247075263, 
0.1322232788823442, -0.29756922279276043, -0.92010857913910227, 
-0.91724271765495935, -0.38421680583580203, -0.36000511091556442, 
-0.68634566544740305, -0.90817165004420131, -0.75100557876092056, 
-1.0243517343329054, -0.80426856269030667, -0.37939954666032105, 
-1.2478707902513786, -0.51777437436841989, -0.94715413755207789, 
-0.7824661635634329, -0.67886845701247989, -0.60795712733498664, 
-1.1951807625124322, -1.3125380009540824, -0.7606339323154705, 
-0.6117131827174096, -1.4827536351283204, -1.0129107401968311, 
-0.91979007417701641, -0.98952884268589503, -0.5000705004046172, 
-0.82672686269330753, -0.99078542580459661, -0.75059598315224896, 
-0.42610038597093858, -0.99802425313008636, -1.4012881515777118, 
-0.98360693843444724, -0.90675070064928187, -0.86020717017996762, 
-0.9638883434735992, -1.0648389636701521, -1.4641480633840482, 
-0.59959812193090523, -0.87299552078717979, -0.55672173144507719, 
-1.2881752363628303, -0.72904903063061388, -0.65677604855318317, 
-0.30655670757344239, -0.49094146604202266, -1.0376935644563889, 
-1.345814452865977, -0.94649176482584862, -1.3056697529472858, 
-0.73300508019009947, -0.97899787532107863, -1.1059273862784671, 
-1.0208289060339732, -1.0727518843508692, -0.95798667294226147, 
-0.82208239300358721, -0.65128129919807209, -0.78178765905797376, 
-1.0319289781682626, -1.0773285990650761, -1.3005295465637385, 
-0.85767411593679255, -0.97333421982761992, -0.86341016422234929, 
-1.3633069264533579, -0.45672716694744109, -1.1620825015097829, 
-0.10089049312724883, -0.23965772361246934, -1.2350837238614158, 
-1.1382206711092873, -0.63525569593853226, -0.9983754493950695, 
-1.4465165854947615, -0.97887434950891272, -0.98866766551673424, 
-1.3219290637713612, -0.78531472063539609, -0.57315285005527195, 
-0.9111076891169716, -0.95197678669664365, -0.96139958155130545, 
-0.62196762209876466, -1.0813395887483965, -1.2477420830247075, 
-1.6085969682171049, -0.46331796023054023, -0.73248270859444664, 
-1.204343540914093, -0.48901353627912097, -0.93181847543269369, 
-0.87010175560428593, -0.62590970208649577, -0.66099957621778627, 
-0.74688300385710815, -0.85828897283419137, -0.87452394752939655, 
-0.89350438568331303, -0.65844227817759915, -1.0852776824134638, 
-2.2444920222364275, -1.5092011839329507, -1.7043597757853473, 
-1.9671908798949662, -1.8103896897266549, -2.2615244779699721, 
-2.0645804963210153, -2.020156374956426, -1.80097054987634, -2.1618345177857226, 
-1.9770176654974161, -2.0733593176959162, -1.9768144517431183, 
-1.8318978751597386, -1.9577268774869516, -1.6013859135514454, 
-1.9107393720219605, -2.1537939621667883, -1.9010392654448642, 
-2.3984757609010798, -2.8528284533709258, -2.0697432480733444, 
-1.928031378039051, -1.4676137542403833, -2.3213515318597739, 
-1.4837031216886192, -1.9442856835585185, -1.9037887844081827, 
-1.9364740181520292, -1.8103227483482025, -1.7611154954931767, 
-1.626252216945312, -1.3997132074656753, -2.1793420258068101, 
-2.0334464273290056, -1.4214434791865216, -1.8070983953453947, 
-1.7736873326235927, -1.978504559246745, -2.1762178972921573, 
-1.8606775742051453, -1.7694185201503685, -1.6315470743993177, 
-2.1062128348650488, -1.9871339279725115, -2.4256858119906597, 
-1.5657642796789648, -2.2875848473090521, -1.5443268890797164, 
-1.8827048078525488, -1.8902925818146066, -1.8104852079925851, 
-2.4511179638122882, -2.0615633997031773, -1.6539688223958704, 
-2.2567888817778385, -1.2508920351041029, -2.0390943241174653, 
-2.0561730044760598, -2.2997571394035718, -2.0627388351185836, 
-1.6588538329383746, -1.4005992087615191, -2.0506121307732705, 
-1.5972798564653241, -1.6513477754824484, -1.7905944471989639, 
-1.659574278998563, -1.2424537980794312, -2.8483458419799401, 
-2.2825989074258533, -2.0195068284433249, -1.4371034803016414, 
-2.1167618973375331, -2.5203474247634654, -1.9482008248979217, 
-1.4871925574994749, -1.7784840327453573, -1.5132729648214001, 
-2.1766262470997448, -2.0441604711713204, -1.7465570169914217, 
-1.3839160501277235, -2.0125937910823009, -1.3670200265375636, 
-1.9834150112940034, -2.1155093345228493, -2.7239505569982674, 
-1.6289055185121641, -1.8069842961028293, -1.8649472185916756, 
-2.077048947642588, -2.1062884314068966, -2.2304048473823301, 
-2.3448658452211859, -1.7091064828949512, -2.1047559864061816, 
-1.6831862712245405, -2.6720653918816781, -1.6384889514365237, 
0.29644428769878034, 0.45131109329203006, 0.39065642660471184, 
0.35773832065742583, 0.48052108960643369, 0.58323849198141164, 
0.49799947970015074, 0.44146644930296358, 0.46436075179777603, 
0.42022303874637318, 0.32484811885352416, 0.50946943813761703, 
0.47940688824057726, 0.24207958621162801, 0.34788826601601769, 
0.46644066607707824, 0.31598196938328721, 0.67051196442855088, 
0.3641005415072589, 0.35434329960808375, 0.49167835470780713, 
0.58248157670949663, 0.00013575268917189774, 0.48021425772470144, 
0.39768414009659864, 0.15422784118056418, 0.35884003793613217, 
0.64324652370208368, 0.29013563272023885, 0.26376885809345713, 
0.5439660781246779, 0.51679764538263828, 0.63333243394601413, 
0.49682053155748712, 0.3671184847974629, 0.46720883416242442, 
0.37557867908274584, 0.54866891381423866, 0.34186148699219149, 
0.24220839174213668, 0.0758476304246435, 0.47577572151660696, 
0.52411275465634444, 0.71267528217488907, 0.4333682144750689, 
0.60328477685214732, 0.56721496857283071, 0.57871959043704368, 
0.28157789255590648, 0.55643907199779519, 1.4351627162289424e-05, 
0.53917929383206264, 0.60759011988590839, 0.63659296936428356, 
0.596828937684592, 0.56049811542698913, 0.5643666369076854, 0.54583631018641476, 
0.24606519053719156, 0.5162201665224655, 0.44029413162062353, 
0.47233876809312852, 0.36603397220923217, 0.75480506808034598, 
0.34909861816562998, 0.36989181683855071, 0.21659909552152221, 
0.48115933111854009, 0.52983009652714019, 0.65502395981381112, 
0.45324643156463629, 0.67057264360692714, 0.42825978853029206, 
0.63153227682602686, 0.54908090017828914, 0.47228242269355802, 
0.54213776541805681, 0.68401349801635891, 0.58391002216144072, 
0.60358202787176662, 0.34261011176725675, 0.29634246526562913, 
0.48266940287620552, 0.48423794328190761, 0.15442217303193254, 
0.29145623678479882, 0.32274673369083356, 0.49784118096591473, 
0.58607371447759338, 0.65780409720433308, 0.43171886427270062, 
0.55357206546213222, 0.45842010198767685, 0.76040075245163596, 
0.6931101646919009, 0.45227006609892395, 0.25511598275023151, 
0.32389667378928982, 0.54657942992592479, 0.29814484663982832, 
0.18979159590485178, -0.13496697483026984, -0.17758641965697658, 
0.10260985790652548, -0.35219081384620265, 0.64050280422662831, 
-0.29832028622899337, 0.23722461160749045, -0.17681029524242362, 
0.32105827725413688, -0.22897332674314386, 0.11471720916945517, 
-0.037959767866787487, -0.080291320923043683, -0.11401046191917719, 
0.024388457294878436, 0.44138441309767723, 0.45226979599324996, 
-0.1805535385080797, -0.41733592903812505, -0.12211670017480387, 
-0.43911071609195346, 0.15503250690431147, 0.27238049678453879, 
0.06586965470686755, -0.3479442151938883, -0.16321527314956355, 
-0.29192819629108979, -0.43929514966143401, 0.41868851673643176, 
-0.16160607301896129, 0.34756298065466906, -0.48039029795121596, 
0.02142943707713622, -0.11725464878274493, -0.10640386258259482, 
0.012251923106582057, 0.065028244887373118, -0.23451361985663657, 
-0.049930646711523682, -0.28682531503991526, -0.072286897602840433, 
-0.28631959591032502, 0.30955176160871278, 0.25428845547387952, 
0.071250468502420464, -0.21334520686626032, 0.6949897238926922, 
0.29850250134463946, -0.48501415877249127, -0.21930691346917924, 
-0.65022343317850184, -0.36927384026132537, -0.53303906637289455, 
-0.033704799091247684, 0.74722981738628746, -0.087987114144909923, 
-0.36278623861872966, -0.0084582418730499813, 0.34618338531921633, 
0.441318871010075, 0.10413083377112832, 0.4975948547296612, -0.28838294345412857, 
0.37054256248914352, 0.073618105642605125, -0.36902989230360306, 
0.18558926986633981, 0.087473773254779119, 0.8322941715165737, 
0.39205987328287256, -0.088059754869673226, -0.78211132857099241, 
0.18543895320954507, -0.060505200595572114, -0.1610851274003729, 
0.22089425757032008, -0.34765127633403586, -0.52864590509192888, 
-0.35634790956294887, 0.22835354369437988, 0.5687062020421324, 
-0.12156475887404705, -0.15291158032717561, 0.054359184197475791, 
0.067390859148448126, 0.53074333882050417, 0.37209766906809638, 
0.76428331620114787, 0.75866216725975744, -0.0524062902602715, 
-0.11946658366648981, 0.14573425420365396, -0.24896069890927391, 
0.40538395402891986, 0.32570023817573218, 0.080942468371645163, 
-0.26203051032241476, 0.10255195625875976, -0.30735854777400917, 
-0.925743327962948, -0.9276574233432956, -0.37839246281637756, 
-0.36292804416798197, -0.69240133187894892, -0.87681034496012633, 
-0.7776850730023287, -0.96356968274256283, -0.81571926551760643, 
-0.38097553320855393, -1.2403580107930248, -0.54420761817461005, 
-0.90316449792055742, -0.79527800104282209, -0.67184215414739534, 
-0.56760988955319192, -1.1887998851701256, -1.3328872015402435, 
-0.7543497797700186, -0.57153684733240262, -1.5035486424142326, 
-1.0344893800068224, -0.94145273035876109, -0.98091655857897286, 
-0.5192077567633786, -0.81586383897185843, -0.99638788202533835, 
-0.73346252100079878, -0.41078665962566369, -0.98952661900841854, 
-1.4033192221738073, -1.0152734410606132, -0.94398503884739571, 
-0.84534201609651483, -0.96736784634107287, -1.026015770954366, 
-1.4347618884090598, -0.58896392067245384, -0.88939448079666938, 
-0.54413084505804843, -1.2235940196623092, -0.68288804289624772, 
-0.66619378423910336, -0.29239761918537882, -0.4805969589528995, 
-0.99067917442635556, -1.3486274373581331, -0.91834806673237768, 
-1.306247090400475, -0.7705322489572497, -0.98906463606491968, 
-1.104482007009203, -1.0366661582722791, -1.0240864808964338, 
-0.91617908203024656, -0.86577491728628153, -0.64404365873932778, 
-0.79737254881467245, -1.0399225820408287, -1.0867402888153845, 
-1.2776574834641776, -0.85206441342008432, -0.96729616177832067, 
-0.86133765528039208, -1.3628089915459469, -0.44420486833513051, 
-1.1658249325156833, -0.1123388200221638, -0.25114551343653679, 
-1.2656136853695372, -1.1160108181117472, -0.54782294221711936, 
-0.99921467542678033, -1.4754369827042073, -0.99925534075106748, 
-0.9865816115303917, -1.3337877538869487, -0.7702257907098653, 
-0.59606462735142207, -0.89937514723303835, -0.94843805546438054, 
-0.94877753715648849, -0.60932560914007916, -1.06545812651279, 
-1.2053739589698471, -1.5928297608470114, -0.47166574862604488, 
-0.73317218719568111, -1.1929133089622652, -0.51691199857978343, 
-0.88840281688841605, -0.87337003614030617, -0.61155359362096395, 
-0.63205882417490844, -0.74857133697714207, -0.84093527400534041, 
-0.85650754813015351, -0.87803334812066325, -0.66575218950095549, 
-1.094802810112719, -2.2683322886191264, -1.5166906753749254, 
-1.6834915580644907, -1.9843893279793032, -1.7750598726485454, 
-2.2314972458358588, -2.0987534848629426, -1.9061645231655673, 
-1.7150796089878448, -2.1522363618559206, -1.9500833374686126, 
-2.0996181588203813, -1.9238657638765844, -1.8532713519212727, 
-1.9745363464332359, -1.5528454333470616, -1.8876696427568733, 
-2.1383412316826038, -1.8685292939419804, -2.32043449507663, 
-2.9836413068603425, -2.0734485668716855, -1.9669431049935977, 
-1.4648139084316918, -2.335873901516873, -1.4923756735749178, 
-1.9522807327901504, -1.8735755182275091, -1.9225216619097274, 
-1.8016860467280951, -1.7519984239542106, -1.6356365718091697, 
-1.3935472507814042, -2.1893029738975649, -2.0420587224253834, 
-1.3802433408149024, -1.8096962475260028, -1.715957920461153, 
-1.9898496452376728, -2.1572666279799657, -1.7910425908758085, 
-1.7044505126512715, -1.6088221116322352, -1.984423990584727, 
-1.8834775003403939, -2.3520400346944754, -1.583992815507808, 
-2.2302556776854363, -1.565802362363222, -1.9525453794137717, 
-1.908607612616237, -1.827612864037, -2.4641043123684945, -2.0316525978229829, 
-1.58447772358032, -2.2952660139373404, -1.2051090710875947, 
-2.059867171745116, -2.0694208013945494, -2.3010674247404306, 
-2.0111007924512903, -1.6316226094043778, -1.3748077450454144, 
-2.0334937140552212, -1.5945794248793876, -1.6451614892285515, 
-1.8042180886262282, -1.6096855440786093, -1.2359674661591551, 
-2.8419087505293819, -2.2632753955266245, -1.8684762706334339, 
-1.4364075024024519, -2.0845881186651107, -2.5848125854833768, 
-1.9645664625467287, -1.4708167359342501, -1.7183377153499124, 
-1.505378076963144, -2.1070869899840901, -2.0503547072411572, 
-1.7501876376670242, -1.3926730890923136, -1.9423817422845382, 
-1.3181550078169557, -1.9706366387226089, -2.1080691641994398, 
-2.7180858983574967, -1.6120116376102154, -1.8061585298705574, 
-1.8988345205953518, -2.0687695014073122, -2.0942759754445524, 
-2.1099575571706843, -2.3507955556525801, -1.7213252915109973, 
-2.0814477396936284, -1.6531970086996619, -2.6487426620783188, 
-1.6549445257421218, 0.19234537976726149, 0.30578524745075741, 
0.24414932045282964, 0.26307534757651385, 0.30696854316420252, 
0.34102536296444308, 0.33021611294063657, 0.29204426816862722, 
0.2372943023494504, 0.26185538001355979, 0.16122810279731437, 
0.30163836656562171, 0.28491625218365124, 0.18090236945830807, 
0.25070354872785161, 0.24037917323929617, 0.19329500993029478, 
0.41636716514676692, 0.24026413428670182, 0.20036700207560304, 
0.34801254272302296, 0.37144619693529235, 0.062034019081924276, 
0.28204721849640235, 0.28865584259542293, 0.20899324349302539, 
0.24605809274042356, 0.35131463534573631, 0.18845088080364111, 
0.15990212671830345, 0.32631083343351364, 0.31287845718921159, 
0.37671013478061249, 0.3046511063394125, 0.22189807108070086, 
0.22971603019377129, 0.18743614715659787, 0.34748063053071104, 
0.2356746376147498, 0.17024840105978142, 0.18418542843892552, 
0.26423089120826571, 0.33105376290409289, 0.41416848143025908, 
0.26023104979957845, 0.33239757700397177, 0.35458813507618281, 
0.33309049244974326, 0.2076962956847675, 0.37672441436946053, 
0.00071730829104344233, 0.3374563193180059, 0.33350900570375203, 
0.37966945268815577, 0.34803089771208695, 0.36304802895243665, 
0.32794539946924689, 0.32800838616142336, 0.18085332028353407, 
0.30096712218825616, 0.19772151218983264, 0.27143046505324914, 
0.21248960436226713, 0.45661581880445523, 0.24882812601738438, 
0.20185838199803854, 0.17604812436311948, 0.29074303445130939, 
0.30523247318161295, 0.40625395004896192, 0.29192849546074368, 
0.40660555086803352, 0.25025497290492643, 0.38449247326997077, 
0.37116956392395306, 0.29483812710162988, 0.29841668976602204, 
0.40491411686667256, 0.36462677464921878, 0.33949493564442196, 
0.22071531187705032, 0.19550726991095693, 0.29397820125522128, 
0.2967447905357295, 0.16334738846329774, 0.22754066408287471, 
0.19810389786117308, 0.29376596658617266, 0.34575490196175812, 
0.37467864001636625, 0.28667956511031628, 0.33735244105483231, 
0.26851493566086337, 0.43031443674452557, 0.44000302749985526, 
0.30313992663790446, 0.12859922437184093, 0.15223388350677267, 
0.32873510778477077, 0.22253536080481798), algo = structure(c(2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), .Label = c("mtram", 
"tramME"), class = "factor"), which = structure(c(1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", 
"beta2", "beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-800L))

bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]),
                                              expression(beta[2]),
                                              expression(beta[3]),
                                              expression(gamma)))
)


## ----sim-mtram_IC1, echo=FALSE, eval=FALSE------------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## beta <- beta[-1]
## 
## set.seed(150594)
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## res <- vector("list", Nsim)
## 
## Ui <- matrix(1, ncol = 1, nrow = Ni)
## S <- tau^2
## Sigma <- S * tcrossprod(Ui) + diag(Ni)
## D <- diag(sqrt(diag(Sigma)))
## 
## aa <- function(i) {
##   Z <- rmvnorm(N, sigma = Sigma)
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   h1 <- function(x) qt(plogis(x), df = 3)
##   # ^^ <- h^{-1}
##   y <- h1(c(D %*% qlogis(pnorm(solve(D) %*% t(Z)))) + x %*% beta)
##   d <- data.frame(y = y, x, cls = cls)
##   d$yS <- Surv(floor(y), ceiling(y), type = "interval2")
##   m <- Colr(yS ~ X1 + X2 + X3, data = d, bounds = c(-Inf, Inf),# add = c(-100, 100),
##             support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
## 
##   mm <- ColrME(yS ~ X1 + X2 + X3 + (1 | cls), data = d,
##                bounds = c(-Inf, Inf),# add = c(-100, 100),
##                support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
## 
##   return(list(rmt = c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                       diag(solve(mt$Hessian))[-(1:7)]),
##               rME = c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)]),
##               lmt = logLik(mt),
##               lME = logLik(mm)))
## }
## 
## for(i in 1:Nsim) {
##   res[[i]] <- try(aa(i), TRUE)
##   while(inherits(res[[i]], 'try-error')){
##     res[[i]] <- try(aa(i), TRUE)
##   }
##   print(i)
## 
##   rmt[i, ] <- res[[i]]$rmt
##   rME[i, ] <- res[[i]]$rME
##   lmt[i] <- res[[i]]$lmt
##   lME[i] <- res[[i]]$lME
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"),
##                                          each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 2)))
## 


## ----sim-mtram_IC1_plot, echo=FALSE-------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(-beta, tau)

coefs <-
structure(list(coef = c(-0.085843938759249172, 0.17305563410872057, 
0.41048391925275524, 0.16758055388335749, 0.22564132740046022, 
0.53639657793926088, 0.12619478857206487, -0.71992346349442571, 
-0.090531777430410243, -0.1998303793309564, -0.069519353083890123, 
-0.37701725030884292, -0.65646928309389219, 0.15030076764678299, 
-0.27436581402706478, 0.095012515968890959, -0.64467414919777666, 
-0.02287807405332282, -0.018342541537073809, -0.29419268892321193, 
0.018328642395684162, 0.2079397708826764, -0.09466848770335648, 
-0.051768813964141416, -0.23844497787724134, -0.24757947673399835, 
0.044199710042638514, -0.0064294748789170382, 0.023749900960735006, 
0.17972650083229788, -0.56738547315405008, 0.39070628858802697, 
0.17543917878782772, -0.33007977771431785, 0.27015834873259181, 
-0.03669581422742426, -0.048419189605348363, -0.057400032216218748, 
0.036531369259613838, 0.030537966280608803, 0.25457484802033081, 
0.41632434449344857, -0.49963434459947942, -0.035192721787554726, 
0.3861418953411575, -0.13603418812856061, 0.2021612503091435, 
0.090255741630774974, 0.15959379686995345, 0.070751583415212857, 
0.016001942998862997, 0.12447319121571418, -0.030140846830098857, 
-0.02330748247512154, 0.52048784953043725, 0.25300018651918765, 
-0.26474645366235489, 0.036041006045886746, -0.039122061217422864, 
-0.058277405240153728, -0.70841913919384158, 0.19333112897395793, 
-0.069726837183200635, -0.074397875989043308, -0.31518957586413521, 
-0.037637340530540672, 0.22718562809364523, -0.42820342007405071, 
0.16866120684226285, -0.30064137522066164, 0.044954045237450023, 
0.12431355968959196, -0.48988439912555881, -0.32620300647692535, 
-0.033241989080373166, 0.18886131916870361, -0.092394980393765688, 
-0.18296183198788141, -0.0071632219576831047, 0.52696654743231475, 
0.063877914987825163, 0.095095984211184634, -0.053852404356871525, 
0.037902860510633046, 0.11982038704176506, -0.20537744787894924, 
0.029464179191288922, 0.0123253157606097, -0.2236498739587075, 
0.18293939888956604, -0.2079716315380977, -0.10157127680067275, 
-0.20241615257359916, 0.2686281967986231, -0.029540713634265709, 
-0.11483200799236665, 0.21237873651011069, 0.44925276727443492, 
0.23758189579166811, -0.37509552354879555, -1.2435616024389891, 
-0.96238425553435614, -0.95451660160699259, -1.2062295137020864, 
-1.3023986338211555, -1.2070934353328251, -0.96802052019233353, 
-1.1965546935432758, -0.83415981019286556, -1.3139154968401467, 
-1.1896781055376424, -0.89071756047304884, -1.1703673889391868, 
-0.85720533585923953, -1.2054054209903309, -1.0430683745529821, 
-1.6002686385254199, -1.2799935662110338, -1.3268974399358564, 
-0.98290195239656197, -0.89632323011070159, -0.74408090623085477, 
-0.7134453163920087, -1.1607010662228627, -0.66887472274997073, 
-1.0469797829284457, -0.51800875502375243, -1.2381286231128417, 
-0.55713249688425737, -0.57592117427895984, -0.96114824729438597, 
-0.99685475278641167, -1.069210592872504, -0.95884243109492628, 
-1.4401347908956479, -0.55918541248647513, -0.70215899606579157, 
-0.87557691197211907, -0.70412954522829785, -1.0322707067499053, 
-1.2355392653310742, -1.0199917884465663, -0.73729580879295942, 
-1.0629152713635051, -0.87877939279652839, -0.90088272569993577, 
-0.70021079543171161, -0.80436890613040801, -1.0180697193382575, 
-0.69439771687744756, -1.2304289613622099, -1.1493789666228313, 
-0.35880772724351095, -1.1719358387928358, -0.88100013653216247, 
-0.85472889875431279, -1.2515588760468326, -1.2965053442563665, 
-1.2244121043509963, -0.99309488038264859, -0.95657867423160803, 
-1.1231467414412954, -0.62924846453587324, -1.1579506889539821, 
-0.54691631478129554, -0.57164567322994042, -0.93973490148851235, 
-0.84470845976772724, -0.75044552272903153, -0.81549124960308239, 
-0.70896189703962842, -0.81804286477804478, -1.1915072896697196, 
-1.0119181616105917, -0.97952689079073474, -0.49750810880670471, 
-1.4259574586895667, -0.95779860788958338, -0.90998401384271332, 
-1.2988550614268206, -0.8369551078555505, -1.1701632918983305, 
-0.95100833526235762, -0.97881295390164436, -1.3514411331200367, 
-1.1668178399374725, -0.80881585529387223, -0.60249773323249056, 
-0.97975940603113632, -1.2770827673984069, -0.59434969933964266, 
-1.3857182213505899, -1.0670732716497302, -0.43363689904654135, 
-0.50005170892406769, -1.0945518186980245, -1.3174898345572983, 
-0.81169557859128205, -0.86585753190138948, -0.54312764161527294, 
-2.0049757579285976, -2.156549412255365, -1.9566584176416255, 
-1.83963884772034, -1.6265095201022357, -1.5877095590877632, 
-2.1031216287427763, -2.3742056155114422, -2.2219230989923089, 
-1.2692607849397686, -1.5169435137354734, -2.090183082414129, 
-1.9142858558495719, -1.7814580167008234, -2.0874949961586462, 
-1.9834665801497033, -2.3043094323520026, -1.788034980865185, 
-2.2208341658459001, -2.5097536396335527, -1.7058597170983556, 
-1.768170367504085, -2.2862634245690501, -1.7677798237951563, 
-2.1342571097461036, -1.9876857829933716, -1.0895236844832747, 
-2.0703272217207553, -1.8262092976390416, -2.1110988139752243, 
-1.9563577064965669, -1.7048446821026055, -2.2570704896338807, 
-1.8803237296228807, -2.457265785616384, -1.3504555521619395, 
-2.2287992861137496, -1.606928304500469, -1.4312525247944159, 
-1.4698364391758882, -2.2276515895570363, -1.8283246718938462, 
-1.4387321384201621, -2.361533380806986, -2.0755644213290747, 
-1.6037740648361931, -1.9950292658134849, -1.938179039400665, 
-2.1570484634278624, -1.2851037520611801, -1.2780225426806551, 
-2.3751436126018421, -2.2528542355583121, -1.9599224848979289, 
-1.9350350959745581, -1.9775585018219717, -2.0478775463131611, 
-2.0555937401558357, -1.4351115082509416, -1.5890636504646964, 
-2.3315602502347055, -2.0050510341948473, -1.795912359571042, 
-1.8411527394850844, -1.7875113094190267, -2.0123424684928422, 
-2.2546199303680043, -2.1019868013377523, -1.949807925794546, 
-2.6085894459759049, -2.0828505911995658, -2.1224001311424097, 
-2.0373962896958298, -2.0827050839589547, -2.0217041481260583, 
-1.9076553852431164, -2.0695462976950196, -2.1678469083756391, 
-2.0548764630033758, -1.935606437799295, -2.1215157468092807, 
-2.1489080347136262, -1.8029652027926917, -2.5624789046656389, 
-1.7496465455648671, -1.8468484849549767, -2.0495670675198641, 
-1.8129030046685979, -2.123197140406075, -1.4009819348817409, 
-1.9833924024224359, -1.5422470032166471, -1.3390684428318966, 
-1.8310023907042647, -1.9529305780651769, -2.2788691643478409, 
-1.8336088907040566, -2.1393186111898603, -1.8264002261503465, 
-1.8594880872253161, 0.801136847518231, 0.72521944221492862, 
0.72782716628790645, 1.0160940268303689, 0.90614960841620806, 
0.86317759636973046, 0.88983538157482778, 1.313410728570122, 
0.91624597967042243, 0.88352477897918613, 1.1894717802955652, 
0.95252685928085323, 0.89232059326616076, 0.74288220381680692, 
0.88269601923421126, 0.71766058657670451, 0.78668869198232216, 
0.88861165672495412, 0.96466070752464339, 0.98703070425119888, 
0.78991644359985647, 0.91114463935660872, 1.0172573979281196, 
0.97043099064231575, 0.74756378211066166, 0.94556485886145769, 
1.1168191478658556, 1.0053134306560954, 0.78728983912957984, 
0.86757377898022414, 0.850798777612362, 0.78362315407906835, 
0.79095013150911542, 0.87897198569663915, 0.87260528468991394, 
0.74467490290261951, 0.84899353599897176, 0.73561394905970023, 
0.89695078282835805, 0.92592373045596665, 1.0762232487670824, 
0.89398094055994792, 0.77930642947334561, 0.83671730231030694, 
0.78437748221235881, 0.75924713874052718, 0.90363464265696591, 
0.97041471733345297, 0.79245756262573464, 0.93553319149185166, 
0.81112879295426343, 0.86735047092332596, 0.69441971375189659, 
0.99963009249476631, 0.94171629481135721, 0.79542780527366919, 
1.0017467456125568, 0.94024964797569877, 0.78061171241098637, 
0.69497875895336447, 1.0789224130975146, 0.90804425046222104, 
0.94725154788072985, 1.0107228331641569, 1.0220325286250436, 
0.75986185412795904, 0.89387337228596808, 1.1786928752633217, 
0.70472696897110876, 0.95644616945817418, 0.96012701248640631, 
0.9281386417344557, 0.95716933034440155, 0.7117538843475929, 
0.95024458693287084, 0.97154534761477296, 1.067591238693496, 
1.0525484616535095, 0.86569672278381193, 0.81750895122672018, 
0.77455096867610485, 0.70253951211957466, 0.84908078571471601, 
0.80100077015651694, 0.93500288872752091, 0.90500619110608604, 
0.94034993801001798, 0.86428016705941946, 0.95016235251929992, 
0.73478142881159891, 0.91023882934260136, 0.72911849306221488, 
0.88006042150638553, 0.79150837330194812, 0.82597372707334837, 
0.73609982685024011, 0.86917062902352293, 0.97166319039975213, 
0.77604657696119683, 0.75822177406893965, -0.042979507739655705, 
0.20341074440619158, 0.39554143264049718, 0.171028301651414, 
0.17507226349521601, 0.48997387544303317, 0.17286703571613279, 
-0.66112609984724302, -0.14736075698320866, -0.23166994386789283, 
-0.0069543106618278833, -0.36502233936633288, -0.62532086476227189, 
0.14123438746784728, -0.26423587481723343, 0.091507822441877815, 
-0.6608875473684459, -0.063078003583219869, -0.029722494402014764, 
-0.28004550589374033, 0.043410056683452457, 0.22365640284642249, 
-0.070471496236418574, -0.033737668731313314, -0.24421456394067129, 
-0.2068985359335096, 0.0008139907229977449, -0.022262574540388234, 
-0.0096425404753765539, 0.19361754785054353, -0.53198172822161371, 
0.3485783200035753, 0.10480433914411567, -0.35118828242993605, 
0.28446490287554449, -0.040954386353657612, -0.09199703138425383, 
-0.061935506870626204, 0.054260710321386157, 0.099245488305486837, 
0.21961358899421438, 0.41794255359386501, -0.5250286386320514, 
-0.08080897312613787, 0.38157609949398541, -0.15697823633894667, 
0.17246022081251902, 0.11205040778034209, 0.19129662002763731, 
0.064390602752300991, 0.021681377113242906, 0.11596376334160118, 
0.007857522543075441, -0.00016395504238051292, 0.50683017748956116, 
0.19813798316585204, -0.26322553592820752, 0.070032519335946217, 
-0.087809895575121061, -0.051287204089817781, -0.65938066919178862, 
0.17911891003258346, -0.052828309647067598, -0.086824559024229572, 
-0.24499175654631805, -0.02250908683726657, 0.24092355965794451, 
-0.39265349287980789, 0.18056136174790596, -0.28573670273070995, 
0.015604391523762489, 0.17048104694335273, -0.50221917167244923, 
-0.29913364348052562, -0.047669084895464307, 0.18980748897904648, 
-0.024456161995052916, -0.1780123697528283, -0.02282082992940036, 
0.49108195142004291, 0.026315786870436435, 0.11962569928367121, 
-0.0076745207081425027, 0.063425111622808131, 0.11578680891369073, 
-0.16513658207920504, 0.022656383995038543, -0.027888945832240497, 
-0.21544425870182793, 0.25151544575909957, -0.19301728264120402, 
-0.094167965360875891, -0.16758049656061741, 0.27140128948316827, 
-0.070431955430559232, -0.093496235969760283, 0.21426643118192143, 
0.41799546185341208, 0.2200658435640242, -0.38091949114004525, 
-1.2507890707639189, -0.97889271084903817, -0.90835838288635629, 
-1.1487787790195516, -1.2361306260247198, -1.1394139329325723, 
-0.98982677358936655, -1.1017070611637916, -0.90997023831690582, 
-1.2183896143358719, -1.2071262528054583, -0.80365805879101604, 
-1.2694626435075174, -0.81726129398132119, -1.1544679504465072, 
-1.08359643871651, -1.6369857050842345, -1.1604002371928506, 
-1.3007955646765781, -0.96394640425717415, -0.86572724041990357, 
-0.74209460050887888, -0.58899624899422875, -1.1039982043406231, 
-0.7229186306077523, -0.9869020280253219, -0.51377163475876786, 
-1.2086525670616026, -0.56714941996418267, -0.61302304426435417, 
-0.97793582902778342, -0.91423618121251082, -1.049550710993898, 
-0.93931315540097238, -1.4285592340252737, -0.52440418463983518, 
-0.72633562523207273, -0.85647647605599997, -0.67365998013934525, 
-0.95011092248043061, -1.1943512449869891, -1.008872188397802, 
-0.70956349492206927, -1.020490813706604, -0.90720396864281527, 
-0.92211596031097076, -0.72720195869074189, -0.82193059536877378, 
-1.0434517985156304, -0.72004669574947122, -1.2891092949637708, 
-1.1873364488590041, -0.39082312718853751, -1.1885515655743757, 
-0.88415903636570858, -0.85045563658640133, -1.202035136885343, 
-1.2215878308208248, -1.24388677679487, -0.98502409543789116, 
-0.84123596231740494, -1.0397580403282847, -0.64166327452024829, 
-1.1391803097032045, -0.54256521997442908, -0.52857767872127648, 
-0.81244638739454711, -0.58424969217313072, -0.73388422547174081, 
-0.84982370804624763, -0.68498640410837208, -0.75253481389222621, 
-1.1486512396297743, -0.96255155781258839, -0.92989541076760551, 
-0.58197541560163513, -1.2849653231358753, -0.9632783570922071, 
-0.8682703622858664, -1.254980374929237, -0.82907490490982527, 
-1.1626117125813591, -0.88008961800879437, -0.97324433720704573, 
-1.3260499244362023, -1.0691448601489679, -0.89448135839918741, 
-0.5889201405573663, -0.94807166878427607, -1.2616208769283115, 
-0.60085465210115785, -1.2313391035659085, -0.98130968873115876, 
-0.41048161979884012, -0.53977959054835223, -1.09286251631955, 
-1.2850982268332161, -0.68148705412014932, -0.89464046220196325, 
-0.45823375988347148, -1.9169975178785428, -2.1539947412710823, 
-1.8202351949573445, -1.7472348593256812, -1.6366005969202959, 
-1.53705035424305, -2.1631834495456328, -2.2026149396382748, 
-2.231284576001741, -1.1965593258504201, -1.4848310253691805, 
-1.9975793656184266, -1.9127009829983823, -1.8563306115834592, 
-2.0643956520608904, -1.9901717297202526, -2.3777233111422458, 
-1.5881435726131907, -2.198923060129566, -2.4830191431720743, 
-1.6884506754291491, -1.6981748815180215, -2.0153357367569362, 
-1.797143497442069, -2.1528079920064638, -1.8857149962049109, 
-1.0606838176536768, -2.0654189008383068, -1.8428990290450777, 
-2.0963757459698567, -1.9510901137930687, -1.6290563864877758, 
-2.2289714531104745, -1.8209557972879939, -2.4299483202228838, 
-1.3576436108815171, -2.2729714610630203, -1.6035632420681871, 
-1.4463990427154045, -1.3150143400624406, -2.2918802789186055, 
-1.7713992936895155, -1.4061882129807708, -2.3968502714372137, 
-2.0498250225472128, -1.6045023940193019, -1.8944546012250258, 
-1.9312728224854079, -2.1073708577985397, -1.235799455663962, 
-1.3397036999993972, -2.3348142629284685, -2.2583303631354164, 
-1.8349044291359649, -1.8947157683544056, -1.9670590165516579, 
-2.0214372482839233, -1.9659940657868737, -1.4243570961038705, 
-1.5450359837707432, -2.1097127766002046, -1.8558710065276558, 
-1.8269518713583148, -1.7881855911159283, -1.8455125500959513, 
-1.9118683330954158, -2.0604266632125956, -1.7707961681669915, 
-1.9673864883736891, -2.5371356017265705, -2.0817754611435038, 
-2.1033239086779685, -1.8946208999399541, -2.0568937428228278, 
-1.8210398568281962, -1.9944768899820478, -1.9303442635835992, 
-2.1433842666021019, -1.9780411719548503, -1.8841855808002179, 
-2.0496094726026257, -2.1225256340296781, -1.5110328510177529, 
-2.6485812466549836, -1.7255264473118319, -1.7032282300332253, 
-2.0662670375324863, -1.7762159927746888, -2.1078002151472193, 
-1.3403783106563196, -1.8602802123686446, -1.375949864851207, 
-1.3031239987466856, -1.7806101422505047, -2.0346095754403528, 
-2.1531637955210989, -1.7922063267234734, -1.9215227476690717, 
-1.8024254173036824, -1.7161604185481778, 0.48388306034055856, 
0.44956057013861322, 0.46130477725449398, 0.58205401080571406, 
0.53610849026358254, 0.47681139030784547, 0.55902761091055997, 
0.70444903471863141, 0.57247069898636049, 0.50162962751647333, 
0.70324205398382444, 0.54170195647380215, 0.52661247583203641, 
0.45815461880429126, 0.53445030446287078, 0.44927268302165257, 
0.45809067773408529, 0.48538398266775634, 0.56149456210553539, 
0.58566277894768404, 0.50259317679114457, 0.54830442474250884, 
0.56825329371459288, 0.57191943884902663, 0.47539649580514937, 
0.56275048334948274, 0.65925472632192483, 0.60116220233120221, 
0.48744165583064497, 0.5226370659469084, 0.52472430468951414, 
0.47480061140353236, 0.49427960652817632, 0.50731607870969408, 
0.54200692829689756, 0.46242881368759886, 0.52594873281857313, 
0.43335932652761489, 0.53741061547842861, 0.51165687223881429, 
0.63044330460173481, 0.54268559502073987, 0.46848993426583596, 
0.50504375333170137, 0.48478842653017523, 0.46571844459429634, 
0.54508937661078483, 0.57370956571842546, 0.45634559020744903, 
0.5356806735055949, 0.51969746204452993, 0.52905820694532557, 
0.45583258130341864, 0.58115631343512941, 0.56868442925053431, 
0.4979071615484259, 0.58498305069901446, 0.56731504144782186, 
0.48113932530000569, 0.38482815595473258, 0.59317334053160342, 
0.49377477965722222, 0.56849067888449001, 0.58567980819906917, 
0.62452948430569899, 0.43713580604809971, 0.51316544179793588, 
0.64315596793600061, 0.44208559542727627, 0.58985668146785353, 
0.56813731762796027, 0.55573651397679702, 0.5394101427241097, 
0.44460488488077898, 0.52737095613051566, 0.60553613930797956, 
0.60721232862273877, 0.63228995034341218, 0.49846315155358384, 
0.48750646853067409, 0.45702674809248744, 0.41303013463954164, 
0.46255250122579894, 0.50385505030496436, 0.52457234645629314, 
0.54996902931853042, 0.53526444895635217, 0.50120434137303616, 
0.54380826374641222, 0.39982732848996561, 0.52369371917524876, 
0.46438899484429585, 0.50740625756867164, 0.47058611495559927, 
0.51791390674195392, 0.45348945884353775, 0.51381789774503472, 
0.55195141068300502, 0.48872706309883479, 0.47389452569366713
), algo = structure(c(2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L), .Label = c("mtram", "tramME"), class = "factor"), 
    which = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", 
    "beta2", "beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-800L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]),
                                              expression(beta[2]),
                                              expression(beta[3]),
                                              expression(gamma)))
)


## ----sim-probit_IC10, echo=FALSE, eval=FALSE----------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## beta <- beta[-1]
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## 
## for (i in 1:nrow(rmt)) {
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   y <- qlogis(pnorm(x %*% beta + rnorm(N, sd = tau)[cls] + rnorm(nrow(x))))
##   ##   ^^^^^^^^^^^^^ <- h^{-1}
##   d <- data.frame(y = y, x, cls = cls)
##   d$yS <- Surv(floor(10*y)/10, ceiling(10*y)/10, type = "interval2")
##   m <- BoxCox(yS ~ X1 + X2 + X3,
##               data = d, bounds = c(-Inf, Inf),# add = c(-100, 100),
##               support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
## 
##   ## mtram model
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
##   rmt[i, ] <- c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                 diag(solve(mt$Hessian))[-(1:7)])
## 
##   mm <- BoxCoxME(yS ~ X1 + X2 + X3 + (1 | cls), data = d,
##                  bounds = c(-Inf, Inf),# add = c(-100, 100),
##                  support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   rME[i, ] <- c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)])
## 
##   lmt[i] <- logLik(mt)
##   lME[i] <- logLik(mm)
## 
##   print(i)
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"), each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 4)))
## 


## ----sim-probit_IC10_plot, echo=FALSE-----------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(beta, tau)

coefs <-
structure(list(coef = c(-0.033380710930649697, -0.067255837238215696, 
0.10630012700232817, -0.2750251234377229, -0.026446589989634728, 
-0.17988514623570545, -0.29014589259595469, 0.22815332868766991, 
-0.26064168799476545, 0.078919775445920784, -0.034682191290044108, 
-0.029098636907761058, 0.2058594791122893, 0.1233179248499766, 
0.11267082755809239, -0.33378146585537632, -0.025070143118020589, 
0.075387202107385892, -0.34707904138934292, 0.054572225487373775, 
0.09993174615465944, 0.020325675137982453, -0.012298337479318628, 
0.18832867001436854, -0.040053865328434787, 0.11105902761482347, 
-0.043946946114069019, 0.067726601928213198, 0.11589240275773957, 
-0.033883154976237582, 0.072051396516612562, 0.17773105918110174, 
0.075529094363382621, 0.3696455906137272, -0.044212857550361577, 
-0.12534323618665791, -0.041462766128459928, 0.067960130740957581, 
-0.19009923819511251, -0.039724395132476741, 0.059496669966122899, 
-0.19340845358685302, -0.036020070594167924, -0.12682435432401856, 
-0.20814737571948683, 0.33679450539952932, 0.24997006781200615, 
0.028462952806768638, 0.0060757662671736233, 0.034615756330278281, 
-0.11018711981238337, -0.033029918756960867, -0.070853316944205089, 
0.027226164263615071, 0.14516557444405762, -0.19758976899079769, 
0.10907617248582145, 0.023618132222410975, -0.18030474192098231, 
-0.1818727230293995, -0.14456278557347621, -0.23686911288050255, 
0.31453973197290791, -0.11660612521459666, 0.12017911496680322, 
-0.16081764617689145, 0.075520790874858523, -0.13482233035285482, 
0.28077385567087143, -0.10689460421333083, 0.18302402465325671, 
0.1270314142765063, -0.27077617478834304, 0.17921215484864178, 
-0.2014424249073799, 0.19405568576971649, -0.10334678652224635, 
0.1110077309204579, 0.048522961600809239, 0.20343573732287323, 
0.021875068823156193, 0.13964825505570944, -0.023111343372104159, 
0.018243501750275965, 0.13778188930434526, -0.089391636311774769, 
0.014223195179074192, -0.162464853805208, -0.26739528867936685, 
0.063183357469879048, 0.21158285152160122, 0.21471966060283743, 
-0.21234438644480758, 0.32494496045336974, -0.23893976712633735, 
0.26396099493496117, -0.11149227855152959, 0.15191428322165718, 
0.045985809974443417, 0.032408416119157721, 0.84567418903541958, 
0.98888834509879009, 1.1063150847794576, 1.1511209380699354, 
1.0333026679075559, 1.1706899901049057, 1.0326262647223332, 0.89574846114422746, 
1.1307597556514908, 1.3339892543992056, 0.89334661909686996, 
1.0344156885754918, 0.94486495582658991, 1.0085372038880176, 
0.90668951186197255, 1.1514429895281277, 0.86400103424538499, 
0.62684350075695672, 0.9098972736992792, 1.2738531264730457, 
1.0617474407170433, 1.2299548162396312, 1.0489693570166876, 1.1151400242197704, 
1.2526712988951698, 1.3800282273005919, 0.83685855059326753, 
0.91614677444698012, 1.1342864344019261, 0.91346800961669239, 
1.1572072499600812, 0.79823474165403985, 1.0962042888632246, 
1.386269583538724, 1.272099865245393, 0.88474771599323943, 1.2367258257979141, 
1.106956602058961, 1.2781492194201856, 1.0767707919385587, 0.98484870016134607, 
0.93807731588275478, 0.97375243030647995, 1.0271594045285068, 
1.0667576699928294, 1.203537968873865, 0.98425028839584583, 0.94875424578768297, 
0.89970149071724448, 0.78504435266873285, 0.96707020806657684, 
0.80499957686940238, 0.97297906996307848, 0.92789942161045724, 
0.93701627913593855, 1.0675986336487939, 0.92209703997308068, 
0.92954409808055449, 1.1630846532205539, 0.84185330882049991, 
1.1789135227964018, 0.95955286539388274, 0.83645370050399259, 
1.0517990314913941, 1.0360816221488465, 0.80150847633502509, 
0.85991157654296491, 0.77383328531621087, 0.77170323178088585, 
1.0565633537644163, 1.0757161608040156, 1.1173869054534111, 0.90925635736997268, 
0.88067525535378122, 1.1301890737619986, 0.93073383291461043, 
1.1018071704883878, 1.2547320265562798, 0.77691186205640272, 
0.97336768316475608, 0.80486481045837588, 0.92192815838362674, 
0.80742426152348856, 0.86196148668731665, 0.83226439935960217, 
1.2164637575958159, 0.87321620863602334, 1.4128433657923551, 
0.95707725498769292, 0.60360620785553853, 1.0219569601877667, 
0.91740100354788512, 1.157856993051801, 0.65618042299410495, 
0.86509756309010499, 1.0540625639603771, 0.99601138216486373, 
0.8298911629007405, 1.1758722519250431, 0.907735465003222, 1.8406829312835107, 
2.2117326680455891, 2.0380672816028533, 2.1184398408583496, 2.0957116343211939, 
1.8715760117933034, 2.2265232010654725, 1.9076937057644738, 2.129328785803009, 
2.2212277857978742, 2.1787308611973319, 1.9494420818327685, 1.8613754990347444, 
2.0748926551792524, 1.8240212144167349, 1.8863927894882679, 2.0843739491129765, 
2.2133581095153319, 1.841945712682695, 1.9999751768472487, 1.7445331457050566, 
1.8078084446516849, 2.1354445836162141, 2.4031285570436127, 2.1073065995030871, 
2.1710864390970697, 1.6662546882933449, 1.7543696106999553, 2.0553694799456146, 
1.9636826029125198, 2.1147422312637634, 2.0387422762952734, 2.0229833725846849, 
1.7835413941412452, 1.9283343454920532, 2.0303607395184504, 2.2532121570857777, 
2.1240472666650394, 1.9390721213481326, 1.9268863425970333, 2.1108537726507275, 
1.7743726778195641, 2.0944283805190622, 1.8633030039033907, 1.8705955660553766, 
1.9714096501610738, 2.116751943142571, 2.0628384578664534, 1.9033751689572205, 
2.1871149090785065, 1.8521009467177831, 1.942518368108838, 2.1022518548780282, 
2.1181317470298477, 2.2200981898127194, 1.8917096011191099, 2.0247798035840638, 
1.853792351473682, 2.0932496615473966, 2.1008200509312691, 1.8993672085477176, 
2.0845930504843575, 2.3803665849607052, 2.1179076561287551, 1.9478777813864085, 
1.8411436690801881, 2.0469967938526326, 2.0838707618280856, 1.5554474683494859, 
2.1735629906723744, 2.088051568870966, 2.0353605751704085, 2.1623537578371996, 
1.8002397773517684, 2.0578529944343344, 2.1181630844988759, 1.9796618573260949, 
1.8221236537905112, 2.1668278221299913, 1.8593265300046122, 1.7683692264967212, 
1.7547274574627845, 1.9604150274066576, 1.9578764211816782, 1.6095951771166941, 
1.9712900762558168, 2.3098823117248535, 1.5377267204840828, 2.3119688683033468, 
1.6350363770669616, 2.1727261079687819, 1.7756682982105729, 2.117947162387865, 
2.0872502653437754, 2.049060446807919, 1.9620120052633814, 1.8114566327196346, 
2.00980531818993, 2.2135097487643201, 1.8043106999313738, 0.47119460994844226, 
0.45553618700839765, 0.66163076323343639, 0.58017755178969144, 
0.51153298942891046, 0.48475843226365817, 0.53579343246148359, 
0.49630718725140582, 0.53850849133155165, 0.57092524725068039, 
0.41331198322336549, 0.51752118783568257, 0.41714352703835378, 
0.61961490818596177, 0.32769640766523661, 0.54016709360804471, 
0.46839560045812362, 0.56421249800372231, 0.55833101072695324, 
0.59503044857144494, 0.40986750567181846, 0.43103414456259265, 
0.54493919694375581, 0.45553125094673597, 0.49025628674441002, 
0.58799711493524764, 0.56831867237026434, 0.68018806334488668, 
0.53863383225686157, 0.56675462383086894, 0.57986682047935889, 
0.61471860128261735, 0.48878769128654531, 0.53854019580996693, 
0.55101443451621424, 0.45056638791741294, 0.54461798051728494, 
0.54224415660147041, 0.53032610464261942, 0.50305094005357631, 
0.52376409954678549, 0.51773259240833336, 0.50242952177451938, 
0.55387263413430288, 0.52829784466172891, 0.50324960431444332, 
0.53126929823831281, 0.42576491332191607, 0.52465667398014915, 
0.46377009686577864, 0.54261210814422001, 0.54682346128092774, 
0.51539769617230236, 0.52081082376070675, 0.52689948007072041, 
0.43653032956808274, 0.61077402141941639, 0.47564312949756987, 
0.57379130000735634, 0.57778938115408429, 0.592698420517205, 
0.59104979099022925, 0.63883677568734643, 0.73097095409765667, 
0.50758989086331807, 0.28955466077490638, 0.47640100171787347, 
0.566601914908055, 0.60803489086389684, 0.43155551931562691, 
0.52318998391684157, 0.57371707543543904, 0.71827492534731718, 
0.51006073381030992, 0.53377124252154418, 0.6235339581640077, 
0.41743681293901358, 0.3970090911550721, 0.71879868442228079, 
0.61570955970158281, 0.42348349977127359, 0.4884762847477308, 
0.50121012053167768, 0.54118230176613735, 0.59068593327362606, 
0.59127403401768808, 0.65097515117726457, 0.39017521153644297, 
0.70794920369926462, 0.49400797527855311, 0.52343405679228905, 
0.54089232511670926, 0.60581607776848545, 0.69583982371887798, 
0.52588009839678063, 0.56058910318587563, 0.53974103149243902, 
0.47448676233194986, 0.5373913532257526, 0.52365314773396976, 
-0.033271496505233207, -0.067477854348646463, 0.10504964918815479, 
-0.27509466346060513, -0.026988731247942422, -0.1797271741635823, 
-0.28989986971253651, 0.22398503682632576, -0.26112528019014336, 
0.0782544842245992, -0.035103346327272293, -0.029007516981352729, 
0.20586049060821768, 0.12346377644991076, 0.11235455733372861, 
-0.33377095415543345, -0.024590482261964527, 0.075164986581518295, 
-0.34756372567283267, 0.055201565924759277, 0.099911888054486322, 
0.019067052904525844, -0.012491448060975577, 0.18792063540027318, 
-0.040537144500268234, 0.10940680860951836, -0.044804892716866773, 
0.066472482471242184, 0.11572877837020165, -0.033860731203268879, 
0.072446112182070199, 0.17809638556887031, 0.074540963870603397, 
0.36931828485767065, -0.044567603768886202, -0.12576966166372083, 
-0.044785984521696957, 0.068005407399029022, -0.19037069485153493, 
-0.040122790104415988, 0.059410079489920463, -0.19241973417677788, 
-0.036298111819748909, -0.12662322026504433, -0.20981037654161028, 
0.33645321145615448, 0.24984777871081618, 0.028238063110300464, 
0.0052799562607206557, 0.034725993163730123, -0.11031387383257883, 
-0.036283263076392493, -0.069351721103629388, 0.027102137909400949, 
0.14403137677926289, -0.19644287920381487, 0.11184697352964423, 
0.02229745807498059, -0.18084402716264167, -0.18203990024053809, 
-0.14417191625962533, -0.23686907494854462, 0.31506359342001339, 
-0.11879902654826131, 0.12006656080279053, -0.16122248874979039, 
0.075615159549587321, -0.13989883627902613, 0.28024663669057415, 
-0.10755363222663432, 0.18395742804669893, 0.12728549892299368, 
-0.27125302844026372, 0.1778021081294123, -0.2013836705500523, 
0.19113228001143096, -0.1033495585121222, 0.11062763964104864, 
0.04840618097430268, 0.20344812366282813, 0.021444957812680059, 
0.13901684440202969, -0.022616668205564006, 0.018109711937035852, 
0.13775875583614397, -0.090205452888811516, 0.013932028633045735, 
-0.16353063734815806, -0.26736604268345598, 0.062599854784935752, 
0.21119405231079674, 0.21364991956397422, -0.21264926577432308, 
0.32476097448729119, -0.23951201977381228, 0.26016711927721886, 
-0.10949964529446975, 0.15182484758414813, 0.045077917890471705, 
0.031607203879364928, 0.84590589378767223, 0.98869132430887441, 
1.0988398085496605, 1.1489288265803992, 1.0333627896081432, 1.1705412143826475, 
1.032811911930384, 0.8943168519709549, 1.1288165684367004, 1.3340640940474704, 
0.89335231422914696, 1.0338985842810267, 0.9446978771625314, 
1.0087273104359464, 0.90654609336819159, 1.1535855781291695, 
0.8639167847626732, 0.62663452599222291, 0.91017778866954557, 
1.2734718385596548, 1.0616266433670185, 1.229554941402595, 1.0486744435141024, 
1.115064541045286, 1.2520540082038063, 1.3797714503954976, 0.83695002288567755, 
0.91566410979147528, 1.1338367229534589, 0.91347095448319593, 
1.1560488697251401, 0.79977427675762924, 1.0957384260400147, 
1.3859837292500661, 1.2722057951254568, 0.88430119854513767, 
1.2259242552825185, 1.1069952683775868, 1.2778929261415191, 1.0763505529784752, 
0.98482627932645372, 0.93732410540281907, 0.97357728651294129, 
1.0279238693351256, 1.0624217532060798, 1.2036189030401709, 0.98415219144528432, 
0.94875612872777682, 0.89745242610945963, 0.78495086447392104, 
0.96722470373293323, 0.81067939165410119, 0.97220122737128079, 
0.9283434913936589, 0.93713017270807464, 1.0669333903218403, 
0.92930242152593356, 0.93006968833593018, 1.1641003684714082, 
0.84167624185229084, 1.1781948763336936, 0.95925678645299006, 
0.83575912195656021, 1.0517600356861296, 1.0364759598556625, 
0.80114911060191596, 0.8591168354920472, 0.77867048893127966, 
0.77170500603391823, 1.056485269916279, 1.0757255094509668, 1.1166677181478997, 
0.90948764600927534, 0.87909623960955474, 1.1301806344663474, 
0.9300251922785574, 1.1017340317730873, 1.2547507203674493, 0.77719448906329058, 
0.97339873194960624, 0.80483490955535153, 0.92149213029106902, 
0.80805353434462956, 0.86228810530605238, 0.83237470744930053, 
1.2167973204316271, 0.87322376899325005, 1.4120922648626906, 
0.95726841942465313, 0.60262392944915855, 1.0218935219276144, 
0.91947545194757618, 1.1575835503693284, 0.65627696598775964, 
0.86437916733108666, 1.050957339235105, 0.99474753150753381, 
0.83057993490347393, 1.1752988055283142, 0.9074360382951312, 
1.841070279248785, 2.2122431072569082, 2.0313286634830758, 2.1163219908747157, 
2.0952317053346063, 1.8719680888856738, 2.2269700798381034, 1.90408767531171, 
2.1274715998920302, 2.2196870989573227, 2.1790571197613131, 1.9491048944049958, 
1.8612733278174507, 2.0750322161624966, 1.8240884134379824, 1.8883950458891106, 
2.0844198997128967, 2.2133612561786817, 1.8405273339307355, 2.0007635560689496, 
1.7441536205876786, 1.8069317528665991, 2.1357781774717024, 2.4036200715146898, 
2.1077658153444667, 2.166702348427306, 1.6665270206564222, 1.7543367337556739, 
2.0551092503242905, 1.9637825720015971, 2.1158559567258446, 2.0394819305097562, 
2.023118223441795, 1.7831527449210376, 1.9285198787848279, 2.0293980711611095, 
2.2478949087582873, 2.1242593406155308, 1.939619834135146, 1.9265975149600227, 
2.111150659476043, 1.7752058186136546, 2.0945593761042174, 1.8630521130363822, 
1.8675681927838732, 1.9716774048959267, 2.116828190430557, 2.0637846185309003, 
1.9030340228400036, 2.1860936747585096, 1.852171128965399, 1.9420162387948507, 
2.1023597230479298, 2.1192497331005695, 2.2200840558151853, 1.8897313769945203, 
2.0230787477792402, 1.853945436981959, 2.094168667857466, 2.10062062165396, 
1.9007614349185153, 2.085277143873344, 2.3796069738819043, 2.1113462125522826, 
1.9478547666959021, 1.8410262058587419, 2.0470321669401255, 2.0814687174179549, 
1.5552658950037459, 2.173661595717602, 2.0892563663756816, 2.0356267658321845, 
2.162562605378898, 1.8001188985111676, 2.058014563720568, 2.1169212530369577, 
1.9793313781551503, 1.8219265940230456, 2.1629363629750102, 1.8592511424254234, 
1.7683376840991105, 1.7547248716401158, 1.9579688555637758, 1.958368383530795, 
1.6095411651044278, 1.9707752347614496, 2.3104614476240548, 1.5363912356709757, 
2.3122432594619209, 1.6345040411160707, 2.1729478261491213, 1.7754954978592794, 
2.1177311673651067, 2.0871764844341412, 2.0488131157491849, 1.9594826645920711, 
1.8094356638126083, 2.0101703701991744, 2.2136432089714297, 1.8048730732485143, 
0.47276303022334454, 0.45603530268960452, 0.66948628993211323, 
0.58021970938848899, 0.51361895378745304, 0.48598840656818798, 
0.53630397546455999, 0.49839457247803198, 0.54344641375827163, 
0.57029692845900137, 0.41428112076654566, 0.5178583062351898, 
0.41790470396010632, 0.61995483915610505, 0.32814376524644084, 
0.54226035528353922, 0.46919600541804901, 0.56372695897757652, 
0.55976374233443305, 0.59789547766297391, 0.41021625565393049, 
0.43219500293261515, 0.54592834683406477, 0.4565848454496409, 
0.49150493390440264, 0.58755173348695411, 0.57005969404369961, 
0.68194507459507969, 0.53746277423196853, 0.56706644244396698, 
0.5825508185687851, 0.61375604279118035, 0.49067441273618884, 
0.53903020932287393, 0.55370253298301764, 0.45192531204829312, 
0.5467281037724665, 0.54188050168470669, 0.53178305305034279, 
0.50266451984567351, 0.52408873387622334, 0.52090376944843308, 
0.50194939639204372, 0.55608214560989833, 0.5315906570474509, 
0.50434970089600828, 0.53135675753169687, 0.42630961894238362, 
0.52629479169528248, 0.46408505735863231, 0.54307660826345638, 
0.54819260395827818, 0.51800967780460982, 0.52224476241037299, 
0.52931490964865646, 0.43875945936107635, 0.61660712731030998, 
0.47609155068274267, 0.57481672165028186, 0.57702664543856319, 
0.59484006095425368, 0.59099990491380061, 0.64225368042807141, 
0.73530338348369118, 0.50814718938825509, 0.29005620304931251, 
0.47661475623878885, 0.57009790835723284, 0.60773021172971187, 
0.43209444637193306, 0.52615354135459047, 0.5764162816934103, 
0.71742005276173471, 0.51043150569352136, 0.53451092434857994, 
0.62803537568279288, 0.41795616550880987, 0.39789902871931149, 
0.72715818795730813, 0.61528127063315607, 0.42401962527822851, 
0.48932161206675556, 0.50298968762629859, 0.54205789268939319, 
0.58976939238882764, 0.59125911017247257, 0.65109484591703182, 
0.39096078901826747, 0.70722004883358125, 0.4957897552860312, 
0.52440421340590249, 0.54476652927003644, 0.60511911512744887, 
0.69322748526588729, 0.52584438557058644, 0.56161028207304409, 
0.54203040473434017, 0.47649801226980038, 0.53638559565796029, 
0.52460003299298685, -0.033380710930649697, -0.067255837238215696, 
0.10630012700232817, -0.2750251234377229, -0.026446589989634728, 
-0.17988514623570545, -0.29014589259595469, 0.22815332868766991, 
-0.26064168799476545, 0.078919775445920784, -0.034682191290044108, 
-0.029098636907761058, 0.2058594791122893, 0.1233179248499766, 
0.11267082755809239, -0.33378146585537632, -0.025070143118020589, 
0.075387202107385892, -0.34707904138934292, 0.054572225487373775, 
0.09993174615465944, 0.020325675137982453, -0.012298337479318628, 
0.18832867001436854, -0.040053865328434787, 0.11105902761482347, 
-0.043946946114069019, 0.067726601928213198, 0.11589240275773957, 
-0.033883154976237582, 0.072051396516612562, 0.17773105918110174, 
0.075529094363382621, 0.3696455906137272, -0.044212857550361577, 
-0.12534323618665791, -0.041462766128459928, 0.067960130740957581, 
-0.19009923819511251, -0.039724395132476741, 0.059496669966122899, 
-0.19340845358685302, -0.036020070594167924, -0.12682435432401856, 
-0.20814737571948683, 0.33679450539952932, 0.24997006781200615, 
0.028462952806768638, 0.0060757662671736233, 0.034615756330278281, 
-0.11018711981238337, -0.033029918756960867, -0.070853316944205089, 
0.027226164263615071, 0.14516557444405762, -0.19758976899079769, 
0.10907617248582145, 0.023618132222410975, -0.18030474192098231, 
-0.1818727230293995, -0.14456278557347621, -0.23686911288050255, 
0.31453973197290791, -0.11660612521459666, 0.12017911496680322, 
-0.16081764617689145, 0.075520790874858523, -0.13482233035285482, 
0.28077385567087143, -0.10689460421333083, 0.18302402465325671, 
0.1270314142765063, -0.27077617478834304, 0.17921215484864178, 
-0.2014424249073799, 0.19405568576971649, -0.10334678652224635, 
0.1110077309204579, 0.048522961600809239, 0.20343573732287323, 
0.021875068823156193, 0.13964825505570944, -0.023111343372104159, 
0.018243501750275965, 0.13778188930434526, -0.089391636311774769, 
0.014223195179074192, -0.162464853805208, -0.26739528867936685, 
0.063183357469879048, 0.21158285152160122, 0.21471966060283743, 
-0.21234438644480758, 0.32494496045336974, -0.23893976712633735, 
0.26396099493496117, -0.11149227855152959, 0.15191428322165718, 
0.045985809974443417, 0.032408416119157721, 0.84567418903541958, 
0.98888834509879009, 1.1063150847794576, 1.1511209380699354, 
1.0333026679075559, 1.1706899901049057, 1.0326262647223332, 0.89574846114422746, 
1.1307597556514908, 1.3339892543992056, 0.89334661909686996, 
1.0344156885754918, 0.94486495582658991, 1.0085372038880176, 
0.90668951186197255, 1.1514429895281277, 0.86400103424538499, 
0.62684350075695672, 0.9098972736992792, 1.2738531264730457, 
1.0617474407170433, 1.2299548162396312, 1.0489693570166876, 1.1151400242197704, 
1.2526712988951698, 1.3800282273005919, 0.83685855059326753, 
0.91614677444698012, 1.1342864344019261, 0.91346800961669239, 
1.1572072499600812, 0.79823474165403985, 1.0962042888632246, 
1.386269583538724, 1.272099865245393, 0.88474771599323943, 1.2367258257979141, 
1.106956602058961, 1.2781492194201856, 1.0767707919385587, 0.98484870016134607, 
0.93807731588275478, 0.97375243030647995, 1.0271594045285068, 
1.0667576699928294, 1.203537968873865, 0.98425028839584583, 0.94875424578768297, 
0.89970149071724448, 0.78504435266873285, 0.96707020806657684, 
0.80499957686940238, 0.97297906996307848, 0.92789942161045724, 
0.93701627913593855, 1.0675986336487939, 0.92209703997308068, 
0.92954409808055449, 1.1630846532205539, 0.84185330882049991, 
1.1789135227964018, 0.95955286539388274, 0.83645370050399259, 
1.0517990314913941, 1.0360816221488465, 0.80150847633502509, 
0.85991157654296491, 0.77383328531621087, 0.77170323178088585, 
1.0565633537644163, 1.0757161608040156, 1.1173869054534111, 0.90925635736997268, 
0.88067525535378122, 1.1301890737619986, 0.93073383291461043, 
1.1018071704883878, 1.2547320265562798, 0.77691186205640272, 
0.97336768316475608, 0.80486481045837588, 0.92192815838362674, 
0.80742426152348856, 0.86196148668731665, 0.83226439935960217, 
1.2164637575958159, 0.87321620863602334, 1.4128433657923551, 
0.95707725498769292, 0.60360620785553853, 1.0219569601877667, 
0.91740100354788512, 1.157856993051801, 0.65618042299410495, 
0.86509756309010499, 1.0540625639603771, 0.99601138216486373, 
0.8298911629007405, 1.1758722519250431, 0.907735465003222, 1.8406829312835107, 
2.2117326680455891, 2.0380672816028533, 2.1184398408583496, 2.0957116343211939, 
1.8715760117933034, 2.2265232010654725, 1.9076937057644738, 2.129328785803009, 
2.2212277857978742, 2.1787308611973319, 1.9494420818327685, 1.8613754990347444, 
2.0748926551792524, 1.8240212144167349, 1.8863927894882679, 2.0843739491129765, 
2.2133581095153319, 1.841945712682695, 1.9999751768472487, 1.7445331457050566, 
1.8078084446516849, 2.1354445836162141, 2.4031285570436127, 2.1073065995030871, 
2.1710864390970697, 1.6662546882933449, 1.7543696106999553, 2.0553694799456146, 
1.9636826029125198, 2.1147422312637634, 2.0387422762952734, 2.0229833725846849, 
1.7835413941412452, 1.9283343454920532, 2.0303607395184504, 2.2532121570857777, 
2.1240472666650394, 1.9390721213481326, 1.9268863425970333, 2.1108537726507275, 
1.7743726778195641, 2.0944283805190622, 1.8633030039033907, 1.8705955660553766, 
1.9714096501610738, 2.116751943142571, 2.0628384578664534, 1.9033751689572205, 
2.1871149090785065, 1.8521009467177831, 1.942518368108838, 2.1022518548780282, 
2.1181317470298477, 2.2200981898127194, 1.8917096011191099, 2.0247798035840638, 
1.853792351473682, 2.0932496615473966, 2.1008200509312691, 1.8993672085477176, 
2.0845930504843575, 2.3803665849607052, 2.1179076561287551, 1.9478777813864085, 
1.8411436690801881, 2.0469967938526326, 2.0838707618280856, 1.5554474683494859, 
2.1735629906723744, 2.088051568870966, 2.0353605751704085, 2.1623537578371996, 
1.8002397773517684, 2.0578529944343344, 2.1181630844988759, 1.9796618573260949, 
1.8221236537905112, 2.1668278221299913, 1.8593265300046122, 1.7683692264967212, 
1.7547274574627845, 1.9604150274066576, 1.9578764211816782, 1.6095951771166941, 
1.9712900762558168, 2.3098823117248535, 1.5377267204840828, 2.3119688683033468, 
1.6350363770669616, 2.1727261079687819, 1.7756682982105729, 2.117947162387865, 
2.0872502653437754, 2.049060446807919, 1.9620120052633814, 1.8114566327196346, 
2.00980531818993, 2.2135097487643201, 1.8043106999313738, 0.47119460994844226, 
0.45553618700839765, 0.66163076323343639, 0.58017755178969144, 
0.51153298942891046, 0.48475843226365817, 0.53579343246148359, 
0.49630718725140582, 0.53850849133155165, 0.57092524725068039, 
0.41331198322336549, 0.51752118783568257, 0.41714352703835378, 
0.61961490818596177, 0.32769640766523661, 0.54016709360804471, 
0.46839560045812362, 0.56421249800372231, 0.55833101072695324, 
0.59503044857144494, 0.40986750567181846, 0.43103414456259265, 
0.54493919694375581, 0.45553125094673597, 0.49025628674441002, 
0.58799711493524764, 0.56831867237026434, 0.68018806334488668, 
0.53863383225686157, 0.56675462383086894, 0.57986682047935889, 
0.61471860128261735, 0.48878769128654531, 0.53854019580996693, 
0.55101443451621424, 0.45056638791741294, 0.54461798051728494, 
0.54224415660147041, 0.53032610464261942, 0.50305094005357631, 
0.52376409954678549, 0.51773259240833336, 0.50242952177451938, 
0.55387263413430288, 0.52829784466172891, 0.50324960431444332, 
0.53126929823831281, 0.42576491332191607, 0.52465667398014915, 
0.46377009686577864, 0.54261210814422001, 0.54682346128092774, 
0.51539769617230236, 0.52081082376070675, 0.52689948007072041, 
0.43653032956808274, 0.61077402141941639, 0.47564312949756987, 
0.57379130000735634, 0.57778938115408429, 0.592698420517205, 
0.59104979099022925, 0.63883677568734643, 0.73097095409765667, 
0.50758989086331807, 0.28955466077490638, 0.47640100171787347, 
0.566601914908055, 0.60803489086389684, 0.43155551931562691, 
0.52318998391684157, 0.57371707543543904, 0.71827492534731718, 
0.51006073381030992, 0.53377124252154418, 0.6235339581640077, 
0.41743681293901358, 0.3970090911550721, 0.71879868442228079, 
0.61570955970158281, 0.42348349977127359, 0.4884762847477308, 
0.50121012053167768, 0.54118230176613735, 0.59068593327362606, 
0.59127403401768808, 0.65097515117726457, 0.39017521153644297, 
0.70794920369926462, 0.49400797527855311, 0.52343405679228905, 
0.54089232511670926, 0.60581607776848545, 0.69583982371887798, 
0.52588009839678063, 0.56058910318587563, 0.53974103149243902, 
0.47448676233194986, 0.5373913532257526, 0.52365314773396976, 
-0.033271496505233207, -0.067477854348646463, 0.10504964918815479, 
-0.27509466346060513, -0.026988731247942422, -0.1797271741635823, 
-0.28989986971253651, 0.22398503682632576, -0.26112528019014336, 
0.0782544842245992, -0.035103346327272293, -0.029007516981352729, 
0.20586049060821768, 0.12346377644991076, 0.11235455733372861, 
-0.33377095415543345, -0.024590482261964527, 0.075164986581518295, 
-0.34756372567283267, 0.055201565924759277, 0.099911888054486322, 
0.019067052904525844, -0.012491448060975577, 0.18792063540027318, 
-0.040537144500268234, 0.10940680860951836, -0.044804892716866773, 
0.066472482471242184, 0.11572877837020165, -0.033860731203268879, 
0.072446112182070199, 0.17809638556887031, 0.074540963870603397, 
0.36931828485767065, -0.044567603768886202, -0.12576966166372083, 
-0.044785984521696957, 0.068005407399029022, -0.19037069485153493, 
-0.040122790104415988, 0.059410079489920463, -0.19241973417677788, 
-0.036298111819748909, -0.12662322026504433, -0.20981037654161028, 
0.33645321145615448, 0.24984777871081618, 0.028238063110300464, 
0.0052799562607206557, 0.034725993163730123, -0.11031387383257883, 
-0.036283263076392493, -0.069351721103629388, 0.027102137909400949, 
0.14403137677926289, -0.19644287920381487, 0.11184697352964423, 
0.02229745807498059, -0.18084402716264167, -0.18203990024053809, 
-0.14417191625962533, -0.23686907494854462, 0.31506359342001339, 
-0.11879902654826131, 0.12006656080279053, -0.16122248874979039, 
0.075615159549587321, -0.13989883627902613, 0.28024663669057415, 
-0.10755363222663432, 0.18395742804669893, 0.12728549892299368, 
-0.27125302844026372, 0.1778021081294123, -0.2013836705500523, 
0.19113228001143096, -0.1033495585121222, 0.11062763964104864, 
0.04840618097430268, 0.20344812366282813, 0.021444957812680059, 
0.13901684440202969, -0.022616668205564006, 0.018109711937035852, 
0.13775875583614397, -0.090205452888811516, 0.013932028633045735, 
-0.16353063734815806, -0.26736604268345598, 0.062599854784935752, 
0.21119405231079674, 0.21364991956397422, -0.21264926577432308, 
0.32476097448729119, -0.23951201977381228, 0.26016711927721886, 
-0.10949964529446975, 0.15182484758414813, 0.045077917890471705, 
0.031607203879364928, 0.84590589378767223, 0.98869132430887441, 
1.0988398085496605, 1.1489288265803992, 1.0333627896081432, 1.1705412143826475, 
1.032811911930384, 0.8943168519709549, 1.1288165684367004, 1.3340640940474704, 
0.89335231422914696, 1.0338985842810267, 0.9446978771625314, 
1.0087273104359464, 0.90654609336819159, 1.1535855781291695, 
0.8639167847626732, 0.62663452599222291, 0.91017778866954557, 
1.2734718385596548, 1.0616266433670185, 1.229554941402595, 1.0486744435141024, 
1.115064541045286, 1.2520540082038063, 1.3797714503954976, 0.83695002288567755, 
0.91566410979147528, 1.1338367229534589, 0.91347095448319593, 
1.1560488697251401, 0.79977427675762924, 1.0957384260400147, 
1.3859837292500661, 1.2722057951254568, 0.88430119854513767, 
1.2259242552825185, 1.1069952683775868, 1.2778929261415191, 1.0763505529784752, 
0.98482627932645372, 0.93732410540281907, 0.97357728651294129, 
1.0279238693351256, 1.0624217532060798, 1.2036189030401709, 0.98415219144528432, 
0.94875612872777682, 0.89745242610945963, 0.78495086447392104, 
0.96722470373293323, 0.81067939165410119, 0.97220122737128079, 
0.9283434913936589, 0.93713017270807464, 1.0669333903218403, 
0.92930242152593356, 0.93006968833593018, 1.1641003684714082, 
0.84167624185229084, 1.1781948763336936, 0.95925678645299006, 
0.83575912195656021, 1.0517600356861296, 1.0364759598556625, 
0.80114911060191596, 0.8591168354920472, 0.77867048893127966, 
0.77170500603391823, 1.056485269916279, 1.0757255094509668, 1.1166677181478997, 
0.90948764600927534, 0.87909623960955474, 1.1301806344663474, 
0.9300251922785574, 1.1017340317730873, 1.2547507203674493, 0.77719448906329058, 
0.97339873194960624, 0.80483490955535153, 0.92149213029106902, 
0.80805353434462956, 0.86228810530605238, 0.83237470744930053, 
1.2167973204316271, 0.87322376899325005, 1.4120922648626906, 
0.95726841942465313, 0.60262392944915855, 1.0218935219276144, 
0.91947545194757618, 1.1575835503693284, 0.65627696598775964, 
0.86437916733108666, 1.050957339235105, 0.99474753150753381, 
0.83057993490347393, 1.1752988055283142, 0.9074360382951312, 
1.841070279248785, 2.2122431072569082, 2.0313286634830758, 2.1163219908747157, 
2.0952317053346063, 1.8719680888856738, 2.2269700798381034, 1.90408767531171, 
2.1274715998920302, 2.2196870989573227, 2.1790571197613131, 1.9491048944049958, 
1.8612733278174507, 2.0750322161624966, 1.8240884134379824, 1.8883950458891106, 
2.0844198997128967, 2.2133612561786817, 1.8405273339307355, 2.0007635560689496, 
1.7441536205876786, 1.8069317528665991, 2.1357781774717024, 2.4036200715146898, 
2.1077658153444667, 2.166702348427306, 1.6665270206564222, 1.7543367337556739, 
2.0551092503242905, 1.9637825720015971, 2.1158559567258446, 2.0394819305097562, 
2.023118223441795, 1.7831527449210376, 1.9285198787848279, 2.0293980711611095, 
2.2478949087582873, 2.1242593406155308, 1.939619834135146, 1.9265975149600227, 
2.111150659476043, 1.7752058186136546, 2.0945593761042174, 1.8630521130363822, 
1.8675681927838732, 1.9716774048959267, 2.116828190430557, 2.0637846185309003, 
1.9030340228400036, 2.1860936747585096, 1.852171128965399, 1.9420162387948507, 
2.1023597230479298, 2.1192497331005695, 2.2200840558151853, 1.8897313769945203, 
2.0230787477792402, 1.853945436981959, 2.094168667857466, 2.10062062165396, 
1.9007614349185153, 2.085277143873344, 2.3796069738819043, 2.1113462125522826, 
1.9478547666959021, 1.8410262058587419, 2.0470321669401255, 2.0814687174179549, 
1.5552658950037459, 2.173661595717602, 2.0892563663756816, 2.0356267658321845, 
2.162562605378898, 1.8001188985111676, 2.058014563720568, 2.1169212530369577, 
1.9793313781551503, 1.8219265940230456, 2.1629363629750102, 1.8592511424254234, 
1.7683376840991105, 1.7547248716401158, 1.9579688555637758, 1.958368383530795, 
1.6095411651044278, 1.9707752347614496, 2.3104614476240548, 1.5363912356709757, 
2.3122432594619209, 1.6345040411160707, 2.1729478261491213, 1.7754954978592794, 
2.1177311673651067, 2.0871764844341412, 2.0488131157491849, 1.9594826645920711, 
1.8094356638126083, 2.0101703701991744, 2.2136432089714297, 1.8048730732485143, 
0.47276303022334454, 0.45603530268960452, 0.66948628993211323, 
0.58021970938848899, 0.51361895378745304, 0.48598840656818798, 
0.53630397546455999, 0.49839457247803198, 0.54344641375827163, 
0.57029692845900137, 0.41428112076654566, 0.5178583062351898, 
0.41790470396010632, 0.61995483915610505, 0.32814376524644084, 
0.54226035528353922, 0.46919600541804901, 0.56372695897757652, 
0.55976374233443305, 0.59789547766297391, 0.41021625565393049, 
0.43219500293261515, 0.54592834683406477, 0.4565848454496409, 
0.49150493390440264, 0.58755173348695411, 0.57005969404369961, 
0.68194507459507969, 0.53746277423196853, 0.56706644244396698, 
0.5825508185687851, 0.61375604279118035, 0.49067441273618884, 
0.53903020932287393, 0.55370253298301764, 0.45192531204829312, 
0.5467281037724665, 0.54188050168470669, 0.53178305305034279, 
0.50266451984567351, 0.52408873387622334, 0.52090376944843308, 
0.50194939639204372, 0.55608214560989833, 0.5315906570474509, 
0.50434970089600828, 0.53135675753169687, 0.42630961894238362, 
0.52629479169528248, 0.46408505735863231, 0.54307660826345638, 
0.54819260395827818, 0.51800967780460982, 0.52224476241037299, 
0.52931490964865646, 0.43875945936107635, 0.61660712731030998, 
0.47609155068274267, 0.57481672165028186, 0.57702664543856319, 
0.59484006095425368, 0.59099990491380061, 0.64225368042807141, 
0.73530338348369118, 0.50814718938825509, 0.29005620304931251, 
0.47661475623878885, 0.57009790835723284, 0.60773021172971187, 
0.43209444637193306, 0.52615354135459047, 0.5764162816934103, 
0.71742005276173471, 0.51043150569352136, 0.53451092434857994, 
0.62803537568279288, 0.41795616550880987, 0.39789902871931149, 
0.72715818795730813, 0.61528127063315607, 0.42401962527822851, 
0.48932161206675556, 0.50298968762629859, 0.54205789268939319, 
0.58976939238882764, 0.59125911017247257, 0.65109484591703182, 
0.39096078901826747, 0.70722004883358125, 0.4957897552860312, 
0.52440421340590249, 0.54476652927003644, 0.60511911512744887, 
0.69322748526588729, 0.52584438557058644, 0.56161028207304409, 
0.54203040473434017, 0.47649801226980038, 0.53638559565796029, 
0.52460003299298685), algo = structure(c(2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), class = "factor", .Label = c("mtram", 
"tramME")), which = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", "beta2", 
"beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-1600L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
)


## ----sim-tramME_IC10, echo=FALSE, eval=FALSE----------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## beta <- beta[-1]
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## res <- vector("list", Nsim)
## 
## aa <- function(i) {
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   y <- qt(plogis(x %*% beta + rnorm(N, sd = tau)[cls] + rlogis(nrow(x))), df = 3)
##   ##   ^^^^^^^^^^ <- h^{-1}
##   d <- data.frame(y = y, x, cls = cls)
##   d$yS <- Surv(floor(10*y)/10, ceiling(10*y)/10, type = "interval2")
##   m <- Colr(yS ~ X1 + X2 + X3,
##             data = d, bounds = c(-Inf, Inf),# add = c(-100, 100),
##             support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
## 
##   mm <- ColrME(yS ~ X1 + X2 + X3 + (1 | cls), data = d,
##                bounds = c(-Inf, Inf),# add = c(-100, 100),
##                support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   return(list(rmt = c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                       diag(solve(mt$Hessian))[-(1:7)]),
##               rME = c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)]),
##               lmt = logLik(mt),
##               lME = logLik(mm)))
## }
## 
## for(i in 1:Nsim) {
##   res[[i]] <- try(aa(i), TRUE)
##   while(inherits(res[[i]], 'try-error')){
##     res[[i]] <- try(aa(i), TRUE)
##   }
##   print(i)
## 
##   rmt[i, ] <- res[[i]]$rmt
##   rME[i, ] <- res[[i]]$rME
##   lmt[i] <- res[[i]]$lmt
##   lME[i] <- res[[i]]$lME
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"),
##                                          each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 2)))
## 


## ----sim-tramME_IC10_plot, echo=FALSE-----------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(-beta, tau)

coefs <-
structure(list(coef = c(0.17662862702935223, -0.053347164063200836, 
-0.16630821334826915, 0.084291034265124631, -0.40129582243451045, 
0.63932264416132256, -0.36909788552943285, 0.17524466035107678, 
-0.10552600452443697, 0.39983405517696186, -0.26225495987128117, 
0.11109894565419431, -0.059392824065891879, -0.13649986971391534, 
-0.13056114027434149, 0.015656118368092189, 0.44054994684193088, 
0.24704572136454639, -0.12847604623939127, -0.37031468123179739, 
-0.16484474206488023, -0.50640776030208801, 0.13948774913886747, 
0.29938004280316738, 0.10032017768902372, -0.32225220626965567, 
0.070850186127783793, -0.31772065866050087, -0.44678289367961543, 
0.43557657779181658, -0.090175375888267753, 0.26593497850298958, 
-0.52252048803723994, 0.037870924751918743, -0.2915932461467205, 
-0.19719404729571297, -0.083744493995233621, 0.14053142175589839, 
-0.022537816976945048, -0.52944798642986068, -0.4919591514330105, 
-0.14484316595388669, -0.49915923076214613, 0.4367603308018328, 
0.058402668887347818, 0.0044681253075181896, -0.41159780370879961, 
0.63745210427726096, 0.18453919539563388, -0.1942169669006453, 
-0.072588036688493604, -0.48014523927705272, -0.35969442477418251, 
-0.54825728672115392, 0.14839697635326571, 0.41561518796179603, 
-0.12512014386805875, -0.16673881795770612, 0.073201105223101068, 
0.22257717591752846, 0.32585067545793034, 0.17648653527931854, 
0.54320829802226189, -0.089142322775651178, 0.72504674114953738, 
0.17035540783086903, 0.048881450687515267, 0.33602326002520028, 
0.27529776237345471, 0.48853639733273885, 0.50819589379078911, 
0.062186368840213613, -0.24714709736967189, 0.068257923320137054, 
0.2793269069306441, -0.17422304603631836, 0.069317068632171935, 
-0.11956173045638546, -0.52586667691339539, -0.039828662999808306, 
0.14656875855161342, 0.3417187533568759, -0.42319288122053789, 
-0.068813493697898048, 0.0035475656585002261, 0.31098350036741096, 
0.86399185202278694, 0.20464790993268397, 0.39866337057345141, 
0.10597358965678068, 0.10933576637454614, -0.44589708746242696, 
0.06218539543548756, -0.098642770027185911, 0.32155312830141214, 
0.12074407197364011, -0.13244070422178061, -0.084011733713165351, 
-0.14996981186512609, -0.52209045101929419, -0.91456333427979486, 
-0.98492953965776209, -0.42168242064994932, -0.40379978380479559, 
-0.75200163588133528, -0.88941119622022946, -0.82455196722320867, 
-0.9947568284134366, -0.82838332193390285, -0.5127340918303781, 
-1.2488489552440145, -0.5301980979354104, -1.0906292427558906, 
-0.80153869266333722, -0.66569061695641374, -0.53254289812728628, 
-1.1616382608763125, -1.3447974817341604, -0.85196300720833751, 
-0.53483895985054297, -1.3760292769286906, -1.0006701244514433, 
-0.93277678498136762, -0.96943037964673195, -0.50674663800296127, 
-0.81987017401856588, -1.075857199656342, -0.76334976769430107, 
-0.48910717307914497, -1.0537057660050961, -1.4381230658894897, 
-0.98786554996985521, -0.86747670582239889, -0.96116175933932468, 
-0.99511176961233849, -1.1354771342118624, -1.3446656206744561, 
-0.83359857680162552, -1.0356166044910249, -0.63838506024017316, 
-1.3508972055687258, -0.78432852180998991, -0.79363877137162331, 
-0.24994730487706251, -0.75077259342630664, -1.1476879605499712, 
-1.3751448577544372, -0.83792795550397192, -1.3009156446653309, 
-1.0029168176827314, -0.91975512222266842, -1.1991083949848917, 
-0.90903417344596671, -1.0897576183921129, -0.81346700960352281, 
-0.85955513200224876, -0.7128242329870409, -0.80511035300421308, 
-1.4102169260879605, -1.1338974901156287, -1.1400117869665389, 
-0.75316679082132287, -1.309757774545981, -0.69951417470470123, 
-1.4156567481158144, -0.65325804939664611, -1.0193414561025091, 
0.12431841441001927, -0.36541101443736462, -1.5161210693300464, 
-1.15054598584809, -0.68842240568981805, -0.86578803829097239, 
-1.2274009837553499, -1.2114701800908811, -1.1423338581533833, 
-1.1343446295625961, -0.93797810085204048, -0.85441326716745503, 
-0.98539839122485373, -1.4483418900405682, -1.3310258028490169, 
-0.64496430391907766, -1.1048427922337263, -0.98055635567623323, 
-1.2141392384319714, -1.0571649102448486, -0.86910982594195041, 
-1.2336215339076637, -0.50889764424764206, -1.0853743422790429, 
-0.68878597506233474, -0.60070737517264505, -0.85683363894233266, 
-0.55947498935374673, -0.85750713804333512, -0.88881649658674799, 
-0.52709904575102839, -0.56632352041136913, -1.2332080414387303, 
-2.317146460652677, -1.5405063675176673, -1.7772568614114566, 
-2.0255721830446487, -1.8405769029745593, -2.2831209121705909, 
-2.045041381852049, -1.9430914463268771, -1.8676458046469364, 
-2.1995336825978224, -1.9610976201353063, -1.9790115810991158, 
-2.0684727936266896, -1.9308712804849206, -2.0525432368167604, 
-1.5783809692350936, -1.9514168105854794, -2.2555419715382174, 
-1.88440101592556, -2.4675245934320733, -2.7712705324987859, 
-2.1046435242499548, -1.9160628686040782, -1.5946837881398022, 
-2.3544404971692687, -1.5250223356209545, -1.9919117602039458, 
-1.9771554054865665, -1.9223834359214316, -1.8845840251903032, 
-1.8134808402942235, -1.7676274579650177, -1.4959940573975015, 
-2.1920810021517902, -1.9510471721636522, -1.3517851973806043, 
-1.8430790215666679, -1.7622301746771627, -1.8604539129009077, 
-2.2557129577330275, -1.8131300034870363, -1.5706583887651244, 
-1.6820838249431225, -2.0854122685465852, -2.2262287863138606, 
-2.3970613552365156, -1.8731983494622251, -2.1459648037338082, 
-1.7899460762877222, -1.8849855613753685, -2.0561699087329406, 
-2.0125039022453537, -2.0108290460930274, -2.1420122156226422, 
-1.6437810973455518, -2.2392400543599376, -1.2770481426099451, 
-1.7964568920532105, -2.0115317286940861, -2.4717364184821071, 
-2.1913738299632195, -1.607764283516264, -1.4378588764598841, 
-1.8605458384278697, -1.6715925694622664, -1.7966401018404006, 
-1.9658387312692669, -1.822817206290988, -0.94161794718329594, 
-2.7389251112102588, -2.3670717019810801, -1.8160520285345012, 
-1.7335465091931104, -1.8240511109624835, -2.0683805961074309, 
-2.0359000122139963, -1.1326734702603178, -1.3417278764711971, 
-2.1020922045076698, -2.2746230830838212, -1.9081979726234697, 
-1.8158231930418409, -1.4652779802217142, -2.5686777484800505, 
-1.8230843023242871, -2.2847053417968439, -1.9067196754502509, 
-2.0519892228366459, -1.5510604947266029, -1.927398871423893, 
-1.9898066553159119, -2.2228801945142975, -2.4029084484423562, 
-2.0416550856022067, -2.1432104464244275, -1.8138158222582577, 
-2.0374376051009651, -1.6913422310206683, -2.2961293340010465, 
-1.5168698139475578, 0.3522409768388936, 0.48359540676593143, 
0.44949150136056876, 0.45027487308483888, 0.4961085990180687, 
0.56846310581784376, 0.52328875628789029, 0.46159490874648462, 
0.53631667534241734, 0.47534049043156601, 0.33463878431981858, 
0.52486397591397405, 0.51782627894024436, 0.20847551086926244, 
0.30979817117500902, 0.44064715446692876, 0.29949506706398449, 
0.65088472754683924, 0.41782744352519324, 0.39787417630098698, 
0.52977865700343829, 0.54059345128924019, 0.21970066313580547, 
0.46856348851710689, 0.38512456850445759, 0.26227024657435011, 
0.43660332132769747, 0.6192603810687668, 0.27243189405203372, 
0.3475908035211413, 0.55213475061125861, 0.5248332186909147, 
0.68380693604115084, 0.53509347345437053, 0.30543763716568073, 
0.44621168980549386, 0.33693098378450154, 0.43592498147185621, 
0.00040210115612861517, 0.18936534255637744, 0.32069621097907069, 
1, 0.44090454156952086, 0.48356780652594505, 0.58657627959106251, 
0.58819276004330479, 0.54565521128345673, 0.81010413851409058, 
0.44154871925883132, 0.41867640865320005, 0.45720797972634375, 
0.56088781151827238, 0.49519352687616186, 0.40598697057164768, 
0.29047946684603487, 0.51497385006758123, 0.6853026145448059, 
0.5156455369740528, 0.48541896085320568, 0.59559040700161192, 
0.27452782195835485, 0.34506561175774197, 0.6366040458143537, 
0.57786833962602946, 0.63772177260572871, 0.52566552252595189, 
0.40951248281780595, 0.46428221957793103, 0.53569603388294329, 
0.65939059261428923, 0.0027687096604416591, 0.21596788737615227, 
0.6427195406084466, 0.43260844031845708, 0.65853807046388457, 
0.6425441874349086, 0.38245844774833143, 0.35987609898099898, 
0.77020292178385275, 0.78247923121300722, 0.5109847569467405, 
0.58744238126906723, 0.42676580649282664, 0.60062034361655658, 
0.7018908187672811, 0.54974232524299183, 0.64294031101560833, 
0.40218860738826773, 0.40604949867271184, 0.5370656150726395, 
0.48894989590125099, 0.1854464522970532, 0.52552815624218163, 
0.57229836202659368, 0.60227019362031053, 0.33295072832720035, 
0.39252070957785673, 0.47110984885938328, 0.52685144880905543, 
0.22981599222731441, 0.17376230735243123, -0.049519650141065547, 
-0.16953453924452302, 0.10757441611872109, -0.38604571250610409, 
0.64411849410108335, -0.37946776835071633, 0.14718569528229108, 
-0.13391594680695371, 0.39991117155328387, -0.29872371595516417, 
0.11098067495348532, -0.058179913405844733, -0.13426822421212842, 
-0.1084750416216369, -0.0039587097662494341, 0.42371086822921789, 
0.27953923513097112, -0.1083862568108855, -0.35079002337316184, 
-0.18349156495829028, -0.50691098347296371, 0.13720984450247303, 
0.30209094401146952, 0.091254802340593424, -0.33290037327394179, 
0.041618163178098057, -0.43986287730135798, -0.44836688629466603, 
0.43285783009702272, -0.071484599874824889, 0.26566109207418775, 
-0.47715535452788715, 0.039601298216117339, -0.30374789732739849, 
-0.18463597328714262, -0.078415380026349943, 0.12065031874121875, 
-0.041661540112075957, -0.52343823049369376, -0.55449844405319249, 
-0.22785266053714917, -0.5044034315000876, 0.38780884546362099, 
0.065695511300163878, -0.032069379968500254, -0.42714041301009353, 
0.64013003353814224, 0.15868917076225456, -0.19843480094102453, 
-0.11075112573182908, -0.50056086520029142, -0.3669377730659199, 
-0.54321774136703627, 0.1504952462307409, 0.40267403692595921, 
-0.1351502501756405, -0.14508684653876314, 0.078373159741588649, 
0.2331264552977329, 0.36892898576950323, 0.19050884666336942, 
0.51939198765607841, -0.075122950425171253, 0.63748571162096523, 
0.19349320481054172, 0.038113893660567805, 0.38281915868556943, 
0.25199725059370537, 0.4968055461046737, 0.53406409434559565, 
0.10639374248550448, -0.28476759562745996, 0.060944968318235593, 
0.29271909799503631, -0.16698146280573295, 0.055734386649650809, 
-0.1125642690069974, -0.50928507501259901, -0.030596329595635775, 
0.12911166317009889, 0.34045088106381882, -0.43076403050474904, 
-0.068681292588779502, 0.0032493517290941689, 0.33678251233881923, 
0.84931802106023901, 0.19587152741848951, 0.41111842568626189, 
0.10071154544603321, 0.15473061551133493, -0.43955033561280465, 
0.059388381208640935, -0.087148738205143517, 0.35298321092074048, 
0.10071325978381082, -0.12934613914036575, -0.05369613683324951, 
-0.17867559983004236, -0.52726070173550321, -0.90573851278482853, 
-0.99419222568503385, -0.42381825885644026, -0.39597835107142482, 
-0.74284304715188598, -0.86933214797037495, -0.84533328940564956, 
-0.9791329525825988, -0.82898218957441505, -0.50556704418616416, 
-1.3413299087307471, -0.54276788160125622, -1.0208315701901998, 
-0.79806174224311655, -0.65947809231684917, -0.50135896573575589, 
-1.1458852207833623, -1.3434594230408754, -0.8283738268901738, 
-0.52036305177843234, -1.3748189927838261, -1.0140144879692208, 
-0.94130155955049721, -0.97310252285093035, -0.52469261536729239, 
-0.79580768970596949, -1.0762768930390734, -0.78798030396030583, 
-0.46765195467617443, -1.0224695132353556, -1.4443670468289531, 
-1.001913402483249, -0.91155129874327334, -0.94752650683061246, 
-1.003030758813189, -1.0782050876988807, -1.3118503209497205, 
-0.77858974868380104, -0.98801552691038219, -0.62040103313832184, 
-1.3474802634789493, -0.8396675136930527, -0.79600677147923515, 
-0.24794254513932737, -0.72628389706572316, -1.079620750649511, 
-1.3812921338090944, -0.84086607561011895, -1.3021087658601522, 
-1.0245999490496678, -0.90367028901030511, -1.1809184109282425, 
-0.90523577343670913, -1.0892861137466769, -0.79872919765297468, 
-0.84771543933950544, -0.75187068137236479, -0.80469779642994743, 
-1.4081619948078645, -1.1449698646846702, -1.1261446813860714, 
-0.74479968387725093, -1.2546855111900648, -0.69294399308781041, 
-1.3670842901445874, -0.63229846068078444, -1.0061933314265119, 
0.12448540996288587, -0.39022817953208494, -1.5067249816099215, 
-1.2093603226215861, -0.6813093181804466, -0.83257137462779107, 
-1.2524817857673942, -1.2408644626691847, -1.1143527826462432, 
-1.1338851321517489, -0.90393299055013332, -0.88838431334300838, 
-0.95275961058353542, -1.3953317333600381, -1.2910635057869955, 
-0.61815532128652262, -1.0752549686674759, -0.94610149505043695, 
-1.2117850957789282, -1.0681342082110374, -0.86773523175215439, 
-1.2457766528194518, -0.53259811644227573, -1.0720411980823841, 
-0.68461950079824041, -0.57796538757070093, -0.87074218018460081, 
-0.52561346362687333, -0.84206723798132521, -0.88818116409376202, 
-0.51856134173986224, -0.59832409559364419, -1.2327751178447204, 
-2.3101266821818336, -1.5446415361501644, -1.7233950539823382, 
-2.0157559627965016, -1.7770950721507393, -2.2625425935053416, 
-2.0443896291123456, -1.9427960748974025, -1.7684997741793811, 
-2.1730114866191057, -1.9999659130079597, -1.9757722836798415, 
-2.0068437116037567, -1.9175570747521344, -2.0614651731939961, 
-1.5402418022625139, -1.9108551206926672, -2.2158736739114349, 
-1.834438655544163, -2.4243924504144734, -2.842683062279812, 
-2.0955685377843358, -1.925515591255434, -1.6020957995570786, 
-2.3670597922528742, -1.515148219486153, -1.9887206392038366, 
-1.9964191263177524, -1.9165710408791761, -1.8520102560619591, 
-1.7988640804074869, -1.7590208853547196, -1.4799681601267001, 
-2.1807193316478295, -1.9759248802639862, -1.2809738927520697, 
-1.8433597583158856, -1.7466669910454224, -1.810578632188993, 
-2.209744997045004, -1.8197970095845912, -1.5946493771091472, 
-1.6615025733798403, -2.0280799686938584, -2.223599579956383, 
-2.2868542971945884, -1.8468680800049151, -2.0837398904813287, 
-1.7917064306085264, -1.9095814581392465, -2.0027572731293235, 
-2.0056490384385293, -2.0211995641287719, -2.1326714211026077, 
-1.6057675121559483, -2.1833560720390279, -1.2826313506985789, 
-1.819575308160595, -1.9922466535064229, -2.4694550984453376, 
-2.2307661248473196, -1.5901790222623977, -1.3463486913380873, 
-1.8280270711719411, -1.6102247713041451, -1.8043041394228037, 
-1.9405934623014045, -1.7895803436583539, -0.93026373515029359, 
-2.7032516059041045, -2.3415814135884836, -1.6802720421775568, 
-1.7187958330987467, -1.8526615112881468, -2.0463016602252213, 
-2.0149378234348534, -1.1395298060421064, -1.2864523894480306, 
-2.0457293465598392, -2.1731506890880277, -1.8585712641844212, 
-1.7271877759496763, -1.4598970126714645, -2.4876645334333007, 
-1.7887232840367644, -2.2613715114448327, -1.8645323001692735, 
-2.0333369041679861, -1.551332804961729, -1.8760360389116544, 
-1.953299322299983, -2.2196531302359053, -2.3142199467032185, 
-2.013297307100181, -2.1136778994817318, -1.7870137558133312, 
-2.0335713691500712, -1.6585929486692403, -2.2732050365244372, 
-1.52998466950652, 0.22182246527630392, 0.32363350718982126, 
0.27837002891981255, 0.30292064779219735, 0.30372492321138456, 
0.33356940525436624, 0.33377692397712783, 0.25980653134900439, 
0.27601291623529484, 0.29635064736171368, 0.21977497447832545, 
0.3084419393424726, 0.29356960345833066, 0.14617572547184834, 
0.2265789135426379, 0.23875051865480426, 0.16870724366317288, 
0.40419693103580501, 0.25912439356860828, 0.25019802262222895, 
0.35275031890400133, 0.34587155870148856, 0.12794119480992822, 
0.28508313553739911, 0.28485473698906105, 0.23540773940066889, 
0.29050505250822006, 0.37034323725447738, 0.18028782870941107, 
0.19060481466510493, 0.32549281505900785, 0.30359535392400178, 
0.39768000673911369, 0.32322170314795934, 0.23176593141661198, 
0.23229701576556325, 0.1674416602129831, 0.2826831642472542, 
0.10818705832254717, 0.11474948958530877, 0.22498056560150978, 
0.32389145539577791, 0.30529255985875148, 0.29204229606622406, 
0.33366950890017921, 0.34120707310353704, 0.34008006902206128, 
0.4784598203929879, 0.26566044167188596, 0.3148055616999707, 
0.21515187802909755, 0.35555694434505297, 0.30053381795872275, 
0.25075151312105387, 0.21861944691776866, 0.31070080647684262, 
0.40359968222897658, 0.32495519482858537, 0.3070191160229685, 
0.38378306127764433, 0.21560722252868322, 0.21100887101140037, 
0.36014286103897891, 0.33713862733384903, 0.36910890294964621, 
0.32199372332364296, 0.2424608542854941, 0.2988418686822002, 
0.31238810823531427, 0.41589360498218841, 0.12180782230413628, 
0.16250781866215469, 0.38612455522609995, 0.26826329754966971, 
0.42130496874902817, 0.38731915821033347, 0.25713507463207275, 
0.26394229643282674, 0.44534232016221287, 0.42756640688630349, 
0.29825074691180725, 0.34797081452894574, 0.29020595034561286, 
0.38167419191290902, 0.41290876483770417, 0.38524399152685262, 
0.35476980832850952, 0.23946116624191541, 0.27169341538262248, 
0.28980871036641748, 0.31085242335117402, 0.18475371328538959, 
0.31328227078156506, 0.34741260493553328, 0.36039049788010835, 
0.24503281256380507, 0.26096505091444555, 0.26679579452912933, 
0.29787448399889632, 0.18233163498869151), algo = structure(c(2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), .Label = c("mtram", 
"tramME"), class = "factor"), which = structure(c(1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", 
"beta2", "beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-800L))


bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
)


## ----sim-mtram_IC10, echo=FALSE, eval=FALSE-----------------------------------
## ### 100 clusters of
## N <- 100
## ### size 5
## Ni <- 5
## cls <- gl(N, Ni)
## tau <- sqrt(3)/pi
## p <- 3
## beta <- c(-.5, 0, 1, 2)
## 
## set.seed(150594)
## beta <- beta[-1]
## rmt <- rME <- matrix(NA, ncol = 4*2, nrow = Nsim)
## lmt <- lME <- numeric(nrow(rmt))
## res <- vector("list", Nsim)
## 
## Ui <- matrix(1, ncol = 1, nrow = Ni)
## S <- tau^2
## Sigma <- S * tcrossprod(Ui) + diag(Ni)
## D <- diag(sqrt(diag(Sigma)))
## 
## aa <- function(i) {
##   Z <- rmvnorm(N, sigma = Sigma)
##   x <- matrix(runif(N * Ni * 3), ncol = p)
##   h1 <- function(x) qt(plogis(x), df = 3)
##   # ^^ <- h^{-1}
##   y <- h1(c(D %*% qlogis(pnorm(solve(D) %*% t(Z)))) + x %*% beta)
##   d <- data.frame(y = y, x, cls = cls)
##   d$yS <- Surv(floor(10*y)/10, ceiling(10*y)/10, type = "interval2")
##   m <- Colr(yS ~ X1 + X2 + X3, data = d, bounds = c(-Inf, Inf),# add = c(-100, 100),
##             support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
##   mt <- mtram(m, ~ (1 | cls), data = d, Hessian = TRUE)
## 
##   mm <- ColrME(yS ~ X1 + X2 + X3 + (1 | cls), data = d,
##                bounds = c(-Inf, Inf),# add = c(-100, 100),
##                support = c(floor(min(y) - 1), ceiling(max(y) + 1)))
## 
##   return(list(rmt = c(coef(mt)[c("X1", "X2", "X3", "gamma1")],
##                       diag(solve(mt$Hessian))[-(1:7)]),
##               rME = c(coef(mm), sqrt(VarCorr(mm)$cls$var), diag(vcov(mm))[-(1:7)]),
##               lmt = logLik(mt),
##               lME = logLik(mm)))
## }
## 
## for(i in 1:Nsim) {
##   res[[i]] <- try(aa(i), TRUE)
##   while(inherits(res[[i]], 'try-error')){
##     res[[i]] <- try(aa(i), TRUE)
##   }
##   print(i)
## 
##   rmt[i, ] <- res[[i]]$rmt
##   rME[i, ] <- res[[i]]$rME
##   lmt[i] <- res[[i]]$lmt
##   lME[i] <- res[[i]]$lME
## }
## 
## coefs <- data.frame(coef = c(rME[, 1:4], rmt[, 1:4]),
##                     algo = as.factor(rep(c("tramME", "mtram"),
##                                          each = Nsim*4)),
##                     which = as.factor(rep(rep(c("beta1", "beta2",
##                                                 "beta3", "gamma"), each = Nsim),
##                                           times = 2)))


## ----sim-mtram_IC10_plot, echo=FALSE------------------------------------------
tau <- sqrt(3)/pi
beta <- c(0, 1, 2)
true_coefs <- c(-beta, tau)

coefs <-
structure(list(coef = c(-0.053190784540107519, 0.16375362366848162, 
0.4391957833744915, 0.11857940790628614, 0.31904899332098546, 
0.4716789929725968, 0.1207471066651134, -0.71201059632378749, 
-0.036740481620781071, -0.20447577717574034, -0.12491076570306123, 
-0.18945132837678813, -0.31621011586858971, -0.30847331565709446, 
-0.077753949344626316, 0.030213879032077041, 0.061853053966510661, 
0.13460668664495806, -0.3218116967183266, -0.072818669382052767, 
-0.16080788928673226, 0.26664408426566555, -0.44241541189644967, 
0.33654757242210892, -0.06089422036781434, -0.021112509807258338, 
-0.08254906003646989, 0.79348265733354162, 0.033750201597541614, 
0.41834785148386083, 0.44801497577793314, 0.21992819142382608, 
-0.10143244240485874, -0.13351546569753811, 0.12660490255996013, 
0.094737887304191803, 0.24713219056793889, 0.13011758581962798, 
0.14892223774261604, -0.042445569223334721, -0.5069395010406994, 
-0.11779784467868058, 0.0075055775275962144, 0.20347282436234906, 
0.46797092670919127, -0.0042693853555648108, -0.27102789272406896, 
0.095345051667613492, 0.66919090077600785, 0.43398855402982495, 
0.30359045059274359, -0.3081565771812852, 0.01785987350130527, 
-0.22168634351434979, -0.13218264236033778, -0.13232135102750872, 
-0.13579635715672372, 0.10568211379767918, 0.17627451826922411, 
-0.41499066507984922, -0.046615498740974509, 0.015357588720835379, 
0.45153061729884625, -0.083872519500295756, -0.49319250116060781, 
0.30294886613471356, -0.43421823988499125, 0.083521340356999402, 
-0.32635614701017768, -0.25474329742345275, 0.13490593784408239, 
0.0091679822311137044, 0.0096823413072637324, 0.47129434420119853, 
-0.025063231842043433, -0.1454455889138008, -0.20619143477135998, 
0.33834905700893192, -0.59076013144946693, -0.090729411794611609, 
-0.043112769453885348, 0.24906273071691626, -0.16487788370818668, 
-0.12060078502996548, -0.15674982879891516, -0.48261097727614688, 
0.32019222256975671, -0.028675874447673666, 0.078872773351546147, 
0.60863219981394645, -0.054344751381974261, -0.43564097461238438, 
-0.1184054523858819, 0.52971019145550269, 0.10613827029371534, 
0.0059498152856215141, 0.37756120727351644, -0.35871400470875109, 
0.055664354733592977, 0.44599709167270818, -1.2374388783138932, 
-0.98695840683829505, -0.99096280836094908, -1.3084276554138807, 
-1.4279516674163812, -1.1810976941284148, -0.96953861314909839, 
-1.1614404666870346, -0.86570324539058796, -1.3425568669342283, 
-1.2292184669113182, -0.72827741011261693, -1.3113382845722998, 
-0.60425724141893611, -1.2851834734648779, -1.5220040236182706, 
-1.1063617166924133, -0.85701789645290194, -0.62673151495030155, 
-0.87624238113044539, -1.14078843518108, -0.72782108795016787, 
-0.75869434503661792, -1.090045751136979, -0.71384361319965117, 
-0.92199409510840735, -1.2843615511179824, -0.61103444163654574, 
-0.98199793959830117, -1.0811769391524046, -0.92189515888246598, 
-0.98573086086502582, -1.3341976449972266, -1.0353573233378943, 
-0.83457595734404544, -0.94082047504756938, -1.5714700587475221, 
-1.1314389077432931, -1.4702166582304874, -0.86560512301067016, 
-0.5796409887300169, -1.2041200437309678, -0.52640188137181798, 
-1.6143825359405048, -1.3027457950191828, -0.7653183426165594, 
-0.71311288140135098, -1.084521716218702, -0.91859452817319631, 
-0.81792117855225899, -0.89082661353857551, -0.77825805033318052, 
-1.1234190380238165, -1.0935850974818475, -0.90634427916046556, 
-1.0474163412023136, -0.86208200030933158, -1.0744994373506829, 
-1.4349415764893989, -1.4057464999391895, -1.1662226966466032, 
-0.84269479571722883, -0.11533589757064772, -0.77820710270654614, 
-1.2714406220893537, -0.64103543625640824, -0.79289680366388982, 
-0.041618311861118251, -1.1751635051926883, -0.8175656737412057, 
-1.2184147954460378, -1.1277933174392796, -0.88293827838959937, 
-0.97555782098965604, -0.49828600540984141, -1.2856984347385452, 
-0.87445795831733353, -0.8720270136023619, -0.88122297170777153, 
-1.0969998462740698, -1.3729866938790998, -0.84250290818583662, 
-0.37215161446265177, -0.8122264515650337, -0.97852289771384415, 
-0.87293594082240666, -0.67643845327596397, -0.56629459875320498, 
-0.76749384427517497, -0.47761439813855344, -0.23591223962881361, 
-0.80701373154742451, -0.78433738631670491, -1.1854019063565409, 
-0.81658372496956844, -0.85811064748539445, -0.66209723409498245, 
-0.8208973180408492, -1.1053322401801098, -0.82061704413188652, 
-2.0507196232469043, -2.1794860816249075, -2.0262558726944517, 
-1.8087997630783377, -1.6194605836679372, -1.627055308779181, 
-2.1453777403132492, -2.3880256618637143, -2.22698896997735, 
-1.3025374853518223, -1.570572621090216, -1.7102751691465041, 
-1.9503953811102457, -2.2331528849840629, -2.2970496081264433, 
-2.0243669217987765, -2.1142997278810869, -1.9708378604793617, 
-2.4304085596890785, -1.6732612176945065, -1.7541140224170217, 
-2.3487394218022191, -1.6518391145155835, -2.3790201126668995, 
-1.9558161881436922, -1.7684909403859086, -2.2284550305370057, 
-2.0836506236612347, -1.5717124082785905, -1.723113432902676, 
-1.7816798251326851, -2.0005247760359279, -2.0160517156246418, 
-1.9069395664739546, -1.6134788852091195, -1.387480445988343, 
-1.9866835684663298, -1.8705048523512178, -1.9729228798971621, 
-2.2058386120492415, -1.6858349239316417, -1.6409790636346873, 
-2.2749788447190737, -1.6932297885021226, -1.8537102581712961, 
-1.8649811332904114, -1.8589265983595309, -2.1092266057923474, 
-1.7032880120013656, -1.6180875037383875, -2.0912831790276618, 
-2.2679582399624256, -2.4834288046180193, -1.7653890831023322, 
-2.4656211460805175, -1.7762753289848232, -2.2751961231271336, 
-1.978115135593528, -1.9705123785746697, -2.0714073262791466, 
-2.1613711050088478, -2.0249428703827799, -1.4264462448794357, 
-1.8570190606985897, -2.1941981978270251, -1.9219337633686022, 
-2.1460547966639472, -1.9824953479012122, -1.4283391884499703, 
-2.7804227318907553, -2.0630334044432308, -1.953800405062178, 
-1.2086127794168977, -1.7392349765923507, -1.5262383166702584, 
-1.8276723422085932, -2.1390027793286155, -2.0739931405236036, 
-2.0232008242417936, -2.2012083417014718, -1.8175354609359642, 
-1.6274929300583718, -1.789257974674624, -1.9481956017297699, 
-1.5532638806704959, -1.9081492030659299, -1.9599221419731825, 
-2.1195778296243346, -1.5930714685884213, -1.8496373779032709, 
-1.9438136226702991, -2.0544090974984011, -1.4748842632103238, 
-1.562138145123863, -1.8816920357851776, -2.3713526067489852, 
-2.1642217670154227, -2.0166791942662563, -1.7471707144850372, 
-2.1815005175695461, 0.79909902156621226, 0.74091001771931742, 
0.80419644107499921, 1.0734221845632297, 0.8919132650387005, 
0.89059066126624509, 0.89286883215706114, 1.2893949882229438, 
0.92337351584082517, 0.85503903651375901, 1.2012267234403771, 
1, 0.88915464902507124, 0.79111895994251402, 0.91915102064984067, 
0.77932775032536061, 0.87675078322231881, 0.84331578992725931, 
0.88396041715939588, 0.99406965254521829, 0.83624942273174452, 
0.9009282018888074, 1.0184699146139184, 1.0193009537021496, 0.80000445157635158, 
0.87497279566830488, 1.0492359999061076, 0.86503560915553424, 
0.81256311817546001, 0.90748171542282197, 0.91056758960409423, 
0.78588743075166523, 0.86206062421090779, 0.83094101621755745, 
0.92034729045825592, 0.79312358089379686, 0.91333253665934577, 
0.77256545877852589, 0.9225493318238932, 0.93054617538506512, 
1.1302737266799585, 0.87077860292230036, 0.82552321673747975, 
0.82655294035566285, 0.72374944572311584, 0.75610207072918745, 
0.80683086086227696, 1.0262846830525076, 0.78368354832054887, 
0.92050982576461515, 0.81520673390439569, 0.9426079900037595, 
0.7735631325761011, 1.0490141976070013, 0.94512486197707191, 
0.90094412845126903, 0.97603819560160321, 0.8898610627541973, 
0.72137640183696294, 0.73186890120181181, 1.0596701286284944, 
0.90635744210130642, 0.93273471427497812, 0.92604863922661684, 
1.0177834573457361, 0.79978379980618775, 0.88711832383618927, 
1.202120461237856, 0.71637762352313505, 1.0059344884262298, 0.92569061614993464, 
0.90050324675826254, 0.96332745344789139, 0.73288748007133631, 
1.0323349570771061, 0.97394076782317973, 1.0537375254865977, 
1.0429264197016699, 0.82506447570812513, 0.8849273206628997, 
0.88408344870409639, 0.6764912006139856, 0.85496755017701642, 
0.82467446489554108, 1.0079270221077377, 0.96002838987054018, 
0.92339078516470197, 0.96041246752649678, 0.99345298905208768, 
0.68104677003066605, 0.90441944089310711, 0.81067123199570523, 
0.95580390778703028, 0.74912640310263967, 0.89829198112358288, 
0.81441177786542773, 0.89256360630033416, 1.0303889992346096, 
0.71802880512060485, 0.81589674240585108, -0.027184986945730037, 
0.17807879992487899, 0.4642773479498008, 0.10724966697783821, 
0.27070599404778156, 0.43191298765379244, 0.14750259614871877, 
-0.65210460031893203, -0.084981769050771039, -0.24181492377102062, 
-0.072980511073650625, -0.33572415138501382, -0.31480135345838828, 
-0.34598965568401269, -0.065828861618523263, 0.018838639975360609, 
0.021908234726970253, 0.12654745630436717, -0.33441115710518271, 
-0.028132096232540128, -0.14758839539810448, 0.27736910556774669, 
-0.25478330199809979, 0.3965978757938442, -0.0708236321028025, 
-0.056080921663310447, -0.098707401810695636, 0.79357637005748805, 
-0.003990147166710532, 0.40220702979427714, 0.48192459637516988, 
0.14105374452691499, -0.098363329845638406, -0.13474612097426195, 
0.099858456785106811, 0.10400694145693855, 0.24921218894554717, 
0.11834553566627866, 0.17354547581349028, -0.045535422923814177, 
-0.49304609895480905, -0.059162951561258371, 0.029706425383450384, 
0.2249780617687748, 0.48714781305710131, 0.059901795124698176, 
-0.28133570258629742, 0.13843547232469114, 0.65697147354818819, 
0.39259975539565234, 0.33163196951539892, -0.26800803272814533, 
0.036242244873654419, -0.17776914706417543, -0.075480516303904996, 
-0.11992532302590128, -0.15370591650412643, 0.12409091857367491, 
0.19245281032613334, -0.40295939124456215, -0.048811781412183071, 
0.020090496578506221, 0.47217823483585614, -0.050572612490690116, 
-0.49325006965538393, 0.24279527649521043, -0.36677503393114008, 
0.036374819923431723, -0.32218267772429332, -0.21172736727714084, 
0.12381793214171076, 0.040721847604545024, 0.092533064138132406, 
0.45309182740992998, 0.027978999374535674, -0.1786721085022388, 
-0.14915690884812, 0.34510790858266527, -0.5678435379542297, 
-0.075948196376346352, -0.01677699263281596, 0.2825626813932392, 
-0.14861506217206388, -0.11524416147029441, -0.1664384997128954, 
-0.56691238134231925, 0.26527048836415129, -0.0032907389168859609, 
0.016403492248170483, 0.57261382431551844, -0.029331769306630524, 
-0.42034106904466156, -0.11356168882533511, 0.46817523508618242, 
0.14076646360197961, 0.045071045489024818, 0.40305037300934943, 
-0.32920133617029146, 0.072725765618351021, 0.41157339144028282, 
-1.2469608682304663, -0.99778943679227283, -0.94278098239866159, 
-1.2271262738350674, -1.3411731186905682, -1.0981291211795177, 
-0.99129855544652179, -1.0886605666926765, -0.9445464002861218, 
-1.227100995395459, -1.2353910461276332, -0.85711867809950348, 
-1.2650202584649295, -0.55675606118510201, -1.205992962839274, 
-1.4371206075064498, -1.0783555459590166, -0.84981178978486793, 
-0.57123729790290501, -0.93445758813575119, -1.1033025211092957, 
-0.7362275039405396, -0.75676658519921858, -0.98687771580708383, 
-0.68607641293570831, -0.84776359315338667, -1.2715470494102603, 
-0.62507322867937742, -1.006527450076431, -1.1216925551093926, 
-0.8550731338372225, -1.0236623700509537, -1.279182313923124, 
-0.96799864417545678, -0.79904053729544966, -0.92413560244021398, 
-1.5832201246593463, -1.125262784096561, -1.4250015675190835, 
-0.81501868995694704, -0.63199960645976205, -1.1344015282630289, 
-0.53129853790048642, -1.5921010255369319, -1.3071320214067816, 
-0.76432542893541755, -0.65759657455813969, -1.0951110251611038, 
-0.88566269020871913, -0.7604834553898977, -0.91362205789164164, 
-0.71463968642336595, -1.1375213933998096, -1.0779484604753413, 
-0.97505580688591398, -1.0489211594347825, -0.87021903371289411, 
-1.0561554155162354, -1.4611852011535393, -1.3883843325696557, 
-1.1087726829770415, -0.78271569384269613, -0.10256436007293984, 
-0.85848112485977934, -1.3054660412946186, -0.61327888895728533, 
-0.75327619375143906, -0.056654244217601339, -1.188018615788264, 
-0.77422664254913232, -1.2066750390548606, -1.1156498877129908, 
-0.81678542942730914, -0.93844937119806215, -0.46583852105066226, 
-1.2976295463384102, -0.8653077321831727, -0.88361838537889958, 
-0.84591250270432128, -1.0598417748215225, -1.2969502593222098, 
-0.82545493258393665, -0.33264342500345562, -0.87421302044528526, 
-0.93068297391440025, -0.84802729475108463, -0.67542013000383616, 
-0.56010903444640814, -0.74221568242392044, -0.53736988814740316, 
-0.20005186495749544, -0.66175776432287603, -0.76905176892056504, 
-1.1318869956200439, -0.79085225268526704, -0.84551627226067316, 
-0.68917722541270499, -0.82272841054392765, -1.0736568147843124, 
-0.82334209081847032, -1.9431627529722446, -2.1675906771687936, 
-1.829335054185121, -1.689735676076306, -1.616656346844596, -1.5855101398426854, 
-2.1913175767974984, -2.2090113194604784, -2.2280974787465389, 
-1.2082053528047179, -1.5187432771083105, -1.9276462068628917, 
-1.8567208452441792, -2.1730969226766743, -2.1754442802522633, 
-1.9832704339594092, -2.0572432971712971, -1.8299102111674328, 
-2.3691430515673879, -1.6969136304918315, -1.7568100595181424, 
-2.3063173266537498, -1.7567561940160326, -2.1979688436728204, 
-1.8968245067157952, -1.7180980342123753, -2.2227687297518575, 
-2.0692799833990105, -1.5505323228227239, -1.7496883310762221, 
-1.7391717246316016, -1.8374743661920965, -1.9381064541980999, 
-1.9229110825598321, -1.6831936092706103, -1.3275473203064305, 
-2.0093157815219413, -1.8730036860268251, -1.9284368950974069, 
-2.0758574887479577, -1.6677322136145449, -1.6180988635188687, 
-2.223975637487217, -1.6544322379034535, -1.8805205271687997, 
-1.8364207343868555, -1.7988255969018254, -2.0271021667974738, 
-1.6384929725807029, -1.5449141273844758, -2.1130187757160219, 
-2.1888270760642934, -2.4031125196346794, -1.8215419053359205, 
-2.47164299132087, -1.7920474547272622, -2.1626823708801783, 
-1.9660026316664738, -1.9341464925544722, -2.0631696731699223, 
-2.1221843268789264, -1.9608345483916794, -1.4431993994168566, 
-1.7738487599152029, -2.1170911476583854, -1.871180376121123, 
-2.0550535401963921, -1.9488670031286364, -1.4525262513386401, 
-2.6919685755753959, -2.0283326944021582, -1.8290384076944755, 
-1.1792502234415394, -1.6789033559892814, -1.4039810382248443, 
-1.8521038846356459, -2.1777339971362166, -2.0012288916871936, 
-1.9803540372339132, -2.1418654132833645, -1.7322720418458561, 
-1.6123642555742932, -1.6790746963047758, -1.9696965276752318, 
-1.4874782784176315, -1.8479418158404439, -1.8760724519885994, 
-2.0768489169166044, -1.5264258628397611, -1.7831305494520535, 
-1.9581648484372061, -1.9204800997601579, -1.3950377523024884, 
-1.5608202618905549, -1.856253207853072, -2.3356456732001205, 
-2.080936851448302, -1.9658480302553158, -1.783566207268372, 
-2.0747499269039329, 0.4771198957731651, 0.46053387059771322, 
0.46287893056763535, 0.60152476954365997, 0.52446595875556101, 
0.49071820558899126, 0.54995314175377596, 0.69279652989268159, 
0.57885239158146251, 0.48393352049051741, 0.71139081620540046, 
0.52264840825649228, 0.50506831183818146, 0.47897251993007955, 
0.54542330311396203, 0.46122672527028535, 0.50890736495814604, 
0.4880468488747543, 0.51947607937624829, 0.60657996009742976, 
0.51845763780096898, 0.54034005307841604, 0.52508149479906974, 
0.60369112641947809, 0.50003205087903957, 0.53040921834688837, 
0.62129769102220123, 0.531178432836382, 0.49370372993485961, 
0.54423039445121613, 0.5483746894758682, 0.41924319617433436, 
0.53418381285839034, 0.49986469031097575, 0.55580217727566383, 
0.47553664930069944, 0.55142953651467208, 0.45453060020472691, 
0.55059175880385358, 0.53145981747056337, 0.67023231758006341, 
0.51205593813678008, 0.50150903266708335, 0.48963958110544303, 
0.44937110572058803, 0.4412941624446145, 0.46892442854509725, 
0.59109220487561476, 0.45125703903667846, 0.52877148522132233, 
0.52256667305923832, 0.56033323304292226, 0.45716124910149436, 
0.61642784058810374, 0.5745042581208486, 0.55919891469219629, 
0.58879455437695138, 0.5019559252881467, 0.4581971546821067, 
0.41979240621394809, 0.6140088118877165, 0.54822171147007381, 
0.56689769487522179, 0.53665076573983062, 0.60714331875321237, 
0.46140079419282209, 0.51035914326875487, 0.69239054321827198, 
0.44454570464830057, 0.61664824438951626, 0.55467608458161588, 
0.52006429337396953, 0.54060756261968312, 0.44459661495812353, 
0.58041023446746587, 0.58303018883532931, 0.62220575426520885, 
0.6026113835048299, 0.49893609121700155, 0.51838951914107134, 
0.49548384797094852, 0.40504671144532461, 0.4881667052367511, 
0.52028206527936649, 0.57931824425211165, 0.50015688014617954, 
0.5074980939183521, 0.55217060273812735, 0.59971904817109489, 
0.43034709084919015, 0.54552807633580003, 0.45877615035815167, 
0.53419416969844558, 0.45134075807391455, 0.53630584212154075, 
0.49273788758786452, 0.53029118783032914, 0.58409212372185759, 
0.44826040289275182, 0.45974559680279525), algo = structure(c(2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L), .Label = c("mtram", 
"tramME"), class = "factor"), which = structure(c(1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L), .Label = c("beta1", 
"beta2", "beta3", "gamma"), class = "factor")), class = "data.frame", row.names = c(NA, 
-800L))

bwplot(coef ~ algo | which, data = coefs, outer = TRUE,
       between = list(x = .5), layout = c(4, 1),
       panel = function(x, y, groups, ...) {
         panel.bwplot(x, y, ...)
         panel.abline(h = true_coefs[panel.number()], col = 2, lty = 2, lwd = 2)
       },
       par.settings = "mytheme",
       fill = "grey", pch = "|", lty = 1,
       scales = list(x = list(relation = "free", rot = 60),
                     y = list(rot = 90)),
       ylab = " ",
       strip = strip.custom(bg = "transparent",
                            factor.levels = c(expression(beta[1]), 
                                              expression(beta[2]), 
                                              expression(beta[3]), 
                                              expression(gamma)))
)


## ----mtram-sessionInfo, echo = FALSE, results = "hide"------------------------
sessionInfo()


## ----mtram-funs, echo = FALSE, results = "hide"-------------------------------
if (file.exists("packages.bib")) file.remove("packages.bib")
pkgversion <- function(pkg) {
    pkgbib(pkg)
    packageDescription(pkg)$Version
}
pkgbib <- function(pkg) {
    x <- citation(package = pkg, auto = TRUE)[[1]]
    b <- toBibtex(x)
    b <- gsub("Buehlmann", "B{\\\\\"u}hlmann", b)
    b[1] <- paste("@Manual{pkg:", pkg, ",", sep = "")
    if (is.na(b["url"])) {
        b[length(b)] <- paste("   URL = {http://CRAN.R-project.org/package=",
                              pkg, "}", sep = "")
        b <- c(b, "}")
    }
    cat(b, sep = "\n", file = "packages.bib", append = TRUE)
}

pkg <- function(pkg) {
    vrs <- try(pkgversion(pkg))
    if (inherits(vrs, "try-error")) return(NA)
    paste("\\\\pkg{", pkg, "} \\\\citep[version~",
          vrs, ",][]{pkg:", pkg, "}", sep = "")
}

pkg("mlt")
pkg("tram")
pkg("SparseGrid")
#pkg("lme4")
cat(c("@Manual{vign:mlt.docreg,",
             "    title = {Most Likely Transformations: The mlt Package},",
             "    author = {Torsten Hothorn},",
             paste("    year = ", substr(packageDescription("mlt.docreg")$Date, 1, 4), ",", sep = ""),
             paste("    note = {R package vignette version ", packageDescription("mlt.docreg")$Version, "},", sep = ""),
             "    url = {https://CRAN.R-project.org/package=mlt.docreg},",
             "}"), file = "packages.bib", append = TRUE, sep = "\n")

