% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pclm_2D.R
\name{pclm2D}
\alias{pclm2D}
\title{Two-Dimensional Penalized Composite Link Model (PCLM-2D)}
\usage{
pclm2D(x, y, nlast, offset = NULL, out.step = 1, ci.level = 95,
  verbose = TRUE, control = list())
}
\arguments{
\item{x}{Vector containing the starting values of the input intervals/bins.
For example: if we have 3 bins \code{[0,5), [5,10) and [10, 15)},
\code{x} will be defined by the vector: \code{c(0, 5, 10)}.}

\item{y}{\code{data.frame} with counts to be ungrouped. The number of rows 
should be equal with the length of \code{x}.}

\item{nlast}{Length of the last interval. In the example above \code{nlast} 
would be 5.}

\item{offset}{Optional offset term to calculate smooth mortality rates. 
A vector of the same length as x and y. See 
\insertCite{rizzi2015;textual}{ungroup} for further details.}

\item{out.step}{Length of estimated intervals in output. 
Values between 0.1 and 1 are accepted. Default: 1.}

\item{ci.level}{Level of significance for computing confidence intervals. 
Default: \code{95}.}

\item{verbose}{Logical value. Indicates whether a progress bar should be 
shown or not. Default: \code{TRUE}.}

\item{control}{List with additional parameters: \itemize{
\item{\code{lambda}} -- Smoothing parameter to be used in pclm estimation.
If \code{lambda = NA} an algorithm will find the optimal values.
\item{kr} -- Knot ratio. Number of internal intervals used for defining 
1 knot in B-spline basis construction. See \code{\link{MortSmooth_bbase}}.
\item{\code{deg}} -- Degree of the splines needed to create equally-spaced 
B-splines basis over an abscissa of data.
\item{\code{int.lambda}} -- If \code{lambda} is optimized an interval to be 
searched needs to be specified. Format: vector containing the end-points.
\item{\code{diff}} -- An integer indicating the order of differences of the 
components of PCLM coefficients.
\item{\code{opt.method}} -- Selection criterion of the model.
Possible values are \code{"AIC"} and \code{"BIC"}.
\item{\code{max.iter}} -- Maximal number of iterations used in fitting 
procedure.
\item{\code{tol}} -- Relative tolerance in PCLM fitting procedure.}}
}
\value{
The output is a list with the following components:
 \item{input}{ A list with arguments provided in input. Saved for 
 convenience.}
 \item{fitted}{ The fitted values of the PCLM model.}
 \item{ci}{ Confidence intervals around fitted values.}
 \item{goodness.of.fit}{ A list containing goodness of fit measures: 
standard errors, AIC and BIC.} 
 \item{smoothPar}{ Estimated smoothing parameters: \code{lambda, kr} 
and \code{deg}.}
 \item{bins.definition}{ Additional values to identify the bins limits and 
location in input and output objects.}
 \item{deep}{ A list of objects created in the fitting process. Useful 
in diagnosis of possible issues.}
 \item{call}{ An unevaluated function call, that is, an unevaluated 
expression which consists of the named function applied to the given 
arguments.}
}
\description{
Fit two-dimensional penalized composite link model (PCLM-2D), 
e.g. simultaneous ungrouping of age-at-death distributions grouped in age 
classes for adjacent years. The PCLM can be extended to a two-dimensional 
regression problem. This is particularly suitable for mortality analysis 
when mortality surfaces are to be estimated to capture both age-specific 
trajectories of coarsely grouped distributions and time trends 
\insertCite{rizzi2019}{ungroup}.
}
\examples{
# Input data
Dx <- ungroup.data$Dx
Ex <- ungroup.data$Ex

# Aggregate data to ungroup it in the examples below
x      <- c(0, 1, seq(5, 85, by = 5))
nlast  <- 26
n      <- c(diff(x), nlast)
group  <- rep(x, n)
y      <- aggregate(Dx, by = list(group), FUN = "sum")[, -1]
offset <- aggregate(Ex, by = list(group), FUN = "sum")[, -1]

# Example 1 ---------------------- 
# Fit model and ungroup data using PCLM-2D
P1 <- pclm2D(x, y, nlast)
summary(P1)
# plot(P1)

\dontrun{
# NOTE: pclm2D does not search for optimal smoothing parameters by default
# (like pclm) because it is more time consuming. If optimization is required
# set lambda = c(NA, NA):

P1 <- pclm2D(x, y, nlast, control = list(lambda = c(NA, NA)))

# Example 2 ---------------------- 
# Ungroup and build a mortality surface
P2 <- pclm2D(x, y, nlast, offset)
summary(P2)

plot(P2)                      # plot
library(rgl)
snapshot3d("plotP2.jpeg")     # save the plot in jpeg format
aspect3d(x = 1, y = 2, z = 1) # modify the aspect ratio
}
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{control.pclm2D}}
\code{\link{plot.pclm2D}}
}
