### This function generates a pdf file with a report
### describing the variables.


#' scaleDiagnosisToPDF
#' 
#' \code{\link{scaleDiagnosis}} provides a number of diagnostics for a scale
#' (an aggregative measure consisting of several items), and
#' scaleDiagnosisToPDF takes the resulting object and generates a PDF file,
#' which is then saved to disk.
#' 
#' 
#' This functon generates a PDF file from a \code{\link{scaleDiagnosis}}
#' object. \code{\link{scaleDiagnosis}} generates an object with several useful
#' statistics and a plot to assess how the elements (usually items) in a scale
#' relate to each other, such as Cronbach's Alpha, omega, the Greatest Lower
#' Bound, a factor analysis, and a correlation matrix.
#' 
#' @param scaleDiagnosisObject An object generated by
#' \code{\link{scaleDiagnosis}}.
#' @param docTitle The title of the PDF file (printed on the first page).
#' @param docAuthor The author to show in the PDF file (printed on the first
#' page).
#' @param pdfLatexPath The path to PdfLaTex. This file is part of a LaTeX
#' installation that creates a pdf out of a .tex file. See
#' \code{\link{rnwString}} for more information.
#' @param rnwPath The path where the temporary files will be stored.
#' @param filename Filename of the PDF (".pdf" is appended).
#' @param digits Number of digits to show.
#' @param rMatrixColsLandscape This number determines when the page(s) in the
#' PDF is/are rotated; pages with matrices that have this number of columns or
#' more are rotated.
#' @param pboxWidthMultiplier Passed on to (unexported method)
#' \code{print.rMatrix}.
#' @param scatterPlotBaseSize Basic size of scatterplots in centimeters. If
#' this number, multiplied by the number of items (i.e. columns/rows in
#' scattermatrix) is larger than maxScatterPlotSize, it is ignored.
#' @param maxScatterPlotSize Maximum size of scatterplots; automatically
#' calculated if NULL.
#' @param pageMargins Margins of landscape pages in millimeters.
#' @param pval Whether to print p-values using the p-value formatting.  Passed
#' on to (unexported method) \code{print.rMatrix}.
#' @return
#' 
#' Nothing is returned; the file is printed to disk.
#' @author Gjalt-Jorn Peters
#' 
#' Maintainer: Gjalt-Jorn Peters <gjalt-jorn@@userfriendlyscience.com>
#' @keywords utilities
#' @examples
#' 
#' 
#' \dontrun{
#' 
#' ### Generate a datafile to use
#' exampleData <- data.frame(item1=rnorm(100));
#' exampleData$item2 <- exampleData$item1+rnorm(100);
#' exampleData$item3 <- exampleData$item1+rnorm(100);
#' exampleData$item4 <- exampleData$item2+rnorm(100);
#' exampleData$item5 <- exampleData$item2+rnorm(100);
#' 
#' ### Use all items and create object
#' scaleDiagnosisObject <- scaleDiagnosis(dat=exampleData);
#' 
#' ### Generate a PDF
#' scaleDiagnosisToPDF(scaleDiagnosisObject);
#' 
#' }
#' 
#' @export scaleDiagnosisToPDF
scaleDiagnosisToPDF <- function(scaleDiagnosisObject,
                            docTitle = "Scale diagnosis", docAuthor = "Author",
                            pdfLatexPath, rnwPath=getwd(),
                            filename = "scaleDiagnosis",
                            digits=2,
                            rMatrixColsLandscape = 6,
                            pboxWidthMultiplier = 1,
                            scatterPlotBaseSize = 4,
                            maxScatterPlotSize = NULL,
                            pageMargins=15,
                            pval=TRUE) {

  if (!('scaleDiagnosis' %in% class(scaleDiagnosisObject))) {
    stop("Argument 'scaleDiagnosisObject' must have class 'scaleDiagnosis'!");
  }
  
  if (!hasLaTeX(pdfLatexPath)) {
    stop('In path "', pdfLatexPath, '", the file pdflatex.exe (Windows) or ',
         'pdflatex (MacOS or Ubuntu (Linux)) does not exist! Please ',
         'locate the file and provide its path (without the last ',
         'slash). See ?rnwString for more information.');
  }
  
  res <- list();
  res$scaleDiagnosis <- scaleDiagnosisObject;
  
  if (is.null(maxScatterPlotSize)) {
    maxScatterPlotSize <- 21 - ((pageMargins*2)/10);
  }
  
  res$rnw <- rnwString.initiate(docTitle, docAuthor,
                                docClassArgs='a4paper,portrait,10pt',
                                newPage=FALSE, pageMargins=pageMargins);
  
  ### Generate correlation table
  res$rMatrix <- rMatrix(res$scaleDiagnosis$dat,
                         x=names(res$scaleDiagnosis$dat));
      
  ### Include correlation table;
  ### whether to print on a portrait page or
  ### on a landscape page depends on number of
  ### columns and rMatrixColsLandscape
  if (length(res$rMatrix$variables.cols) < rMatrixColsLandscape) {
    res$rnw <-
      paste0(res$rnw,
             '\n\\begin{minipage}{\\textwidth}\n\\maxsizebox{\\textwidth}{\\textheight}{\n');
  }
  else {
    res$rnw <-
      paste0(res$rnw,
             '\\begin{landscape}\n\\maxsizebox{', 297 - 2*pageMargins, 'mm}{', 210 - 2*pageMargins, 'mm}{\n');
  }
  
  res$rnw <-
    paste0(res$rnw, '<< echo=FALSE, results="asis" >>=\n',
           'print(res$rMatrix, digits=digits, output="LaTeX", pboxWidthMultiplier=pboxWidthMultiplier, pval=pval);\n',
           '@\n');

  if (length(res$rMatrix$variables.cols) < rMatrixColsLandscape) {
    res$rnw <- paste0(res$rnw, '}\n\\end{minipage}\n');
  }
  else {
    res$rnw <- paste0(res$rnw, '}\n\\end{landscape}\n');
  }
      
  ### The size of each panel in the scattermatrix depends
  ### on the number of items - therefore, we need to adjust
  ### the plot sizes to the number of items. This is mainly
  ### necessary because in ggpairs.print (which you can
  ### view with "getAnywhere('print.ggpairs');"), the
  ### fontsize is fixed at 15.
  ### knitr wants unit for outputsize, no unit for figure draw
  ### size (but this must be specified in inches).
  if (res$scaleDiagnosis$scaleReliability$input$n.items * scatterPlotBaseSize > maxScatterPlotSize) {
    figSizeInOutput <- maxScatterPlotSize;
  }
  else {
    figSizeInOutput <- res$scaleDiagnosis$scaleReliability$input$n.items * scatterPlotBaseSize;
  }
  ### For two items on a page (i.e. plots of roughly 9x9 cm),
  ### the labels of the plots have roughly the right size,
  ### so we multiply 9 cm with the number of items.
  figSizeToDraw <- (9 / 2.54) * res$scaleDiagnosis$scaleReliability$input$n.items;
  ### If figSizeToDraw is smaller than output size, set to output size
  if (figSizeToDraw < (figSizeInOutput / 2.54)) {
    figSizeToDraw <- figSizeInOutput / 2.54;
  }
  ### Add unit to size in output
  figSizeInOutput <- paste0(figSizeInOutput, "cm");
  
  res$rnw <-
    paste0(res$rnw,
           '\\begin{minipage}{180mm}\n',
           '<< echo=FALSE, warning=FALSE, dev="pdf", fig.width=', figSizeToDraw, ', fig.height=', figSizeToDraw, ', out.width="', figSizeInOutput, '", out.height="', figSizeInOutput, '" >>=\n',
           'print(res$scaleDiagnosis$ggpairs.combined);\n',
           '@\n',
           '<< echo=FALSE, results="asis" >>=\n',
           '@\n',
           '\\end{minipage}%\n');
  
  ### Finalize rnwString and generate the PDF
  res$rnw <- rnwString.terminate(res$rnw);
  rnwString.generate(res$rnw, rnwPath, fileName=filename, pdfLatexPath);
  
  ### Store result for later inspection
  class(res) <- c('scaleDiagnosisPDF');
  return(res);
  
}
