% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/viewsheds.R
\name{viewsheds}
\alias{viewsheds}
\title{Computes cumulated viewsheds}
\usage{
viewsheds(data, positions, angular_res, vox_res, cut_off, pb)
}
\arguments{
\item{data}{LAS class object containing the xyx coordinates of a 3D point
cloud.}

\item{positions}{data.frame or data.table with 3 columns containing the xyz
coordinates of the animal locations from which the viewsheds will be computed.}

\item{angular_res}{numeric. The angular resolution of a single sightline.
Default = 1.}

\item{vox_res}{numeric. The resolution of the output voxel cloud. Default =
0.2.}

\item{cut_off}{(optional) numeric. Defines a cut-off distance for each
individual viewshed. Speeds up the process
when \code{viewsheds} is applied to big datasets.}

\item{pb}{logical. If \code{FALSE}, desables the progress bar.}
}
\value{
A LAS class object containing the coordinates of the voxel cloud
(X, Y, Z), and the number of times each voxel was visible from any position
(N_visible).
}
\description{
Computes cumulated viewsheds within a 3D point cloud and return
a voxel cloud accounting for the number of times each voxel was visible.
}
\details{
Sightline directions in each viewshed are computed from the method
described by Malkin (2016). This ensures that every sightline explores
a similar portion of the 3d space.
}
\note{
In most cases, a ground reconstruction should be performed before
viewsheds computation. This can be done with
the \code{\link[lidR]{classify_ground}} and \code{\link[lidR]{grid_terrain}}
functions from the \code{\link[lidR]{lidR-package}}.
}
\examples{
\donttest{
#- import the tree_line_plot dataset
file = system.file("extdata", "tree_line_plot.laz", package="viewshed3d")
tls = lidR::readLAS(file,select="xyz")

#- remove noise to avoid visibility estimates error
tls_clean = viewshed3d::denoise_scene(tls,method="sd",
                                       filter=6)

#- RECONSTRUCT THE GROUND
#- classify ground points
class=lidR::classify_ground(tls_clean, lidR::csf(rigidness = 1L,
                                           class_threshold = 0.1,
                                           sloop_smooth = TRUE), FALSE)

#- reconstruct the ground. No need for a very fine ground reconstruction.
ground = lidR::grid_terrain(class, 0.05, lidR::knnidw())

#- build the final scene
reconstructed = na.omit(raster::as.data.frame(ground, xy = TRUE))
names(reconstructed)=c("X","Y","Z")
recons=rbind(lidR::LAS(na.omit(reconstructed)),tls_clean)

#- CREATE THE POSITIONS WITH RANDOM POINTS
N_positions = 10 #- how many points ?
height = 2 #- points height relative to the ground
positions=data.table::data.table(reconstructed[runif(N_positions,
                                               1,nrow(reconstructed)),])
positions[,Z:=Z+height]

#- compute the cumulated viewsheds from the positions
cumulated=viewshed3d::viewsheds(data=recons,
                                positions = positions ,
                                angular_res = 1,
                                vox_res = 0.2)

#- plot the result
x=lidR::plot(cumulated,color="N_visible",size=3,
             colorPalette=viridis::cividis(nrow(positions)+1),trim=6)

#- add the positions
lidR::add_treetops3d(x,sp::SpatialPointsDataFrame(positions,positions),
                     radius=0.5,col="red",add=TRUE)
}
}
\references{
Malkin, Z. (2016). A new method to subdivide a spherical surface
into equal-area cells. arXiv:1612.03467.
}
