% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vmgeom_vst.R
\name{vmgeomVst}
\alias{vmgeomVst}
\alias{vmgeomVstFromMagn}
\alias{vmgeomVstToR}
\title{Variance-Stabilizing Transformation for Geometric Visual Meteor Magnitudes}
\usage{
vmgeomVstFromMagn(m, lm)

vmgeomVstToR(tm, log = FALSE, deriv.degree = 0L)
}
\arguments{
\item{m}{integer; meteor magnitude.}

\item{lm}{numeric; limiting magnitude.}

\item{tm}{numeric; transformed magnitude.}

\item{log}{logical; if \code{TRUE}, the logarithm of the population index \code{r} is returned.}

\item{deriv.degree}{integer; the order of the derivative at \code{tm} to return
instead of \code{r} or \code{log(r)}. Must be \code{0}, \code{1}, or \code{2}.}
}
\value{
\itemize{
\item \code{vmgeomVstFromMagn}: numeric value, the transformed meteor magnitude.
\item \code{vmgeomVstToR}: numeric value of the population index \code{r}, derived from
the mean of \code{tm}.
}

The argument \code{deriv.degree} can be used to apply the delta method.
If \code{log = TRUE}, the logarithm of \code{r} is returned.
}
\description{
Applies a variance-stabilizing transformation to visual meteor magnitudes
under the geometric model.
}
\details{
Many linear models require the variance of visual meteor magnitudes to be
homoscedastic. The function \code{vmgeomVstFromMagn} applies a transformation
that produces homoscedastic distributions of visual meteor magnitudes if the
underlying distribution follows a geometric model.

The geometric model of visual meteor magnitudes
depends on the \link[=vmgeom]{population index} \code{r} and the limiting magnitude \code{lm},
resulting in a two-parameter distribution. Without detection probabilities,
the magnitude distribution is purely geometric, and for integer limiting
magnitudes the variance depends only on the population index \code{r}. Since the
limiting magnitude \code{lm} is a fixed parameter and never estimated
statistically, the magnitudes can be transformed such that, for example,
the mean of the transformed magnitudes directly provides an estimate of \code{r}
using the function \code{vmgeomVstToR}.

A key advantage of this transformation is that the limiting magnitude \code{lm}
is already incorporated into subsequent analyses. In this sense, the
transformation acts as a normalization of meteor magnitudes and yields a
variance close to \code{1.0}.

This transformation is valid for \eqn{1.4 \le r \le 3.5}.
The numerical form of the transformation is version-specific and may change
substantially in future releases. Do not rely on equality of transformed
values across package versions.
}
\examples{
N <- 100
r <- 2.0
limmag <- 6.3

# Simulate magnitudes
m <- rvmgeom(N, limmag, r)

# Variance-stabilizing transformation
tm <- vmgeomVstFromMagn(m, limmag)
tm.mean <- mean(tm)
tm.var  <- var(tm)

# Estimator for r from the transformed mean
r.hat  <- vmgeomVstToR(tm.mean)

# Derivative dr/d(tm) at tm.mean (needed for the delta method)
dr_dtm <- vmgeomVstToR(tm.mean, deriv.degree = 1L)

# Variance of the sample mean of tm
var_tm.mean <- tm.var / N

# Delta method: variance and standard error of r.hat
var_r.hat <- (dr_dtm^2) * var_tm.mean
se_r.hat  <- sqrt(var_r.hat)

# Results
print(r.hat)
print(se_r.hat)
}
\seealso{
\link{vmgeom}
}
