% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wsd.R
\name{wsd}
\alias{wsd}
\title{Windowed Scalogram Difference}
\usage{
wsd(signal1,
           signal2,
           dt = 1,
           scaleparam = NULL,
           windowrad = NULL,
           rdist = NULL,
           delta_t = NULL,
           normalize = FALSE,
           wname = c("MORLET", "DOG", "PAUL", "HAAR", "HAAR2"),
           wparam = NULL,
           waverad = NULL,
           border_effects = c("BE", "INNER", "PER", "SYM"),
           mc_nrand = 0,
           commutative = TRUE,
           wscnoise = 0.02,
           compensation = 0,
           energy_density = TRUE,
           parallel = FALSE,
           makefigure = TRUE,
           time_values = NULL,
           figureperiod = TRUE)
}
\arguments{
\item{signal1}{A vector containing the first signal.}

\item{signal2}{A vector containing the second signal (its length should be equal to
that of \code{signal1}).}

\item{dt}{Numeric. The time step of the signals.}

\item{scaleparam}{A vector of three elements with the minimum scale, the maximum scale
and the number of suboctaves per octave for constructing power 2 scales (following
Torrence and Compo 1998). If NULL, they are automatically computed.}

\item{windowrad}{Integer. Time radius for the windows, measured in \code{dt}. By
default, it is set to \eqn{celing(length(signal1) / 20)}.}

\item{rdist}{Integer. Log-scale radius for the windows measured in suboctaves. By
default, it is set to \eqn{ceiling(length(scales) / 20)}.}

\item{delta_t}{Integer. Increment of time for the construction of windows central
times, measured in \code{dt}. By default, it is set to
\eqn{ceiling(length(signal1) / 256)}.}

\item{normalize}{Logical. Set to TRUE if the signals use different units.}

\item{wname}{A string, equal to "MORLET", "DOG", "PAUL", "HAAR" or "HAAR2". The
difference between "HAAR" and "HAAR2" is that "HAAR2" is more accurate but slower.}

\item{wparam}{The corresponding nondimensional parameter for the wavelet function
(Morlet, DoG or Paul).}

\item{waverad}{Numeric. The radius of the wavelet used in the computations for the cone
of influence. If it is not specified, it is computed by \code{wavelet_radius} for DoG
and Paul wavelets. For Haar and Morlet it is assumed to be 1 and 3 respectively.}

\item{border_effects}{String, equal to "BE", "INNER", "PER" or "SYM",
which indicates how to manage the border effects which arise usually when a convolution
is performed on finite-lenght signals.
\itemize{
\item "BE": With border effects, padding time series with zeroes.
\item "INNER": Normalized inner scalogram with security margin adapted for each
different scale.
\item "PER": With border effects, using boundary wavelets (periodization of the
original time series).
\item "SYM": With border effects, using a symmetric catenation of the original time
series.
}}

\item{mc_nrand}{Integer. Number of Montecarlo simulations to be performed in order to
determine the 95\% and 5\% significance contours.}

\item{commutative}{Logical. If TRUE (default) the commutative windowed scalogram
difference. Otherwise a non-commutative (but simpler) version is computed (see Bolós et
al. 2017).}

\item{wscnoise}{Numeric in \eqn{[0,1]}. If a (windowed) scalogram takes values close to
zero, some problems may appear because we are considering relative differences.
Specifically, we can get high relative differences that in fact are not relevant, or
even divisions by zero.

  If we consider absolute differences this would not happen but, on the other hand,
  using absolute differences is not appropriate for scalogram values not close to zero.

  So, the parameter \code{wscnoise} stablishes a threshold for the scalogram values
  above which a relative difference is computed, and below which a difference
  proportional to the absolute difference is computed (the proportionality factor is
  determined by requiring continuity).

  Finally, \code{wscnoise} can be interpreted as the relative amplitude of the noise in
  the scalograms and is chosen in order to make a relative (\eqn{= 0}), absolute
  (\eqn{= 1}) or mix (in \eqn{(0,1)}) difference between scalograms. Default value is
  set to \eqn{0.02}.}

\item{compensation}{Numeric in \eqn{[0,1]}. It is an alternative to \code{wscnoise} for
preventing numerical errors or non-relevant high relative differences when scalogram
values are close to zero (see Bolós et al. 2017).}

\item{energy_density}{Logical. If TRUE (default), divide the scalograms by the square
root of the scales for convert them into energy density. Note that it does not affect
the results if \code{wscnoise} \eqn{= 0}.}

\item{parallel}{Logical. If TRUE (default) uses function \code{parApply} from package
\code{parallel} for the Montecarlo simulations. When FALSE it uses the normal
\code{apply} function.}

\item{makefigure}{Logical. If TRUE (default), a figure with the WSD is plotted.}

\item{time_values}{A numerical vector of length \code{length(signal)} containing custom
time values for the figure. If NULL (default), it will be computed starting at 0.}

\item{figureperiod}{Logical. If TRUE (default), periods are used in the figure instead
of scales.}
}
\value{
A list with the following fields:
\itemize{
\item \code{wsd}: A matrix of size \code{length(tcentral)} x \code{length(scales)}
containing the values of the windowed scalogram differences at each scale and at each
time window.
\item \code{tcentral}: The vector of central times used in the computations of the
windowed scalograms.
\item \code{scales}: The vector of scales.
\item \code{windowrad}: Radius for the time windows of the windowed scalograms,
measured in \code{dt}.
\item \code{rdist}: The log-scale radius for the windows measured in suboctaves.
\item \code{signif95}: A logical matrix of size \code{length(tcentral)} x
\code{length(scales)}. If TRUE, the corresponding point of the \code{wsd} matrix is in
the 95\% significance.
\item \code{signif05}: A logical matrix of size \code{length(tcentral)} x
\code{length(scales)}. If TRUE, the corresponding point of the \code{wsd} matrix is in
the 5\% significance.
\item \code{fourier_factor}: A factor for converting scales to periods.
\item \code{coi_maxscale}: A vector of length \code{length(tcentral)} containing the
values of the maximum scale from which there are border effects for the respective
central time.
}
}
\description{
This function computes the Windowed Scalogram Difference of two signals.
The definition and details can be found in (Bolós et al. 2017).
}
\section{References}{

C. Torrence, G. P. Compo. A practical guide to wavelet analysis. B. Am. Meteorol. Soc.
79 (1998), 61–78.

V. J. Bolós, R. Benítez, R. Ferrer, R. Jammazi. The windowed scalogram difference: a
novel wavelet tool for comparing time series. Appl. Math. Comput., 312 (2017), 49-65.
}

\examples{

nt <- 1500
time <- 1:nt
sd_noise <-  0.2 #\% In BolÃ³s et al. 2017 Figure 1, sd_noise = 1.
signal1 <- rnorm(n = nt, mean = 0, sd = sd_noise) + sin(time / 10)
signal2 <- rnorm(n = nt, mean = 0, sd = sd_noise) + sin(time / 10)
signal2[500:1000] = signal2[500:1000] + sin((500:1000) / 2)
\dontrun{
wsd <- wsd(signal1 = signal1, signal2 = signal2)
}

}
