% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/windowed_scalogram.R
\name{windowed_scalogram}
\alias{windowed_scalogram}
\title{Windowed scalograms of a signal}
\usage{
windowed_scalogram(signal,
                          dt = 1,
                          scales = NULL,
                          powerscales = TRUE,
                          windowrad = NULL,
                          delta_t = NULL,
                          wname = c("MORLET", "DOG", "PAUL", "HAAR", "HAAR2"),
                          wparam = NULL,
                          waverad = NULL,
                          border_effects = c("BE", "INNER", "PER", "SYM"),
                          energy_density = TRUE,
                          makefigure = TRUE,
                          time_values = NULL,
                          figureperiod = TRUE,
                          xlab = "Time",
                          ylab = NULL,
                          main = "Windowed Scalogram")
}
\arguments{
\item{signal}{A vector containing the signal whose windowed scalogram is wanted.}

\item{dt}{Numeric. The time step of the signal.}

\item{scales}{A vector containing the wavelet scales at wich
the windowed scalogram is computed. This can be either a vector with all the scales, or
(if \code{powerscales} is TRUE) following Torrence and Compo 1998, a vector of three
elements with the minimum scale, the maximum scale and the number of suboctaves per
octave. If NULL, they are automatically computed.}

\item{powerscales}{Logical. If TRUE (default), construct power 2 scales from
\code{scales}. If \code{scales} is NULL, they are automatically computed.}

\item{windowrad}{Integer. Time radius for the windows, measured in \code{dt}. By
default, it is set to \eqn{ceiling(length(signal) / 20)}.}

\item{delta_t}{Integer. Increment of time for the construction of windows central
times, measured in \code{dt}. By default, it is set to
\eqn{ceiling(length(signal) / 256)}.}

\item{wname}{A string, equal to "MORLET", "DOG", "PAUL", "HAAR" or "HAAR2". The
difference between "HAAR" and "HAAR2" is that "HAAR2" is more accurate but slower.}

\item{wparam}{The corresponding nondimensional parameter for the wavelet function
(Morlet, DoG or Paul).}

\item{waverad}{Numeric. The radius of the wavelet used in the computations for the cone
of influence. If it is not specified, it is asumed to be \eqn{\sqrt{2}} for Morlet and DoG,
\eqn{1/\sqrt{2}} for Paul and 0.5 for Haar.}

\item{border_effects}{String, equal to "BE", "INNER", "PER" or "SYM", which indicates
how to manage the border effects which arise usually when a convolution is performed on
finite-lenght signals.
\itemize{
\item "BE": With border effects, padding time series with zeroes.
\item "INNER": Normalized inner scalogram with security margin adapted for each
    different scale. Although there are no border effects, it is shown as a regular COI
    the zone in which the length of \eqn{J(s)} (see Benítez et al. 2010) is smaller and
    it has to be normalized.
\item "PER": With border effects, using boundary wavelets (periodization of the
original time series).
\item "SYM": With border effects, using a symmetric catenation of the original time
series.
}}

\item{energy_density}{Logical. If TRUE (default), divide the scalograms by the square
root of the scales for convert them into energy density.}

\item{makefigure}{Logical. If TRUE (default), a figure with the scalograms is plotted.}

\item{time_values}{A numerical vector of length \code{length(signal)} containing custom
time values for the figure. If NULL (default), it will be computed starting at 0.}

\item{figureperiod}{Logical. If TRUE (default), periods are used in the figure instead
of scales.}

\item{xlab}{A string giving a custom X axis label.}

\item{ylab}{A string giving a custom Y axis label. If NULL (default) the Y label is
either "Scale" or "Period" depending on the value of \code{figureperiod} if
\code{length(scales) > 1}, or "Windowed Scalogram" if \code{length(scales) == 1}.}

\item{main}{A string giving a custom main title for the figure.}
}
\value{
A list with the following fields:
\itemize{
\item \code{wsc}: A matrix of size \code{length(tcentral)} x \code{length(scales)}
containing the values of the windowed scalograms at each scale and at each time window.
\item \code{tcentral}: The vector of central times at which the windows are centered.
\item \code{scales}: The vector of the scales.
\item \code{windowrad}: Radius for the time windows, measured in \code{dt}.
\item \code{fourierfactor}: A factor for converting scales into periods.
\item \code{coi_maxscale}: A vector of length \code{length(tcentral)} containing the
values of the maximum scale from which there are border effects for the respective
central time.
}
}
\description{
This function computes the normalized windowed scalograms of a signal for
the scales given. It is computed using time windows with radius \code{windowrad}
centered at a vector of central times with increment of time \code{delta_t}. It is
important to note that the notion of scalogram here is analogous to the spectrum of the
Fourier transform. It gives the contribution of each scale to the total energy of the
signal. For each scale \eqn{s} and central time \eqn{tc}, it is defined as the square
root of the integral of the squared modulus of the wavelet transform w.r.t the time
variable \eqn{t}, i.e.

  \deqn{WS_{windowrad}(tc,s):=
  (\int_{tc-windowrad}^{tc+windowrad}|Wf(t,s)|^2 dt)^{1/2}.}{WS_{windowrad}(tc,s):=
  (integral_{tc-windowrad}^{tc+windowrad}|Wf(t,s)|^2 dt)^{1/2}.}

"Normalized" means that the windowed scalograms are divided by the square root of the
length of the respective time windows in order to be comparable between them.
}
\section{References}{


C. Torrence, G. P. Compo. A practical guide to wavelet analysis. B. Am. Meteorol. Soc.
79 (1998), 61–78.

V. J. Bolós, R. Benítez, R. Ferrer, R. Jammazi. The windowed scalogram difference: a
novel wavelet tool for comparing time series. Appl. Math. Comput., 312 (2017), 49-65.

R. Benítez, V. J. Bolós, M. E. Ramírez. A wavelet-based tool for studying
non-periodicity. Comput. Math. Appl. 60 (2010), no. 3, 634-641.
}

\examples{
dt <- 0.1
time <- seq(0, 50, dt)
signal <- c(sin(pi * time), sin(pi * time / 2))
wscalog <- windowed_scalogram(signal = signal, dt = dt)


}
