% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module2_estimation.R
\name{estimate_items}
\alias{estimate_items}
\alias{estimate_items_3pl_bme}
\alias{estimate_items_3pl_jmle}
\alias{estimate_items_3pl_mmle}
\title{Estimate Item Parameters}
\usage{
estimate_items(responses, model = c("3pl"), method = c("jmle", "mmle",
  "bme"), ...)

estimate_items_3pl_jmle(responses, people, fix = list(), iter = 30,
  conv = 0.01, bound = list(a = 2, b = 3.5, c = 0.25), init = list(a = 1,
  b = 0, c = 0), debug = FALSE)

estimate_items_3pl_mmle(responses, fix = list(), iter = 30, conv = 0.01,
  bound = list(a = 2, b = 3.5, c = 0.25), init = list(a = 1, b = 0, c = 0),
  debug = FALSE)

estimate_items_3pl_bme(responses, fix = list(), iter = 30, conv = 0.01,
  prior = list(a.mean = 0, a.sd = 0.2, b.mean = 0, b.sd = 1, c.alpha = 5,
  c.beta = 43), bound = list(a = 2, b = 3.5, c = 0.25), init = list(a = 1, b
  = 0, c = 0.1), debug = FALSE)
}
\arguments{
\item{responses}{a data frame or matrix of responses}

\item{model}{the IRT model}

\item{method}{the estimation method}

\item{...}{further arguments}

\item{people}{a data frame of people parameters}

\item{fix}{a list of fixed item parameters}

\item{iter}{the maximum iterations}

\item{conv}{the convergence criterion}

\item{bound}{the bounds of item parameters}

\item{init}{a list of initial item parameters}

\item{debug}{\code{TRUE} to turn on debugging mode}

\item{prior}{a list of prior distributions parameters}
}
\value{
a data frame of estimated item parameters

estimated item parameters and diagnosis data if debug mode is on
}
\description{
Estimate item parameters

\code{estimate_items_3pl_jmle} is the joint maximum likelihood estimator (JML) of item parameters

\code{estimate_items_3pl_mmle} is the marginal maximum likelihood estimator (MMLE) of item parameters

\code{estimate_items_3pl_bme} is the bayesian estimator of item parameters
}
\details{
For the joint maximum likelihood estimation (JMLE), refer to Baker and Kim (2004), pp. 46-54. \cr
To fixing item parameters, pass in a number or a vector (the same length with the items) to the \code{fix} argument.\cr
The debug mode prints convergence process and add \code{diagnosis} data to output

For the marginal maximum likelihood estimation (MMLE), refer to Baker and Kim (2004), pp.166-174.

For the Bayesian estimation, refer to Baker and Kim (2004), pp.183-191.
In \code{estimate_items_3pl_bme}, a parameters is assumed to have a lognormal prior distribution, 
b parameters a normal prior distribution, and c parameters a beta prior distribution.
}
\examples{
\dontrun{
# data generation
library(ggplot2)
library(reshape2)
library(dplyr)
data <- irt_model("3pl")$gendata(2000, 50)

# Estimate item parameters: JMLE
x.jmle <- estimate_items(data$responses, "3pl", "jmle", people=data$people)
# Estimate item parameters: MMLE
x.mmle <- estimate_items(data$responses, "3pl", "mmle")
# Estimate item parameters: BME
x.bme <- estimate_items(data$responses, "3pl", "bme")
# Comparison with true parameters
sapply(list(jmle=x.jmle, mmle=x.mmle, bme=x.bme), function(x) diag(cor(x$items, data$items)))
sapply(list(jmle=x.jmle, mmle=x.mmle, bme=x.bme), function(x) rmse(x$items, data$items))
x <- rbind(data.frame(method="jmle", melt(x.jmle$items), true=melt(data$items)$value),
           data.frame(method="mmle", melt(x.mmle$items), true=melt(data$items)$value),
           data.frame(method="bme", melt(x.bme$items), true=melt(data$items)$value))
ggplot(data=x, aes(x=true, y=value, color=method)) + geom_point(pch=1) +
  facet_grid(variable ~ method, scales="free") + xlab("True") + ylab("Est.") +
  theme_bw() + theme(legend.key=element_blank())
}
}

