% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/module3_ata.R
\name{ata}
\alias{ata}
\alias{ata_constraint}
\alias{ata_item_enemy}
\alias{ata_item_fixedvalue}
\alias{ata_item_maxselect}
\alias{ata_obj_absolute}
\alias{ata_obj_relative}
\alias{ata_solve}
\alias{ata_solve_lpsolve}
\alias{plot.ata}
\alias{print.ata}
\title{Automated Test Assembly (ATA)}
\usage{
ata(pool, nforms = 1, len = NULL, maxselect = NULL)

\method{print}{ata}(x, ...)

\method{plot}{ata}(x, ...)

ata_obj_relative(x, coef, mode = c("max", "min"), negative = FALSE,
  flatten = NULL, forms = NULL, collapse = FALSE)

ata_obj_absolute(x, coef, target, forms = NULL, collapse = FALSE)

ata_constraint(x, coef, min = NA, max = NA, level = NULL, forms = NULL,
  collapse = FALSE)

ata_item_maxselect(x, maxselect, items = NULL)

ata_item_enemy(x, items)

ata_item_fixedvalue(x, items, min = NA, max = NA, forms = NULL,
  collapse = FALSE)

ata_solve(x, as.list = TRUE, timeout = 10, mip_gap = 0.1,
  verbose = "neutral", warning = FALSE, ...)

ata_solve_lpsolve(x, ...)
}
\arguments{
\item{pool}{item pool}

\item{nforms}{the number of forms}

\item{len}{the test length}

\item{maxselect}{the maximum selection of each item}

\item{x}{the ata object}

\item{...}{further arguments}

\item{coef}{the coefficients of the objective function or the constraint}

\item{mode}{the optimization mode: 'max' for maximization and 'min' for minimization}

\item{negative}{\code{TRUE} when the expected value of the objective function is negative}

\item{flatten}{the flatten parameter to make the objective function flat}

\item{forms}{the indices of forms where objectives are added. \code{NULL} for all forms}

\item{collapse}{\code{TRUE} to collapse all forms into one objective function}

\item{target}{the target of the objective function}

\item{min}{the lower bound of the constraint}

\item{max}{the upper bound of the constraint}

\item{level}{the level value for categorical variable}

\item{items}{a vector of item indices}

\item{as.list}{\code{TRUE} to return results in a list; otherwise, data frame}

\item{timeout}{the time limit in seconds}

\item{mip_gap}{the mip gap parameter}

\item{verbose}{the message parameter}

\item{warning}{\code{TRUE} to output warning message when LP is not solved}
}
\description{
\code{ata} creates an \code{ata} object

\code{ata_obj_relative} adds relative (maximize/minimize) objectives to LP

\code{ata_obj_absolute} adds absolute objectives to LP

\code{ata_constraint} adds a constraint to LP

\code{ata_item_maxselect} sets the maximum selection for items

\code{ata_item_enemy} adds enemy item relationship to LP

\code{ata_item_fixedvalue} sets a fixed value range for items

\code{ata_solve} solves the LP

\code{ata_solve_lpsolve} solves the LP using LpSolve
}
\details{
The \code{ata} object stores LP definitions in \code{obj}, \code{mat}, 
\code{dir}, \code{rhs}, \code{types}, \code{bounds}, \code{max}. 
When calling \code{ata_solve}, it converts these definitions to the 
real LP object. If solved successfully, two results are added to the object:
\code{result} (a matrix of binary selection results) and \code{items} (a list 
or data frame of selected items).\cr

To maximize the LP, it is to maximize y while subject to sum(x) - y >= 0 and <= F (flatten parameter).
To minimize the LP, it is to minimize y while subject to sum(x) - y <= 0 and >= F.
By default, y is non-negative. When \code{negative=TRUE}, y is set to be negative. \cr 
When \code{coef} is a pool-size or form-size numeric vector, coefficients are used directly.
When \code{coef} is a variable name, variable values are used as coefficients.
When \code{coef} is a numeric vector unequal to pool size, information at those points are used as coefficients.\cr

\code{ata_obj_absolute} is to minimize y while subject to sum(x) + y >= T and sum(x) - y <= T. \cr

For \code{ata_constraint}, set \code{coef} to a variable name and \code{level} a level 
of that variable to add a categorical constraint. Set \code{coef} to a variable name and
leave \code{level} to default value (\code{NULL} or \code{NA}) to add a quantitative constraint. 
Set \code{coef} to a constant or a vector to add a constraint directly.\cr

In \code{ata_solve}, additional control parameters will be passed into solvers.
When passing control parameters to the GLPK solver, use the correct parameter name
(see \code{?glpkAPI::glpkConstants}).
}
\examples{
\dontrun{
library(dplyr)
## generate a 100-item pool
pool <- model_3pl()$gendata(1, 100)$items
pool$content <- sample(1:3, 100, replace=TRUE)
pool$time <- round(rlnorm(100, log(60), .2))

## ex. 1: 6 forms, 10 items, maximize b parameter
x <- ata(pool, 6, len=10, maxselect=1)
x <- ata_obj_relative(x, "b", "max")
x <- ata_solve(x)
sapply(x$items, function(x) {
  c(mean=mean(x$b), sd=sd(x$b), min=min(x$b), max=max(x$b))
}) \%>\% t() \%>\% round(., 2)

## ex. 2: 4 forms, 10 items, minimize b parameter
x <- ata(pool, 3, len=10, maxselect=1)
x <- ata_obj_relative(x, "b", "min", negative=TRUE)
x <- ata_solve(x, as.list=FALSE, timeout=5)
group_by(x$items, form) \%>\% 
  summarise(mean=mean(b), sd=sd(b), min=min(b), max=max(b)) \%>\%
  round(., 2)
  
## ex. 3: 2 forms, 10 items, mean(b) = 0, sd(b) = 1.0, content = (3, 3, 4)
x <- ata(pool, 2, len=10, maxselect=1) \%>\%
     ata_obj_absolute(pool$b, 0 * 10) \%>\%
     ata_obj_absolute((pool$b - 0)^2, 1 * 10) \%>\%
     ata_constraint("content", min=3, max=3, level=1) \%>\%
     ata_constraint("content", min=3, max=3, level=2) \%>\%
     ata_constraint("content", min=4, max=4, level=3)
x <- ata_solve(x, verbose="normal", timeout=5)
sapply(x$items, function(x) {
  c(mean=mean(x$b), sd=sd(x$b))
}) \%>\% t() \%>\% round(., 2)

# ex. 4: 2 forms, 10 items, flat TIF over [-1, 1]
x <- ata(pool, 2, len=10, maxselect=1) \%>\%
     ata_obj_relative(seq(-1, 1, .5), "max", flatten=0.05)
x <- ata_solve(x)
plot(x)
}
}

