#' Optimize the optimal value of GRNN smoothing parameter based on AUC 
#'
#' The function \code{grnn.optmiz_auc} optimize the optimal value of GRNN smoothing parameter by cross-validation.
#' It is applicable to the classification.
#'
#' @param net    A GRNN object generated by grnn.fit() 
#' @param lower  A scalar for the lower bound of the smoothing parameter
#' @param upper  A scalar for the upper bound of the smoothing parameter
#' @param nfolds A scalar for the number of n-fold, 4 by default
#' @param seed   The seed value for the n-fold cross-validation, 1 by default
#' @param method A scalar referring to the optimization method, 1 for Golden section searc and 2 for Brent’s method
#'
#' @return The best outcome
#'
#' @seealso \code{\link{grnn.search_auc}}
#'
#' @examples
#' data(iris, package = "datasets")
#' Y <- ifelse(iris[, 5] == "setosa", 1, 0)
#' X <- scale(iris[, 1:4])
#' gnet <- grnn.fit(x = X, y = Y)
#' \dontrun{
#' grnn.optmiz_auc(net = gnet, lower = 3, upper = 7, nfolds = 2)
#' }

grnn.optmiz_auc <- function(net, lower = 0, upper, nfolds = 4, seed = 1, method = 1) {
  if (class(net) != "General Regression Neural Net") stop("net needs to be a GRNN.", call. = F)
  if (!(method %in% c(1, 2))) stop("the method is not supported.", call. = F)

  fd <- folds(seq(nrow(net$x)), n = nfolds, seed = seed)
  cv <- function(s) {
    cls <- parallel::makeCluster(min(nfolds, parallel::detectCores() - 1), type = "PSOCK")
    obj <- c("fd", "net", "grnn.fit", "grnn.predone", "grnn.predict")
    parallel::clusterExport(cls, obj,  envir = environment())
    rs <- Reduce(rbind,
            parallel::parLapply(cls, fd,
              function(f) data.frame(ya = net$y[f],
                                     yp = grnn.predict(grnn.fit(net$x[-f, ], net$y[-f],  sigma = s),
                                                       net$x[f, ]))))
    parallel::stopCluster(cls)
    return(MLmetrics::AUC(y_pred = rs$yp, y_true = rs$ya))
  }

  if (method == 1) {
    rst <- optimize(f = cv, interval = c(lower, upper), maximum = T)
  } else if (method == 2) {
    rst <- optim(par = mean(lower, upper), fn = cv, lower = lower, upper = upper,
                 method = "Brent", control = list(fnscale = -1))
  }
  return(data.frame(sigma = rst[[1]], auc = rst[[2]]))
}
