






#![crate_name = "ndarray"]
#![doc(html_root_url = "https://docs.rs/ndarray/0.15/")]
#![doc(html_logo_url = "https://rust-ndarray.github.io/images/rust-ndarray_logo.svg")]
#![allow(
    clippy::many_single_char_names,
    clippy::deref_addrof,
    clippy::unreadable_literal,
    clippy::manual_map, // is not an error
    clippy::while_let_on_iterator, // is not an error
    clippy::from_iter_instead_of_collect, // using from_iter is good style
    clippy::redundant_closure, // false positives clippy #7812
)]
#![doc(test(attr(deny(warnings))))]
#![doc(test(attr(allow(unused_variables))))]
#![doc(test(attr(allow(deprecated))))]
#![cfg_attr(not(feature = "std"), no_std)]



























































































extern crate alloc;

#[cfg(feature = "std")]
extern crate std;
#[cfg(not(feature = "std"))]
extern crate core as std;

#[cfg(feature = "blas")]
extern crate cblas_sys;

#[cfg(feature = "docs")]
pub mod doc;

use std::marker::PhantomData;
use alloc::sync::Arc;

pub use crate::dimension::dim::*;
pub use crate::dimension::{Axis, AxisDescription, Dimension, IntoDimension, RemoveAxis};
pub use crate::dimension::{DimAdd, DimMax};

pub use crate::dimension::IxDynImpl;
pub use crate::dimension::NdIndex;
pub use crate::error::{ErrorKind, ShapeError};
pub use crate::indexes::{indices, indices_of};
pub use crate::order::Order;
pub use crate::slice::{
    MultiSliceArg, NewAxis, Slice, SliceArg, SliceInfo, SliceInfoElem, SliceNextDim,
};

use crate::iterators::Baseiter;
use crate::iterators::{ElementsBase, ElementsBaseMut, Iter, IterMut};

pub use crate::arraytraits::AsArray;
#[cfg(feature = "std")]
pub use crate::linalg_traits::NdFloat;
pub use crate::linalg_traits::LinalgScalar;

#[allow(deprecated)] // stack_new_axis
pub use crate::stacking::{concatenate, stack, stack_new_axis};

pub use crate::math_cell::MathCell;
pub use crate::impl_views::IndexLonger;
pub use crate::shape_builder::{Shape, ShapeBuilder, ShapeArg, StrideShape};

#[macro_use]
mod macro_utils;
#[macro_use]
mod private;
mod aliases;
#[macro_use]
mod itertools;
mod argument_traits;
#[cfg(feature = "serde")]
mod array_serde;
mod arrayformat;
mod arraytraits;
pub use crate::argument_traits::AssignElem;
mod data_repr;
mod data_traits;

pub use crate::aliases::*;

pub use crate::data_traits::{
    Data, DataMut, DataOwned, DataShared, RawData, RawDataClone, RawDataMut,
    RawDataSubst,
};

mod free_functions;
pub use crate::free_functions::*;
pub use crate::iterators::iter;

mod error;
mod extension;
mod geomspace;
mod indexes;
mod iterators;
mod layout;
mod linalg_traits;
mod linspace;
mod logspace;
mod math_cell;
mod numeric_util;
mod order;
mod partial;
mod shape_builder;
#[macro_use]
mod slice;
mod split_at;
mod stacking;
mod low_level_util;
#[macro_use]
mod zip;

mod dimension;

pub use crate::zip::{FoldWhile, IntoNdProducer, NdProducer, Zip};

pub use crate::layout::Layout;


mod imp_prelude {
    pub use crate::dimension::DimensionExt;
    pub use crate::prelude::*;
    pub use crate::ArcArray;
    pub use crate::{
        CowRepr, Data, DataMut, DataOwned, DataShared, Ix, Ixs, RawData, RawDataMut, RawViewRepr,
        RemoveAxis, ViewRepr,
    };
}

pub mod prelude;


pub type Ix = usize;

pub type Ixs = isize;













































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































































pub struct ArrayBase<S, D>
where
    S: RawData,
{


    data: S,


    ptr: std::ptr::NonNull<S::Elem>,

    dim: D,

    strides: D,
}























pub type ArcArray<A, D> = ArrayBase<OwnedArcRepr<A>, D>;






















pub type Array<A, D> = ArrayBase<OwnedRepr<A>, D>;

















pub type CowArray<'a, A, D> = ArrayBase<CowRepr<'a, A>, D>;












pub type ArrayView<'a, A, D> = ArrayBase<ViewRepr<&'a A>, D>;












pub type ArrayViewMut<'a, A, D> = ArrayBase<ViewRepr<&'a mut A>, D>;






















pub type RawArrayView<A, D> = ArrayBase<RawViewRepr<*const A>, D>;






















pub type RawArrayViewMut<A, D> = ArrayBase<RawViewRepr<*mut A>, D>;

pub use data_repr::OwnedRepr;





#[derive(Debug)]
pub struct OwnedArcRepr<A>(Arc<OwnedRepr<A>>);

impl<A> Clone for OwnedArcRepr<A> {
    fn clone(&self) -> Self {
        OwnedArcRepr(self.0.clone())
    }
}





#[derive(Copy, Clone)]

pub struct RawViewRepr<A> {
    ptr: PhantomData<A>,
}

impl<A> RawViewRepr<A> {
    #[inline(always)]
    fn new() -> Self {
        RawViewRepr { ptr: PhantomData }
    }
}





#[derive(Copy, Clone)]

pub struct ViewRepr<A> {
    life: PhantomData<A>,
}

impl<A> ViewRepr<A> {
    #[inline(always)]
    fn new() -> Self {
        ViewRepr { life: PhantomData }
    }
}





pub enum CowRepr<'a, A> {

    View(ViewRepr<&'a A>),

    Owned(OwnedRepr<A>),
}

impl<'a, A> CowRepr<'a, A> {

    pub fn is_view(&self) -> bool {
        match self {
            CowRepr::View(_) => true,
            CowRepr::Owned(_) => false,
        }
    }


    pub fn is_owned(&self) -> bool {
        match self {
            CowRepr::View(_) => false,
            CowRepr::Owned(_) => true,
        }
    }
}




mod impl_clone;

mod impl_internal_constructors;
mod impl_constructors;

mod impl_methods;
mod impl_owned_array;
mod impl_special_element_types;


impl<A, S, D> ArrayBase<S, D>
where
    S: Data<Elem = A>,
    D: Dimension,
{
    #[inline]
    fn broadcast_unwrap<E>(&self, dim: E) -> ArrayView<'_, A, E>
    where
        E: Dimension,
    {
        #[cold]
        #[inline(never)]
        fn broadcast_panic<D, E>(from: &D, to: &E) -> !
        where
            D: Dimension,
            E: Dimension,
        {
            panic!(
                "ndarray: could not broadcast array from shape: {:?} to: {:?}",
                from.slice(),
                to.slice()
            )
        }

        match self.broadcast(dim.clone()) {
            Some(it) => it,
            None => broadcast_panic(&self.dim, &dim),
        }
    }



    #[inline]
    fn broadcast_assume<E>(&self, dim: E) -> ArrayView<'_, A, E>
    where
        E: Dimension,
    {
        let dim = dim.into_dimension();
        debug_assert_eq!(self.shape(), dim.slice());
        let ptr = self.ptr;
        let mut strides = dim.clone();
        strides.slice_mut().copy_from_slice(self.strides.slice());
        unsafe { ArrayView::new(ptr, dim, strides) }
    }

    fn raw_strides(&self) -> D {
        self.strides.clone()
    }


    fn try_remove_axis(self, axis: Axis) -> ArrayBase<S, D::Smaller> {
        let d = self.dim.try_remove_axis(axis);
        let s = self.strides.try_remove_axis(axis);

        unsafe {
            self.with_strides_dim(s, d)
        }
    }
}


#[cfg(feature = "rayon")]
extern crate rayon_ as rayon;
#[cfg(feature = "rayon")]
pub mod parallel;

mod impl_1d;
mod impl_2d;
mod impl_dyn;

mod numeric;

pub mod linalg;

mod impl_ops;
pub use crate::impl_ops::ScalarOperand;

#[cfg(any(feature = "approx", feature = "approx-0_5"))]
mod array_approx;


mod impl_views;


mod impl_raw_views;


mod impl_cow;


pub(crate) fn is_aligned<T>(ptr: *const T) -> bool {
    (ptr as usize) % ::std::mem::align_of::<T>() == 0
}
