use core::ops::{Div, Rem};

pub trait Euclid: Sized + Div<Self, Output = Self> + Rem<Self, Output = Self> {



















    fn div_euclid(&self, v: &Self) -> Self;
























    fn rem_euclid(&self, v: &Self) -> Self;
}

macro_rules! euclid_forward_impl {
    ($($t:ty)*) => {$(
        #[cfg(has_div_euclid)]
        impl Euclid for $t {
            #[inline]
            fn div_euclid(&self, v: &$t) -> Self {
                <$t>::div_euclid(*self, *v)
            }

            #[inline]
            fn rem_euclid(&self, v: &$t) -> Self {
                <$t>::rem_euclid(*self, *v)
            }
        }
    )*}
}

macro_rules! euclid_int_impl {
    ($($t:ty)*) => {$(
        euclid_forward_impl!($t);

        #[cfg(not(has_div_euclid))]
        impl Euclid for $t {
            #[inline]
            fn div_euclid(&self, v: &$t) -> Self {
                let q = self / v;
                if self % v < 0 {
                    return if *v > 0 { q - 1 } else { q + 1 }
                }
                q
            }

            #[inline]
            fn rem_euclid(&self, v: &$t) -> Self {
                let r = self % v;
                if r < 0 {
                    if *v < 0 {
                        r - v
                    } else {
                        r + v
                    }
                } else {
                    r
                }
            }
        }
    )*}
}

macro_rules! euclid_uint_impl {
    ($($t:ty)*) => {$(
        euclid_forward_impl!($t);

        #[cfg(not(has_div_euclid))]
        impl Euclid for $t {
            #[inline]
            fn div_euclid(&self, v: &$t) -> Self {
                self / v
            }

            #[inline]
            fn rem_euclid(&self, v: &$t) -> Self {
                self % v
            }
        }
    )*}
}

euclid_int_impl!(isize i8 i16 i32 i64 i128);
euclid_uint_impl!(usize u8 u16 u32 u64 u128);

#[cfg(all(has_div_euclid, feature = "std"))]
euclid_forward_impl!(f32 f64);

#[cfg(not(all(has_div_euclid, feature = "std")))]
impl Euclid for f32 {
    #[inline]
    fn div_euclid(&self, v: &f32) -> f32 {
        let q = <f32 as crate::float::FloatCore>::trunc(self / v);
        if self % v < 0.0 {
            return if *v > 0.0 { q - 1.0 } else { q + 1.0 };
        }
        q
    }

    #[inline]
    fn rem_euclid(&self, v: &f32) -> f32 {
        let r = self % v;
        if r < 0.0 {
            r + <f32 as crate::float::FloatCore>::abs(*v)
        } else {
            r
        }
    }
}

#[cfg(not(all(has_div_euclid, feature = "std")))]
impl Euclid for f64 {
    #[inline]
    fn div_euclid(&self, v: &f64) -> f64 {
        let q = <f64 as crate::float::FloatCore>::trunc(self / v);
        if self % v < 0.0 {
            return if *v > 0.0 { q - 1.0 } else { q + 1.0 };
        }
        q
    }

    #[inline]
    fn rem_euclid(&self, v: &f64) -> f64 {
        let r = self % v;
        if r < 0.0 {
            r + <f64 as crate::float::FloatCore>::abs(*v)
        } else {
            r
        }
    }
}

pub trait CheckedEuclid: Euclid {


    fn checked_div_euclid(&self, v: &Self) -> Option<Self>;



    fn checked_rem_euclid(&self, v: &Self) -> Option<Self>;
}

macro_rules! checked_euclid_forward_impl {
    ($($t:ty)*) => {$(
        #[cfg(has_div_euclid)]
        impl CheckedEuclid for $t {
            #[inline]
            fn checked_div_euclid(&self, v: &$t) -> Option<Self> {
                <$t>::checked_div_euclid(*self, *v)
            }

            #[inline]
            fn checked_rem_euclid(&self, v: &$t) -> Option<Self> {
                <$t>::checked_rem_euclid(*self, *v)
            }
        }
    )*}
}

macro_rules! checked_euclid_int_impl {
    ($($t:ty)*) => {$(
        checked_euclid_forward_impl!($t);

        #[cfg(not(has_div_euclid))]
        impl CheckedEuclid for $t {
            #[inline]
            fn checked_div_euclid(&self, v: &$t) -> Option<$t> {
                if *v == 0 || (*self == Self::min_value() && *v == -1) {
                    None
                } else {
                    Some(Euclid::div_euclid(self, v))
                }
            }

            #[inline]
            fn checked_rem_euclid(&self, v: &$t) -> Option<$t> {
                if *v == 0 || (*self == Self::min_value() && *v == -1) {
                    None
                } else {
                    Some(Euclid::rem_euclid(self, v))
                }
            }
        }
    )*}
}

macro_rules! checked_euclid_uint_impl {
    ($($t:ty)*) => {$(
        checked_euclid_forward_impl!($t);

        #[cfg(not(has_div_euclid))]
        impl CheckedEuclid for $t {
            #[inline]
            fn checked_div_euclid(&self, v: &$t) -> Option<$t> {
                if *v == 0 {
                    None
                } else {
                    Some(Euclid::div_euclid(self, v))
                }
            }

            #[inline]
            fn checked_rem_euclid(&self, v: &$t) -> Option<$t> {
                if *v == 0 {
                    None
                } else {
                    Some(Euclid::rem_euclid(self, v))
                }
            }
        }
    )*}
}

checked_euclid_int_impl!(isize i8 i16 i32 i64 i128);
checked_euclid_uint_impl!(usize u8 u16 u32 u64 u128);

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn euclid_unsigned() {
        macro_rules! test_euclid {
            ($($t:ident)+) => {
                $(
                    {
                        let x: $t = 10;
                        let y: $t = 3;
                        assert_eq!(Euclid::div_euclid(&x, &y), 3);
                        assert_eq!(Euclid::rem_euclid(&x, &y), 1);
                    }
                )+
            };
        }

        test_euclid!(usize u8 u16 u32 u64);
    }

    #[test]
    fn euclid_signed() {
        macro_rules! test_euclid {
            ($($t:ident)+) => {
                $(
                    {
                        let x: $t = 10;
                        let y: $t = -3;
                        assert_eq!(Euclid::div_euclid(&x, &y), -3);
                        assert_eq!(Euclid::div_euclid(&-x, &y), 4);
                        assert_eq!(Euclid::rem_euclid(&x, &y), 1);
                        assert_eq!(Euclid::rem_euclid(&-x, &y), 2);
                        let x: $t = $t::min_value() + 1;
                        let y: $t = -1;
                        assert_eq!(Euclid::div_euclid(&x, &y), $t::max_value());
                    }
                )+
            };
        }

        test_euclid!(isize i8 i16 i32 i64 i128);
    }

    #[test]
    fn euclid_float() {
        macro_rules! test_euclid {
            ($($t:ident)+) => {
                $(
                    {
                        let x: $t = 12.1;
                        let y: $t = 3.2;
                        assert!(Euclid::div_euclid(&x, &y) * y + Euclid::rem_euclid(&x, &y) - x
                        <= 46.4 * <$t as crate::float::FloatCore>::epsilon());
                        assert!(Euclid::div_euclid(&x, &-y) * -y + Euclid::rem_euclid(&x, &-y) - x
                        <= 46.4 * <$t as crate::float::FloatCore>::epsilon());
                        assert!(Euclid::div_euclid(&-x, &y) * y + Euclid::rem_euclid(&-x, &y) + x
                        <= 46.4 * <$t as crate::float::FloatCore>::epsilon());
                        assert!(Euclid::div_euclid(&-x, &-y) * -y + Euclid::rem_euclid(&-x, &-y) + x
                        <= 46.4 * <$t as crate::float::FloatCore>::epsilon());
                    }
                )+
            };
        }

        test_euclid!(f32 f64);
    }

    #[test]
    fn euclid_checked() {
        macro_rules! test_euclid_checked {
            ($($t:ident)+) => {
                $(
                    {
                        assert_eq!(CheckedEuclid::checked_div_euclid(&$t::min_value(), &-1), None);
                        assert_eq!(CheckedEuclid::checked_rem_euclid(&$t::min_value(), &-1), None);
                        assert_eq!(CheckedEuclid::checked_div_euclid(&1, &0), None);
                        assert_eq!(CheckedEuclid::checked_rem_euclid(&1, &0), None);
                    }
                )+
            };
        }

        test_euclid_checked!(isize i8 i16 i32 i64 i128);
    }
}
